"""Module for importing mesh files."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.adcirc.file_io.fort14_reader import Fort14Reader
from xms.cmsflow._cmsflow_cmcards_import import CmCardsImporter
from xms.components.dmi.xms_data import XmsData
from xms.guipy.dialogs.feedback_thread import ExitError, ExpectedError, FeedbackThread

# 4. Local modules
from xms.ptm.model.model import MeshDefinition, simulation_model


class ImportMeshThread(FeedbackThread):
    """Import mesh thread."""
    def __init__(self, data: XmsData):
        """
        Construct the worker.
        """
        super().__init__(xms_data=data)
        self.display_text |= {
            'title': 'Reading mesh file',
            'working_prompt': 'Reading mesh file. Please wait...',
        }
        self.data = data
        self.grid = None
        self.name = None
        self.projection = None

    def _read_adcirc_mesh(self, section):
        """Read a mesh using the ADCIRC reader."""
        path_string = section.group('mesh').parameter('mesh_file').value
        if not path_string or not path_string.strip():
            raise ExpectedError('Please select a valid mesh file in the Model Control.')

        reader = Fort14Reader(path_string)
        reader.constraint_file = ''

        try:
            reader.read_grid()
        except FileNotFoundError:
            raise ExpectedError(f'File "{path_string}" did not exist.')
        except Exception:
            raise ExpectedError(f'File "{path_string}" could not be read.')

        self.grid = reader.cogrid
        self.grid_name = reader.built_data['domain_mesh'].name
        self.projection = reader.built_data['domain_mesh'].projection

    def _read_cms_grid(self, section):
        """Read a grid using the CMS reader."""
        path_string = section.group('mesh').parameter('grid_file').value
        if not path_string or not path_string.strip():
            raise ExpectedError('Please select a valid cmcards file in the Model Control.')
        if not Path(path_string).exists():
            raise ExpectedError(f'File "{path_string}" does not exist.')
        importer = CmCardsImporter(path_string)
        importer.read()
        if importer.quad_ugrid is None:
            raise ExitError()
        self.grid = importer.quad_ugrid
        self.grid_name = Path(importer.telescoping_filename).name
        self.projection = importer.grid_projection

    def _run(self):
        """Run the thread."""
        section = simulation_model()
        sim_data = self.data.simulation_data
        section.restore_values(sim_data.global_values)
        grid_type = section.group('mesh').parameter('mesh_type').value

        if grid_type == MeshDefinition.cms:
            self._read_cms_grid(section)
        elif grid_type == MeshDefinition.adcirc:
            self._read_adcirc_mesh(section)
        elif grid_type == MeshDefinition.linked:
            raise ExpectedError("Cannot import mesh because it's defined as a linked mesh.")
        else:
            raise AssertionError('Unrecognized grid type.')
        self.data.add_grid(self.grid, self.grid_name, self.projection)
