"""Module for the .source file writer."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['write_sources']

# 1. Standard Python modules
from datetime import datetime
from pathlib import Path
from typing import TextIO

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.ptmio.source.sources import LineMassDatum, Pt3d, Sources


def write_sources(sources: Sources, where: str | Path | TextIO):
    """
    Write sources to a file.

    Args:
        sources: The sources to write.
        where: Where to write them.
    """
    if isinstance(where, (str, Path)):
        with open(where, 'w') as f:
            write_sources(sources, f)
            return

    _write_instant_sources(sources, where)
    _write_point_sources(sources, where)
    _write_line_sources(sources, where)
    _write_polygon_sources(sources, where)


def _write_instant_sources(sources: Sources, where: TextIO):
    """Write all the instant sources."""
    where.write(f'{len(sources.instant_sources)} ! Number of instant mass sources\n')
    for source in sources.instant_sources:
        where.write(f' {source.source_id}  {len(source.instructions)}  "{source.label}"\n')
        for instruction in source.instructions:
            _write_time(instruction.time, where)
            _write_location(instruction.location, where)
            where.write(
                f'{instruction.parcel_mass} {instruction.h_radius} {instruction.v_radius} '
                f'{instruction.mass_rate} {instruction.grain_size} {instruction.stdev} '
                f'{instruction.density} {instruction.velocity} '
                f'{instruction.initiation} {instruction.deposition}'
                '\n'
            )


def _write_point_sources(sources: Sources, where: TextIO):
    """Write all the point sources."""
    where.write(f'{len(sources.point_sources)} ! Number of point mass sources\n')
    for source in sources.point_sources:
        where.write(f' {source.source_id}  {len(source.instructions)}  "{source.label}"\n')
        for instruction in source.instructions:
            _write_time(instruction.time, where)
            _write_location(instruction.location, where)
            where.write(
                f'{instruction.parcel_mass} {instruction.h_radius} {instruction.v_radius} {instruction.mass_rate} '
                f'{instruction.grain_size} {instruction.stdev} {instruction.density} '
                f'{instruction.velocity} {instruction.initiation} {instruction.deposition}'
                '\n'
            )


def _write_line_sources(sources: Sources, where: TextIO):
    """Write all the line sources."""
    where.write(f'{len(sources.line_sources)} ! Number of line mass sources\n')
    for source in sources.line_sources:
        where.write(f' {source.source_id}  {len(source.instructions)}  "{source.label}"')
        _write_datum(source.datum, where)

        for instruction in source.instructions:
            _write_time(instruction.time, where)
            _write_location(instruction.start, where)
            _write_location(instruction.end, where)
            where.write(
                f'{instruction.parcel_mass} {instruction.h_radius} {instruction.v_radius} '
                f'{instruction.mass_rate} {instruction.grain_size} {instruction.stdev} '
                f'{instruction.density} {instruction.velocity} '
                f'{instruction.initiation} {instruction.deposition}'
                '\n'
            )


def _write_polygon_sources(sources: Sources, where: TextIO):
    """Write all the polygon sources."""
    where.write(f'{len(sources.polygon_sources)} ! Number of polygon mass sources\n')
    for source in sources.polygon_sources:
        where.write(
            f' {source.source_id}  {len(source.instructions[0].points)}  '
            f'{len(source.instructions)}  "{source.label}"'
            '\n'
        )
        for instruction in source.instructions:
            _write_time(instruction.time, where, end='\n')
            for point in instruction.points:
                where.write('   ')
                _write_location(point, where, end='\n')
            where.write(
                f'   {instruction.parcel_mass} {instruction.h_radius} {instruction.v_radius} '
                f'{instruction.mass_rate} {instruction.grain_size} {instruction.stdev} '
                f'{instruction.density} {instruction.velocity} '
                f'{instruction.initiation} {instruction.deposition}'
                '\n'
            )


def _write_time(time: datetime, where: TextIO, end: str = ' '):
    """Write a time to the file."""
    where.write(f'  {time.year:2} {time.month:2} {time.day:2} {time.hour:2} {time.minute:2} {time.second:2}{end}')


def _write_location(location: Pt3d, where: TextIO, end: str = ' '):
    """
    Write a location to the file.

    Args:
        location: The location to write.
        where: Where to write it.
        end: What to write after the last point. The default is a space so that other fields can follow it, but some
            places use a newline where there are no more fields.
    """
    x, y, z = location
    where.write(f'{x} {y} {z}{end}')


def _write_datum(datum: LineMassDatum, where: TextIO):
    """Write the line mass datum, or just a newline if not needed."""
    if datum == LineMassDatum.none:
        value = ''
    elif datum == LineMassDatum.bed_datum:
        value = 'beddatum'
    elif datum == LineMassDatum.surface_datum:
        value = 'surfacedatum'
    elif datum == LineMassDatum.depth_distributed:
        value = 'depthdistributed'
    else:
        raise ValueError('Unknown LineMassDatum.')

    if value:
        where.write(f' {value}\n')
    else:
        where.write('\n')
