"""Module for classes describing the content of a .sources file."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = [
    'Sources',
    'InstantMassInstruction',
    'InstantMassSource',
    'PointMassInstruction',
    'PointMassSource',
    'Line',
    'LineMassDatum',
    'LineMassInstruction',
    'LineMassSource',
    'Polygon',
    'PolygonMassInstruction',
    'PolygonMassSource',
    'Pt3d',
]

# 1. Standard Python modules
from dataclasses import dataclass, field
from datetime import datetime
from enum import auto, Enum
from typing import TypeAlias

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules

Pt3d: TypeAlias = tuple[float, float, float]
Line: TypeAlias = tuple[Pt3d, Pt3d]
Polygon: TypeAlias = list[Pt3d]
default_date_time = datetime(year=1950, month=1, day=1)


class LineMassDatum(Enum):
    """The datum that appears next to a line mass source's header."""
    none = auto()
    bed_datum = auto()
    surface_datum = auto()
    depth_distributed = auto()


@dataclass
class InstantMassInstruction:
    """
    An instant mass instruction.

    Args:
        time: The time the instruction is issued at.
        location: Where the instruction is issued at. Units are meters.
        parcel_mass: The mass of each parcel the source generates, in kilograms.
        h_radius: Horizontal radius of the source, in meters.
        v_radius: Vertical radius of the source, in meters.
        mass_rate: The rate at which the source generates mass, in kilograms.
        grain_size: Size of a grain, in meters.
        stdev: Standard deviation of the sediment distribution, in Phi-units.
        density: The density, in kilograms/meter^3.
        velocity: The fall velocity, in meters/second. -1.0 to have PTM compute it.
        initiation: Tau critical initiation, in Newtons/meter^2. -1.0 to have PTM compute it.
        deposition: Tau critical deposition, in Newtons/meter^2. -1.0 to have PTM compute it.
    """
    time: datetime = default_date_time
    location: Pt3d = (0.0, 0.0, 0.0)
    parcel_mass: float = 0.0
    h_radius: float = 0.0
    v_radius: float = 0.0
    mass_rate: float = 0.0
    grain_size: float = 0.0
    stdev: float = 0.0
    density: float = 0.0
    velocity: float = 0.0
    initiation: float = 0.0
    deposition: float = 0.0


@dataclass
class InstantMassSource:
    """
    Class representing a single instant source.

    Args:
        source_id: The source's ID.
        label: The source's label.
        instructions: List of instructions to apply to the source.
    """
    source_id: int = -1
    label: str = ''
    instructions: list[InstantMassInstruction] = field(default_factory=list)


@dataclass
class PointMassInstruction:
    """
    An instant mass instruction.

    Args:
        time: The time the instruction is issued at.
        location: Where the instruction is issued at. Units are meters.
        parcel_mass: The mass of each parcel, in kilograms.
        h_radius: Horizontal radius of the source, in meters.
        v_radius: Vertical radius of the source, in meters.
        mass_rate: The rate at which mass is generated, in kilograms/second.
        grain_size: Size of a grain, in meters.
        stdev: Standard deviation of the sediment distribution, in Phi-units.
        density: The density, in kilograms/meter^3.
        velocity: The fall velocity, in meters/second. -1.0 to have PTM compute it.
        initiation: Tau critical initiation, in Newtons/meter^2. -1.0 to have PTM compute it.
        deposition: Tau critical deposition, in Newtons/meter^2. -1.0 to have PTM compute it.
    """
    time: datetime = default_date_time
    location: Pt3d = (0.0, 0.0, 0.0)
    parcel_mass: float = 0.0
    h_radius: float = 0.0
    v_radius: float = 0.0
    mass_rate: float = 0.0
    grain_size: float = 0.0
    stdev: float = 0.0
    density: float = 0.0
    velocity: float = 0.0
    initiation: float = 0.0
    deposition: float = 0.0


@dataclass
class PointMassSource:
    """
    Class representing a single point source.

    Args:
        source_id: The source's ID.
        label: The source's label.
        instructions: List of instructions to apply to the source.
    """
    source_id: int = -1
    label: str = ''
    instructions: list[PointMassInstruction] = field(default_factory=list)


@dataclass
class LineMassInstruction:
    """
    A line mass instruction.

    Args:
        time: The time the instruction is issued at.
        start: First point defining the line where the instruction is issued at. Units are meters.
        end: Second point defining the line where the instruction is issued at. Units are meters.
        parcel_mass: The mass of each parcel, in kilograms.
        h_radius: Horizontal radius of the source, in meters.
        v_radius: Vertical radius of the source, in meters.
        mass_rate: The rate at which mass is generated, in kilograms/second/meter.
        grain_size: Size of a grain, in meters.
        stdev: Standard deviation of the sediment distribution, in Phi-units.
        density: The density, in kilograms/meter^3.
        velocity: The fall velocity, in meters/second. -1.0 to have PTM compute it.
        initiation: Tau critical initiation, in Newtons/meter^2. -1.0 to have PTM compute it.
        deposition: Tau critical deposition, in Newtons/meter^2. -1.0 to have PTM compute it.
    """
    time: datetime = default_date_time
    start: Pt3d = (0.0, 0.0, 0.0)
    end: Pt3d = (0.0, 0.0, 0.0)
    parcel_mass: float = 0.0
    h_radius: float = 0.0
    v_radius: float = 0.0
    mass_rate: float = 0.0
    grain_size: float = 0.0
    stdev: float = 0.0
    density: float = 0.0
    velocity: float = 0.0
    initiation: float = 0.0
    deposition: float = 0.0


@dataclass
class LineMassSource:
    """
    Class representing a single line source.

    Args:
        source_id: The source's ID.
        label: The source's label.
        instructions: List of instructions to apply to the source.
    """
    source_id: int = -1
    label: str = ''
    datum: LineMassDatum = LineMassDatum.bed_datum
    instructions: list[LineMassInstruction] = field(default_factory=list)


@dataclass
class PolygonMassInstruction:
    """
    A polygon mass instruction.

    Args:
        time: The time the instruction is issued at.
        points: Locations defining the outer ring of the polygon. Holes are unsupported.
        parcel_mass: The mass of each parcel, in kilograms.
        h_radius: Horizontal radius of the source, in meters.
        v_radius: Vertical radius of the source, in meters.
        mass_rate: The rate at which mass is produced, in kilograms/second/meter^2.
        grain_size: Size of a grain, in meters.
        stdev: Standard deviation of the sediment distribution, in Phi-units.
        density: The density, in kilograms/meter^3.
        velocity: The fall velocity, in meters/second. -1.0 to have PTM compute it.
        initiation: Tau critical initiation, in Newtons/meter^2. -1.0 to have PTM compute it.
        deposition: Tau critical deposition, in Newtons/meter^2. -1.0 to have PTM compute it.
    """
    time: datetime = default_date_time
    points: list[Pt3d] = field(default_factory=list)
    parcel_mass: float = 0.0
    h_radius: float = 0.0
    v_radius: float = 0.0
    mass_rate: float = 0.0
    grain_size: float = 0.0
    stdev: float = 0.0
    density: float = 0.0
    velocity: float = 0.0
    initiation: float = 0.0
    deposition: float = 0.0


@dataclass
class PolygonMassSource:
    """
    Class representing a single polygon source.

    Args:
        source_id: The source's ID.
        label: The source's label.
        instructions: List of instructions to apply to the source.
    """
    source_id: int = -1
    label: str = ''
    instructions: list[PolygonMassInstruction] = field(default_factory=list)


@dataclass
class Sources:
    """
    Class representing a .sources file.

    Args:
        instant_sources: Instant mass sources in file.
        point_sources: Point mass sources in file.
        line_sources: Line mass sources in file.
        polygon_sources: Polygon mass sources in file.
    """
    instant_sources: list[InstantMassSource] = field(default_factory=list)
    point_sources: list[PointMassSource] = field(default_factory=list)
    line_sources: list[LineMassSource] = field(default_factory=list)
    polygon_sources: list[PolygonMassSource] = field(default_factory=list)
