"""BainComponent class."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from functools import cached_property
from pathlib import Path
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.component_bases.coverage_component_base import CoverageComponentBase
from xms.gmi.data.generic_model import Section
from xms.gmi.gui.material_section_dialog import MaterialSectionDialog
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.rsm.data.rule_curve_data import RuleCurveData


class RuleCurveComponent(CoverageComponentBase):
    """A Dynamic Model Interface (DMI) component base class."""
    def __init__(self, main_file: Optional[str | Path]):
        """
        Initialize the component.

        Args:
            main_file: The component's main-file.
        """
        super().__init__(main_file)
        self.tree_commands = [('Rule curves...', self._open_rule_curves)]

    def _get_section(self, target: TargetType) -> Section:
        return self.data.generic_model.section_from_target_type(target)

    @cached_property
    def data(self) -> RuleCurveData:
        """The component's data manager."""
        return RuleCurveData(self.main_file)

    def _open_rule_curves(self, query, params, parent):
        """Opens the Assign arc dialog and saves component data state on OK.

        Args:
            query (Query): Object for communicating with XMS
            params (list[dict]): Generic map of parameters. Contains selection map and component id files.
            parent (QWidget): The window container.

        Returns:
            Messages and requests.
        """
        section = self.data.generic_model.material_parameters
        section.restore_values(self.data.values)

        dlg = MaterialSectionDialog(
            parent=parent,
            section=section,
            is_interior=False,
            dlg_name=f'{self.module_name}.assign_material_dialog',
            window_title='Rule Curves',
            enable_unchecked_groups=True,
            get_curve=None,
            add_curve=None,
            hide_checkboxes=True,
        )
        if dlg.exec():
            self.data.values = dlg.section.extract_values()
            # If the user changed the list, we need to save it. But values don't matter here, so discard them.
            dlg.section.clear_values()
            model = self.data.generic_model
            model.material_parameters = dlg.section
            self.data.generic_model = model
            self.data.commit()

        return [], []
