"""ParameterRunsFilterProxyModel class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QSortFilterProxyModel, Qt

# 3. Aquaveo modules

# 4. Local modules


class ParameterRunsFilterProxyModel(QSortFilterProxyModel):
    """QSortFilterProxyModel derived class for enabling/disabling cells in the runs table."""
    def __init__(self, param_data, parent=None):
        """Initializes the filter model.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            param_data (ParameterData): Parameter data.
        """
        super().__init__(parent)
        self._param_data = param_data

    def flags(self, index):
        """Set flags for an item in the model.

        Args:
            index (QModelIndex): The item's model index

        Returns:
            (Qt.ItemFlags): Updated flags for the item
        """
        ret_flags = super().flags(index)
        col = index.column()
        if col == 0:  # Runs column
            return ret_flags

        model = index.model()
        column_name = model.headerData(col, Qt.Horizontal)
        parameter = self._param_data.parameter_from_column_name(column_name)
        if not parameter:
            raise RuntimeError('Error matching column to parameter.')  # pragma no cover - hard to test exceptions

        ret_flags = parameter.flags(index, column_name, ret_flags)
        return ret_flags
