"""ModelControlSediment class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import pandas as pd
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler

# 4. Local modules
from xms.srh.data.sediment_cohesive import SedimentCohesive
from xms.srh.data.sediment_equation_parameters import SedimentEquationParameters
from xms.srh.data.sediment_parameters import SedimentParameters


class ModelControlSediment(param.Parameterized):
    """Manages data file for the hidden simulation component."""
    # Sediment transport
    shear_partitioning_option = param.ObjectSelector(
        default='Percentage',
        objects=['Percentage', 'Scaled D90'],
        doc='Option for how to specify the shear partitioning for materials.',
        precedence=10,
    )
    shear_partitioning_scaled_d90_factor = param.Number(
        label='Shear partitioning scaled D90 factor',
        default=2.5,
        bounds=(0.0, None),
        doc='Global scaled D90 factor for shear partitioning.',
        precedence=10.2,
    )
    shear_partitioning_percentage = param.Number(
        label='Shear partitioning percentage',
        default=100.0,
        bounds=(0.0, 100.0),
        doc='Global percentage for shear partitioning.',
        precedence=10.3,
    )
    specify_sediment_simulation_start_time = param.Boolean(
        default=False,
        doc='Option to specify the sediment start time.',
        precedence=11,
    )
    sediment_simulation_start_time = param.Number(
        default=0.0,
        bounds=(0.0, None),
        doc='The sediment start time',
        precedence=11.1,
    )
    sediment_specific_gravity = param.Number(
        default=2.65,
        bounds=(0.0, None),
        doc='specific gravity',
        precedence=20,
    )
    particle_diameter_threshold = param.DataFrame(
        default=pd.DataFrame(data=[0.0], columns=['Particle Diameter Threshold (mm)']),
        doc='list of particle diameter thresholds',
        precedence=30,
    )
    transport_equation_parameters = param.ClassSelector(
        label='',
        class_=SedimentEquationParameters,
        doc='The sediment transport equations data',
        precedence=40,
    )
    transport_parameters = param.ClassSelector(
        label='',
        class_=SedimentParameters,
        doc='The sediment transport parameters that do not depend on the transport equation',
        precedence=50,
    )
    enable_cohesive_sediment_modeling = param.Boolean(
        default=False,
        doc='unsteady flow',
        precedence=60,
    )
    cohesive = param.ClassSelector(
        label='',
        class_=SedimentCohesive,
        doc='The sediment transport cohesion parameters',
        precedence=70,
    )

    def __init__(self):
        """Initializes the data class.

        Args:
            data_file (:obj:`str`): The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.

        """
        super().__init__()
        self.transport_equation_parameters = SedimentEquationParameters(include_mixed=True)
        self.transport_parameters = SedimentParameters()
        self.cohesive = SedimentCohesive()
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=[
                'shear_partitioning_option', 'specify_sediment_simulation_start_time',
                'enable_cohesive_sediment_modeling'
            ],
            depends={
                'Scaled D90': ['shear_partitioning_scaled_d90_factor'],
                'Percentage': ['shear_partitioning_percentage'],
                'specify_sediment_simulation_start_time': ['sediment_simulation_start_time'],
                'enable_cohesive_sediment_modeling': ['cohesive']
            }
        )

    @param.depends('shear_partitioning_option', watch=True)
    def _update_shear_partitioning_option(self):
        self.enabler.do_enabling(force_disable=False)

    @param.depends('specify_sediment_simulation_start_time', watch=True)
    def _update_specify_sediment_simulation_start_time(self):
        self.enabler.do_enabling(force_disable=False)

    @param.depends('enable_cohesive_sediment_modeling', watch=True)
    def _update_enable_cohesive_sediment_modeling(self):
        self.enabler.do_enabling(force_disable=False)
