"""BcDataCulvert class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler  # noqa: I202

# 4. Local modules
from xms.srh.data.par.bc_data_weir_type import BcDataWeirType


class BcDataCulvert(param.Parameterized):
    """Bc type culvert data."""
    invert_elevation = param.Number(
        label='Upstream invert elevation (ZI)',
        default=0.0,
        doc='',
        precedence=2,
    )
    barrel_height = param.Number(
        label='Interior height of barrel (Dc)',
        default=0.0,
        doc='',
        precedence=3,
    )
    barrel_length = param.Number(
        label='Length of barrel (Lc)',
        default=0.0,
        doc='',
        precedence=4,
    )
    barrel_area = param.Number(
        label='Area of barrel (Ac)',
        default=0.0,
        doc='',
        precedence=5,
    )
    barrel_hydraulic_radius = param.Number(
        label='Hydraulic radius of barrel (Rh)',
        default=0.0,
        doc='',
        precedence=6,
    )
    barrel_slope = param.Number(
        label='Slope of barrel (Slp)',
        default=0.0,
        doc='',
        precedence=7,
    )
    units = param.ObjectSelector(
        default='Feet',
        objects=['Feet', 'Meters'],
        doc='',
        precedence=8,
    )
    num_barrels = param.Integer(
        label='Number of identical barrels',
        default=0,
        doc='',
        precedence=9,
    )
    entrance_type = param.ObjectSelector(
        label='Entrance type (m_in)',
        default='mitered',
        objects=['mitered', 'non-mitered'],
        doc='',
        precedence=10,
    )
    inlet_coefficients = param.ObjectSelector(
        label='Culvert inlet coefficients (Kp, M, cp, Y)',
        default='Concrete - Circular - Headwall; square edge',
        objects=[
            'Concrete - Circular - Headwall; square edge',
            'Concrete - Circular - Headwall; grooved edge',
            'Concrete - Circular - Projecting; grooved edge',
            'Cor metal - Circular - Headwall',
            'Cor metal - Circular - Mitered to slope',
            'Cor metal - Circular - Projecting',
            'Concrete - Circular - Beveled ring; 45 deg bevels',
            'Concrete - Circular - Beveled ring; 33.7 deg bevels',
            'Concrete - Rectangular - Wingwalls; 30-75 deg flares; square edge',
            'Concrete - Rectangular - Wingwalls; 90 and 15 deg flares; square edge',
            'Concrete - Rectangular - Wingwalls; 0 deg flares; square edge',
            'Concrete - Rectangular - Wingwalls; 45 deg flares; beveled edge',
            'Concrete - Rectangular - Wingwalls; 18-33.7 deg flare; beveled edge',
            'Concrete - Rectangular - Headwall; 3/4 in chamfers',
            'Concrete - Rectangular - Headwall; 45 deg bevels',
            'Concrete - Rectangular - Headwall; 33.7 deg bevels',
            'Concrete - Rectangular - Headwall; 45 deg skew; 3/4 in chamfers',
            'Concrete - Rectangular - Headwall; 30 deg skew; 3/4 in chamfers',
            'Concrete - Rectangular - Headwall; 15 deg skew; 3/4 in chamfers',
            'Concrete - Rectangular - Headwall; 10-45 deg skew; 45 deg bevels',
            'Concrete - Rectangular - Wingwalls; non-offset 45 deg flares; 3/4 in chamfers',
            'Concrete - Rectangular - Wingwalls; non-offset 18.4 deg flares; 3/4 in chamfers',
            'Concrete - Rectangular - Wingwalls; non-offset 18.4 deg flares; 30 deg skewed barrel',
            'Concrete - Rectangular - Wingwalls; offset 45 deg flares; beveled top edge',
            'Concrete - Rectangular - Wingwalls; offset 33.7 deg flares; beveled top edge',
            'Concrete - Rectangular - Wingwalls; offset 18.4 deg flares; beveled top edge',
            'Cor metal - Rectangular - Headwall',
            'Cor metal - Rectangular - Projecting; thick wall',
            'Cor metal - Rectangular - Projecting; thin wall',
            'Concrete - Horizontal ellipse - Headwall; square edge',
            'Concrete - Horizontal ellipse - Headwall; grooved edge',
            'Concrete - Horizontal ellipse - Projecting; grooved edge',
            'Cor metal - Pipe arch (18in corner) - Headwall',
            'Cor metal - Pipe arch (18in corner) - Mitered to slope',
            'Cor metal - Pipe arch (18in corner) - Projecting',
            'Structural plate - Pipe arch (18in corner) - Projecting',
            'Structural plate - Pipe arch (18in corner) - Headwall; square edge',
            'Structural plate - Pipe arch (18in corner) - Headwall; beveled edge',
            'Structural plate - Pipe arch (31in corner) - Projecting',
            'Structural plate - Pipe arch (31in corner) - Headwall; square edge',
            'Structural plate - Pipe arch (31in corner) - Headwall; beveled edge',
            'Cor metal - Arch - Headwall',
            'Cor metal - Arch - Mitered to slope',
            'Cor metal - Arch - Projecting',
            'Concrete - Circular - Tapered throat',
            'Cor metal - Circular - Tapered throat',
            'Concrete - Rectangular - Tapered throat',
        ],
        doc='',
        precedence=11,
    )
    loss_coefficient = param.Number(
        label='Entrance loss coeff (Ke)',
        default=0.5,
        doc='',
        precedence=12,
    )
    mannings_n = param.Number(
        label='Manning roughness coefficient in barrel (Nc)',
        default=0.012,
        doc='',
        precedence=13,
    )
    total_head = param.Boolean(
        label='Use total head (velocity and water surface)',
        default=False,
        doc='',
        precedence=14,
    )
    crest_elevation = param.Number(
        default=0.0,
        doc='',
        precedence=15,
    )
    length_of_weir_over_culvert = param.Number(
        default=0.0,
        doc='',
        precedence=16,
    )
    weir = param.ClassSelector(
        class_=BcDataWeirType,
        doc='',
        precedence=17,
    )

    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self.weir = BcDataWeirType()
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=[],
            depends={},
        )
