"""BcDataLink class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import pandas as pd
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler  # noqa: I202

# 4. Local modules
from xms.srh.data.par.bc_data_weir import BcDataWeir


class BcDataLink(param.Parameterized):
    """Bc type link data."""
    inflow_type = param.ObjectSelector(
        label='Link inflow (Q) type',
        default='Rating curve',
        objects=['Constant', 'Time series', 'Weir', 'Rating curve'],
        doc='',
        precedence=1,
    )
    constant_q = param.Number(
        label='Constant Q',
        default=0.0,
        doc='',
        precedence=2,
    )
    constant_q_units = param.ObjectSelector(
        label='',
        default='cfs',
        objects=['cfs', 'cms'],
        doc='',
        precedence=3,
    )
    time_series_q = param.DataFrame(
        default=pd.DataFrame(columns=['hrs', 'vol_per_sec']),
        doc='time series for Q',
        precedence=4,
    )
    time_series_q_units = param.ObjectSelector(
        label='',
        default='hrs -vs- cfs',
        objects=['hrs -vs- cfs', 'hrs -vs- cms'],
        doc='units for time series Q',
        precedence=5,
    )
    weir = param.ClassSelector(
        label='',
        class_=BcDataWeir,
        doc='Link weir bc parameters',
        precedence=6,
    )
    rating_curve = param.DataFrame(
        default=pd.DataFrame(columns=['vol_per_sec', 'WSE']),
        doc='',
        precedence=7,
    )
    rating_curve_units = param.ObjectSelector(
        label='',
        default='cfs -vs- feet',
        objects=['cfs -vs- feet', 'cms -vs- meters'],
        doc='',
        precedence=8,
    )
    link_lag_method = param.ObjectSelector(
        default='Specified',
        objects=['Specified', 'Computed'],
        doc='',
        precedence=9,
    )
    specified_lag = param.Number(
        label='Specified lag (sec)',
        default=0.0,
        doc='',
        precedence=10,
    )
    conduit_length = param.Number(
        default=0.0,
        doc='',
        precedence=11,
    )
    conduit_diameter = param.Number(
        default=0.0,
        doc='',
        precedence=12,
    )
    conduit_slope = param.Number(
        default=0.0,
        doc='',
        precedence=13,
    )
    conduit_mannings = param.Number(
        label='Conduit Manning\'s N',
        default=0.0,
        doc='',
        precedence=14,
    )
    conduit_units = param.ObjectSelector(
        default='Feet',
        objects=['Feet', 'Meters'],
        doc='',
        precedence=15,
    )

    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self.weir = BcDataWeir()
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=['inflow_type', 'link_lag_method'],
            depends={
                'Constant': ['constant_q', 'constant_q_units'],
                'Time series': ['time_series_q', 'time_series_q_units'],
                'Weir': ['weir'],
                'Rating curve': ['rating_curve', 'rating_curve_units'],
                'Specified': ['specified_lag'],
                'Computed': [
                    'conduit_length', 'conduit_diameter', 'conduit_slope', 'conduit_mannings', 'conduit_units'
                ],
            },
        )

    @param.depends('inflow_type', watch=True)
    def _update_inflow_type(self):
        """Watches changes made to the inflow_type object."""
        self.enabler.do_enabling(force_disable=False)

    @param.depends('link_lag_method', watch=True)
    def _update_link_lag_method(self):
        """Watches changes made to the link_lag_method object."""
        self.enabler.do_enabling(force_disable=False)
