"""Creates an Monitor coverage hidden component."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.data_objects.parameters import Component

# 4. Local modules
from xms.srh.data.monitor_data import MonitorData


def build_monitor_component(cov_uuid, comp_dir, new_comp_uuid=None):
    """Create the obstructions coverage's hidden component and data.

    Args:
        cov_uuid (:obj:`str`): UUID of the obstructions coverage geometry
        comp_dir (:obj:`str`): Path to the XMS temp components folder (where the component will be created).
        new_comp_uuid (:obj:`str`): UUID of the new BC component. Useful for testing, usually just want
            randomly generated.

    Returns:
        (:obj:`xms.data_objects.parameters.Component`): data_object for the new Obstructions component
    """
    # Create a new UUID and folder for the component data
    if new_comp_uuid:  # Testing, use hard-coded UUID
        comp_uuid = new_comp_uuid
    else:  # pragma: no cover
        comp_uuid = str(uuid.uuid4())
    monitor_comp_dir = os.path.join(comp_dir, comp_uuid)
    os.makedirs(monitor_comp_dir, exist_ok=True)
    monitor_main_file = os.path.join(monitor_comp_dir, 'monitor.nc')

    # Create the data_object Component to send back to SMS
    monitor_do_comp = Component(
        comp_uuid=comp_uuid, main_file=monitor_main_file, model_name='SRH-2D', unique_name='Monitor_Component'
    )

    data = MonitorData(monitor_main_file)
    data.info.attrs['cov_uuid'] = cov_uuid
    data.commit()

    return monitor_do_comp
