"""BcCoverageLooper class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.bridge.bridge_component import BridgeComponent, BridgeIo

# 4. Local modules
from xms.srh.file_io.report import plots, report_util
from xms.srh.file_io.report.coverage_looper_base import CoverageLooperBase


class BridgeCoverageLooper(CoverageLooperBase):
    """Adds bridge coverages to the report."""
    def __init__(self, notes_db, report_dir, query, logger):  # pragma: no cover
        """Initializes the class.

        Args:
            notes_db (:obj:`Notes`): Notes object.
            report_dir (:obj:`str`): Path to directory where report files are created.
            query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS
            logger(:obj:`logger`): The logger
        """
        super().__init__(notes_db, report_dir, query, '3D Bridge', 'Bridge', '3D Bridge')
        self._logger = logger

    def _create_component(self, main_file):  # pragma: no cover
        """Constructs and returns the SRH component given its main file.

        Args:
            main_file: The main file associated with this component.

        Returns:
            See description.
        """
        return BridgeComponent(main_file)

    def _store_coverage_data(self, coverage, component):  # pragma: no cover  # noqa: C901  (too complex)
        """Main method to extract data from the coverage and store it in a dict for use with jinja.

        Args:
            coverage (:obj:`xms.data_objects.parameters.Coverage`): The coverage.
            component (:obj:`BridgeComponent`): The bridge component.

        Returns:
            (:obj:`dict`):A dict of the coverage data for use with jinja.
        """
        return self.get_jinja(coverage, component, self._notes_db, self._report_dir, self._logger)

    @staticmethod
    def get_jinja(coverage, component, notes_db, report_dir, logger):  # pragma: no cover  # noqa: C901  (too complex)
        """Main method to extract data from the coverage and store it in a dict for use with jinja.

        Args:
            coverage (:obj:`xms.data_objects.parameters.Coverage`): The coverage.
            component (:obj:`BridgeComponent`): The bridge component.
            notes_db (:obj:`Notes`): Notes object.
            report_dir (:obj:`str`): Path to directory where report files are saved.
            logger(:obj:`logger`): The logger.

        Returns:
            (:obj:`dict`):A dict of the coverage data for use with jinja.
        """
        # Store all jinja stuff in a dict
        coverage_jinja = {'name': coverage.name}
        report_util.add_object_notes(notes_db, coverage.uuid, coverage_jinja)

        # Read the bridge
        bridge_io = BridgeIo()
        bridge = bridge_io.read(component.main_file)

        # scale the curves 0.0 to 1.0 and then scale them to the max x value
        x = 'Distance'
        y = 'Elevation'
        top, up, down = bridge.df_top, bridge.df_upstream, bridge.df_downstream
        max_x = max(top[x].max(), up[x].max(), down[x].max())
        top[x] /= top[x].max()
        up[x] /= up[x].max()
        down[x] /= down[x].max()
        top[x] *= max_x
        up[x] *= max_x
        down[x] *= max_x

        # Get the x,y data into a dict for convenience
        xy_data = {
            'top': {
                'x': top[x].tolist(),
                'y': top[y].tolist()
            },
            'up': {
                'x': up[x].tolist(),
                'y': up[y].tolist()
            },
            'down': {
                'x': down[x].tolist(),
                'y': down[y].tolist()
            },
        }

        if logger:
            logger.info(f'Creating plot for bridge coverage "{coverage.name}"')
        coverage_jinja['plot'] = plots.plot_bridge(xy_data, coverage.name, report_dir)

        return coverage_jinja
