"""Functions used in other report code, put here to avoid circular dependencies."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import math
import os
import sys

# 2. Third party modules

# 3. Aquaveo modules
import xms.gdal.utilities.gdal_utils as gu

# 4. Local modules


def make_filename_unique(filepath):
    """Appends a suffix to the file if needed to make it unique.

    Example: file.txt becomes file(2).txt

    Args:
        filepath (:obj:`str`): Filepath to consider.

    Returns:
        The unique filename.
    """
    if not os.path.exists(filepath):
        return filepath

    path, name = os.path.split(filepath)
    name, extension = os.path.splitext(name)

    for i in range(2, sys.maxsize):
        unique_filename = os.path.join(path, f'{name}({i}){extension}')
        if not os.path.exists(unique_filename):
            return unique_filename

    return None  # pragma: no cover


def transform_points(points, current_crs_wkt, new_crs_wkt):
    """Transforms the points from their current CRS (coordinate reference system) to the new one.

    See https://gis.stackexchange.com/questions/226200/how-to-call-gdaltransform-with-its-input

    Args:
        points (:obj:`list`): List of points.
        current_crs_wkt (:obj:`str`): The well known text of the CRS the points are in.
        new_crs_wkt (:obj:`str`): The well known text of the new CRS the points will be transformed to.

    Returns:
        (:obj:`tuple`): tuple containing:

            rv (:obj:`bool`): Return value.

            new_points (:obj:`list`): The transformed points.
    """
    rv = True
    try:
        # Set up the source and target coordinate systems
        transform = gu.get_coordinate_transformation(
            current_crs_wkt, new_crs_wkt, keep_vertical=True, set_traditional_axis_mapping=True
        )

        # Transform each point
        new_points = []
        for point in points:
            if len(point) == 2:
                coords = transform.TransformPoint(point[0], point[1])
                new_points.append(coords[0:2])
            else:
                coords = transform.TransformPoint(point[0], point[1], point[2])
                new_points.append(coords[0:3])
    except Exception:  # pragma: no cover
        logger = logging.getLogger("Summary Report")
        logger.warning(f'Error transforming points from\n{current_crs_wkt}\nto\n{new_crs_wkt}')
        new_points = points
        rv = False

    return rv, new_points


def transform_points_to_geographic(points, current_crs_wkt):
    """Transforms the points to geographic (longitude, latitude).

    Args:
        points (:obj:`list`): List of points.
        current_crs_wkt (:obj:`str`): The well known text of the CRS (coordinate reference systsem) the points are in.

    Returns:
        new_points (:obj:`list`): The transformed points.
    """
    if current_crs_wkt.lower().startswith('local_cs'):  # pragma: no cover
        logger = logging.getLogger("Summary Report")
        logger.warning('Cannot transform points from local to geographic.')
        return False, points
    _, epsg_wkt = gu.wkt_from_epsg(4326)
    return transform_points(points, current_crs_wkt, epsg_wkt)


def utm_zone_from_geographic(point):
    """Returns the UTM zone given a point in geographic (longitude, latitude).

    Longitude entered in DEGREES!!! Degrees West should be NEGATIVE!!!
    See laCalcUTMZone().

    Args:
        point: List or tuple with x,y (if z is included it's ignored) in longitude,latitude.

    Returns:
        The UTM zone. Always positive.
    """
    tol = 1e-7
    zone = int(((point[0] + 180.0) / 6.0) + 1.0)
    if zone > 60:
        zone = 60
    # If right on the line and in the western hemisphere, make it the lower zone,
    # the higher negative longitudes
    if point[0] < 0.0 and math.fabs(math.fmod(point[0], 6.0)) < tol:
        zone -= 1
        if zone == 0:
            zone = 60

    # Special Cases for Norway & Svalbard
    # https://en.wikipedia.org/wiki/Universal_Transverse_Mercator_coordinate_system#Exceptions
    if point[1] >= 56.0 and zone == 31 and point[1] < 64 and point[0] >= 3:
        zone = 32
    elif point[1] >= 72 and zone == 32 and point[0] < 9:
        zone = 31
    elif point[1] >= 72 and zone == 32 and point[0] >= 9:
        zone = 33
    elif point[1] >= 72 and zone == 34 and point[0] < 21:
        zone = 33
    elif point[1] >= 72 and zone == 34 and point[0] >= 21:
        zone = 35
    elif point[1] >= 72 and zone == 36 and point[0] < 33:
        zone = 35
    elif point[1] >= 72 and zone == 36 and point[0] >= 33:
        zone = 37
    return zone


def utm_wkt_from_geographic(point, geographic_wkt):
    """Returns the well known text for the UTM CRS (coordinate reference system) for the UTM zone containing the point.

    See imGeographicSystemToUTM().

    Args:
        point: tuple or list with x,y (longitude,latitude) coordinate of a point in geographic coordinates.
        geographic_wkt (:obj:`str`): Geographic coordinate system well known text.

    Returns:
        utm_wkt (:obj:`str`): The well known text of the UTM crs.
    """
    crs = gu.wkt_to_sr(geographic_wkt)
    zone = utm_zone_from_geographic(point)
    north = 1 if point[1] >= 0.0 else 0
    crs.SetUTM(zone, north)
    utm_wkt = crs.ExportToWkt()
    return utm_wkt


def add_object_notes(notes_db, uuid, object_dict):  # pragma: no cover
    """Adds the notes for the object with uuid to the object_dict under object_dict['notes'].

    Args:
        notes_db (:obj:`Notes`): Notes object.
        uuid (:obj:`str`): UUID of the object.
        object_dict(:obj:`dict`): Dict with jinja info about the object.
    """
    if notes_db:
        notes_jinja = []
        try:
            notes = notes_db.get_notes(uuid)
        except RuntimeError:
            notes = []
        for note in notes:
            note_jinja = {'type': note[0], 'date_time': note[1], 'note': note[2]}
            notes_jinja.append(note_jinja)
        object_dict['notes'] = notes_jinja
