"""StructureCoverageLooper class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from logging import Logger

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.bridge.structure_component import StructureComponent
from xms.coverage import coverage_util
from xms.data_objects.parameters import Coverage
from xms.guipy.notes import Notes

# 4. Local modules
from xms.srh.file_io.report.coverage_looper_base import CoverageLooperBase


class StructureCoverageLooper(CoverageLooperBase):
    """Gets data from structure coverages."""
    def __init__(self, notes_db: Notes | None, report_dir: str, query: Query, logger: Logger | None):
        """Initializes the class.

        Args:
            notes_db: Notes object.
            report_dir: Path to directory where report files are created.
            query: Object for communicating with SMS
            logger: The logger
        """
        super().__init__(
            notes_db,
            report_dir,
            query,
            coverage_type='3D Structure',
            component_unique_name='Structure',
            model_name='3D Bridge'
        )
        self._logger = logger

    def _create_component(self, main_file: str):
        """Constructs and returns the SRH component given its main file.

        Args:
            main_file: The main file associated with this component.

        Returns:
            See description.
        """
        return StructureComponent(main_file)

    def _store_coverage_data(self, coverage: Coverage, component):
        """Main method to extract data from the coverage and store it in a dict for use with jinja.

        Args:
            coverage: The coverage.
            component: The bridge component.

        Returns:
            A dict of the coverage data for use with jinja.
        """
        return self.get_jinja(coverage, component)

    @staticmethod
    def get_jinja(coverage: Coverage, component):
        """Main method to extract data from the coverage and store it in a dict for use with jinja.

        Args:
            coverage: The coverage.

        Returns:
            A dict of the coverage data for use with jinja.
        """
        # Store all jinja stuff in a dict
        extents = coverage_util.coverage_extents(coverage)
        average_x = (extents[0][0] + extents[1][0]) / 2
        average_y = (extents[0][1] + extents[1][1]) / 2
        coverage_jinja = {
            'name': coverage.name,
            'extents': [average_x, average_y],
            'structure_type': component.data.data_dict['structure_type']
        }

        return coverage_jinja
