"""Module for the control file writing function."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
import typing

# 2. Third party modules

# 3. Aquaveo modules
from xms.srh.file_io import file_writer as srh_file_writer
from xms.swmm.dmi.xms_data import SwmmData
from xms.swmm.file_io import input_file_writer as swmm_input_file_writer

# 4. Local modules
from xms.srh_swmm_coupler.dmi.xms_data import CouplerData
from xms.srh_swmm_coupler.file_io import util


def write(xms_data: CouplerData, out_dir: typing.Optional[str | Path] = None):
    """Writes the SWMM input file.

    Args:
        xms_data (CouplerData): Simulation data.
        out_dir: The output directory.
    """
    swmm_data = SwmmData(xms_data.query)
    swmm_data.sim_uuid = xms_data.swmm_sim[0].uuid
    cur_dir = out_dir
    if out_dir is None:
        component_main_file = xms_data.swmm_sim[1].main_file
        cur_dir = Path(component_main_file).parent
    swmm_input_file_writer.write(swmm_data, Path(cur_dir) / 'swmm.inp', util.get_logger())
    if out_dir is None:
        component_main_file = xms_data.srh_sim[1].main_file
        cur_dir = Path(component_main_file).parent
    srh_file_writer.write(xms_data.query, cur_dir, util.get_logger(), xms_data.srh_sim[0].uuid)
