"""Module for exporting a CMS-Wave simulation with feedback."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg, ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists

# 4. Local modules
from xms.stwave.file_io.export_sim import ExportSim


class ExportSimulationRunner(ProcessFeedbackThread):
    """Class for exporting a CMS-Wave simulation in a worker thread."""

    def __init__(self, parent):
        """Constructor.

        Args:
            parent (QWidget): Parent of the QThread, probably the hidden parent dialog created by XMS.
        """
        super().__init__(parent=parent, do_work=self._do_work)

    def _do_work(self):
        """Creates the snap preview of coverages onto the mesh."""
        xms_data = {}
        exporter = ExportSim(xms_data)
        exporter.export()


def export_simulation_with_feedback(worker=None):
    """Export a simulation with a feedback dialog.

    Args:
        worker (ExportSimulationRunner): Tool replacement for mocking.
    """
    ensure_qapplication_exists()
    if worker is None:
        worker = ExportSimulationRunner(None)
    display_text = {
        'title': 'STWAVE Export Simulation',
        'working_prompt': 'Exporting STWAVE simulation files. Please wait...',
        'warning_prompt': 'Warning(s) encountered while exporting simulation. Review log output for more details.',
        'error_prompt': 'Error(s) encountered while exporting simulation. Review log output for more details.',
        'success_prompt': 'Successfully exported simulation',
        'note': '',
        'auto_load': 'Close this dialog automatically when exporting is finished.'
    }
    feedback_dlg = ProcessFeedbackDlg(display_text=display_text, logger_name='xms', worker=worker,
                                      parent=None)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec():
        if LogEchoQSignalStream.logged_error:
            XmEnv.report_export_error()
    else:
        XmEnv.report_export_aborted()
