"""Data class for TidalComponent."""

# 1. Standard Python modules
import datetime

# 2. Third party modules
import numpy as np
import pkg_resources
from PySide2.QtCore import QDateTime
import xarray as xr

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.core.filesystem import filesystem as io_util
from xms.guipy.time_format import ISO_DATETIME_FORMAT, qdatetime_to_datetime

# 4. Local modules


ADCIRC_INDEX = 0
LEPROVOST_INDEX = 1
FES2014_INDEX = 2
TPX08_INDEX = 3
TPX09_INDEX = 4
USER_DEFINED_INDEX = 5

DEFAULT_TIDAL_DATA_FILE = 'tidal_comp.nc'

TDB_SOURCES = [
    'ADCIRC2015',
    'LeProvost',
    'FES2014',
    'TPXO8',
    'TPXO9',
    'User defined',
]


USER_SPECIFIED_OPTION = 'User specified'


STANDARD_CONSTITUENTS = [
    'M2',
    'S2',
    'N2',
    'K1',
    'M4',
    'O1',
    'M6',
    'NU2',
    'MU2',
    '2N2',
    'Q1',
    'T2',
    'P1',
    'L2',
    'K2',
    USER_SPECIFIED_OPTION,
]


class TidalData(XarrayBase):
    """Manages data file for the tidal constituent component."""

    def __init__(self, data_file):
        """Initializes the data class.

        Args:
            data_file (str): The netcdf file (with path) associated with this instance data. Probably the owning
                component's main file.
        """
        # base class constructor
        super().__init__(data_file)
        self._user = None
        self._cons = None
        if self._info.attrs['FILE_TYPE'] == 'UNDEFINED':
            # this is the fist time the data has been created for this instance of the component
            self._create_default_data()
        else:
            self._check_for_simple_migrations()

    def _check_for_simple_migrations(self):
        """Check for migrations that do not need an XML-based migration script."""
        # Make sure the reftime string representation is locale agnostic.
        try:  # Try to parse using current locale
            qreftime = QDateTime.fromString(self.info.attrs['reftime'])
            reftime = qdatetime_to_datetime(qreftime)
        except Exception:
            try:  # Try to parse using ISO format
                reftime = datetime.datetime.strptime(self.info.attrs['reftime'], ISO_DATETIME_FORMAT)
            except Exception:  # Set default to 01/01/1950 epoch
                reftime = datetime.datetime(1950, 1, 1)
        self.info.attrs['reftime'] = reftime.strftime(ISO_DATETIME_FORMAT)  # Reformat to ISO datetime representation
        if 'run_duration' not in self.info.attrs:
            self.info.attrs['run_duration'] = 0.0
        if 'Velocity Amplitude X' not in self.user:
            user = self.user
            dimension = user.sizes['index'] if 'index' in user.sizes else 0
            default_value = np.array([''] * dimension, dtype=np.unicode_)
            user['Force Velocity'] = xr.DataArray(data=np.array([0] * dimension, dtype=np.int_))
            user['Velocity Amplitude X'] = xr.DataArray(default_value)
            user['Velocity Amplitude Y'] = xr.DataArray(default_value)
            user['Velocity Phase X'] = xr.DataArray(default_value)
            user['Velocity Phase Y'] = xr.DataArray(default_value)

    @property
    def user(self):
        """Load the output parameters dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the output parameters in the main file
        """
        if self._user is None:
            self._user = self.get_dataset('user', False)
        return self._user

    @user.setter
    def user(self, dset):
        """Setter for the output dataset attribute."""
        if dset:
            self._user = dset

    @property
    def cons(self):
        """Load the tidal constituent parameters dataset from disk.

        Returns:
            xarray.Dataset: Dataset interface to the tidal constituent parameters in the main file
        """
        if self._cons is None:
            self._cons = self.get_dataset('cons', False)
        return self._cons

    @cons.setter
    def cons(self, dset):
        """Setter for the tidal constituent dataset attribute."""
        if dset:
            self._cons = dset

    def _create_default_data(self):
        """Create the default data values."""
        self._info.attrs['FILE_TYPE'] = 'TIDE_CONST'
        self._info.attrs['VERSION'] = pkg_resources.get_distribution('xmstides').version
        self._info.attrs['source'] = LEPROVOST_INDEX
        self._info.attrs['reftime'] = ''
        self._info.attrs['run_duration'] = 0.0  # Hours
        user_data = {
            'Constituent': xr.DataArray(data=np.array([], dtype=np.int32)),
            'Name': xr.DataArray(data=np.array([], dtype=np.unicode_)),
            'Tidal __new_line__ Potential __new_line__ Amplitude': xr.DataArray(data=np.array([], dtype=np.float64)),
            'Frequency': xr.DataArray(data=np.array([], dtype=np.float64)),
            'Nodal __new_line__ Factor': xr.DataArray(data=np.array([], dtype=np.float64)),
            'Equilibrium __new_line__ Argument': xr.DataArray(data=np.array([], dtype=np.float64)),
            'Earth Tide __new_line__ Reduction __new_line__ Factor': xr.DataArray(data=np.array([], dtype=np.float64)),
            'Amplitude': xr.DataArray(data=np.array([], dtype=np.unicode_)),
            'Phase': xr.DataArray(data=np.array([], dtype=np.unicode_)),
            'Force Velocity': xr.DataArray(data=np.array([], dtype=np.int_)),
            'Velocity Amplitude X': xr.DataArray(data=np.array([], dtype=np.unicode_)),
            'Velocity Amplitude Y': xr.DataArray(data=np.array([], dtype=np.unicode_)),
            'Velocity Phase X': xr.DataArray(data=np.array([], dtype=np.unicode_)),
            'Velocity Phase Y': xr.DataArray(data=np.array([], dtype=np.unicode_)),
        }
        self._user = xr.Dataset(data_vars=user_data)
        cons = {
            'name': xr.DataArray(data=np.array([], dtype=np.unicode_)),
            'enabled': xr.DataArray(data=np.array([], dtype=np.int32)),
        }
        self._cons = xr.Dataset(data_vars=cons)
        self.commit()

    def commit(self):
        """Save current in-memory component parameters to data file."""
        super().commit()  # Recreates the NetCDF file if vacuuming
        if self._user is not None:
            self._user.close()
            self._drop_h5_groups(['user'])
            self._user.to_netcdf(self._filename, group='user', mode='a')
        if self._cons is not None:
            self._cons.close()
            self._drop_h5_groups(['cons'])
            self._cons.to_netcdf(self._filename, group='cons', mode='a')

    def vacuum(self):
        """Rewrite all SimData to a new/wiped file to reclaim disk space.

        All tidal constituent datasets that need to be written to the file must be loaded into memory before calling
        this method.
        """
        if self._info is None:
            self._info = self.get_dataset('info', False)
        if self._user is None:
            self._user = self.get_dataset('user', False)
        if self._cons is None:
            self._cons = self.get_dataset('cons', False)
        io_util.removefile(self._filename)  # Delete the existing NetCDF file
        self.commit()  # Rewrite all datasets

    @staticmethod
    def get_default_con_dataset(model):
        """Get a default constituent dataset for a Tidal database model.

        Args:
            model (harmonica.Constituents): The database model

        Returns:
            xarray.Dataset: See description
        """
        available_cons = list(model._current_model.resources.available_constituents())
        con_data = {
            'name': xr.DataArray(data=np.array(available_cons, dtype=np.unicode_)),
            'enabled': xr.DataArray(data=np.array([0 for _ in available_cons], dtype=np.int32)),
        }
        return xr.Dataset(data_vars=con_data)
