"""AngleConvention Algorithm."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from logging import Logger
import warnings

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import Grid
from xms.constraint.ugrid_activity import CellToPointActivityCalculator
from xms.datasets import vectors as xmdv
from xms.datasets.dataset_reader import DatasetReader
from xms.datasets.dataset_writer import DatasetWriter

# 4. Local modules
from xms.tool.utilities.dataset_tool import set_builder_activity_flags


def angle_convention(dataset_name: str, dataset_reader: DatasetReader, dset_grid: Grid, input_type: str,
                     output_type: str, logger: Logger) -> DatasetWriter:
    """Converts an angle convention dataset to another angle convention.

    Args:
        dataset_name: The name for the output dataset
        dataset_reader: Dataset reader for the input dataset.
        dset_grid: The dataset's constrained grid.
        input_type: The type being converted from
        output_type: The type being converted to
        logger: Logger for user output.

    Returns:
        The DatasetWriter for the output dataset
    """
    # Create a place for the output dataset file
    builder = DatasetWriter()
    builder.name = dataset_name
    builder.geom_uuid = dataset_reader.geom_uuid
    builder.num_components = 1
    builder.ref_time = dataset_reader.ref_time
    builder.time_units = dataset_reader.time_units
    builder.null_value = dataset_reader.null_value

    # Get is_vector and conversion_type
    is_vector = dataset_reader.num_components == 2

    # Input type should be Cartesian when input dataset is vectors
    if is_vector:
        input_type = 'Cartesian'

    conversion_type = f'{input_type} to {output_type}'

    # Convert angle convention dataset to other angle convention.
    dset_ugrid = None
    if dset_grid:
        dset_ugrid = dset_grid.ugrid

    if dataset_reader.activity and dataset_reader.values and dset_ugrid:
        if dataset_reader.values.shape != dataset_reader.activity.shape:
            # Nodal dataset values with cell activity
            dataset_reader.activity_calculator = CellToPointActivityCalculator(dset_ugrid)
            builder.activity_calculator = CellToPointActivityCalculator(dset_ugrid)

    time_count = len(dataset_reader.times)
    for tsidx in range(time_count):
        logger.info(f'Processing time step {tsidx + 1} of {time_count}...')
        data, activity = dataset_reader.timestep_with_activity(tsidx)
        set_builder_activity_flags(activity, builder)

        if is_vector:  # Convert Vx/Vy to Cartesian directions
            data = xmdv.vx_vy_to_direction(data[:, 0], data[:, -1])

        with warnings.catch_warnings():  # Ignore numpy warning about nan (inactive) values in input dataset.
            warnings.filterwarnings(action='ignore', message='invalid value encountered in remainder')
            data = _convert_angle(data, conversion_type)

        builder.append_timestep(dataset_reader.times[tsidx], data, activity)

    # Write output direction dataset to XMDF file
    logger.info('Writing output direction dataset to XMDF file...')
    builder.appending_finished()

    return builder


def _convert_angle(data, conversion_type):
    """Convert a scalar dataset of direction angles to a specified convention.

    Args:
        data (np.ndarray): The dataset values
        conversion_type: Conversion string: '{from_convention} to {to_convention}'

    Returns:
        np.ndarray: The input direction dataset in the specified convention
    """
    if conversion_type == 'Cartesian to Meteorologic':
        data = (270.0 - data) % 360.0
    elif conversion_type == 'Cartesian to Oceanographic':
        data = (90.0 - data) % 360.0
    elif conversion_type == 'Meteorologic to Cartesian':
        data = (270.0 - data) % 360.0
    elif conversion_type == 'Meteorologic to Oceanographic':
        data = (data - 180.0) % 360.0
    elif conversion_type == 'Oceanographic to Cartesian':
        data = (90.0 - data) % 360.0
    elif conversion_type == 'Oceanographic to Meteorologic':
        data = (180.0 + data) % 360.0
    return data
