"""ContoursFromRasterTool whitebox tool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.gdal.rasters import raster_utils as ru
from xms.gdal.vectors import VectorInput
from xms.tool_core import IoDirection, Tool
from xms.tool_core.table_definition import FloatColumnType, TableDefinition

# 4. Local modules
from xms.tool.utilities.coverage_conversion import convert_lines_to_coverage

VALUE_HEADER = "Value"

ARG_INPUT_RASTER = 0
ARG_CONTOUR_VALUES = 1
ARG_OUTPUT_COVERAGE = 2


def _create_table_definition():
    """Creates a blank/default table definition."""
    tool_tips = ["Value"]
    columns = [
        FloatColumnType(header=VALUE_HEADER, tool_tip=tool_tips[0], default=0.0)
    ]
    return TableDefinition(columns)


class ContoursFromRasterTool(Tool):
    """ContoursFromRasterTool whitebox tool class."""

    def __init__(self):
        """Initializes the class."""
        super().__init__('Contours from Raster')
        self._raster = None

    def initial_arguments(self):
        """Get initial arguments for tool.

        Returns:
            (list): A list of the initial tool arguments.
        """
        table_def = _create_table_definition()
        df = table_def.to_pandas()
        arguments = [
            self.raster_argument(name='input_raster', description='Input raster'),
            self.table_argument(name='contour_values', description='Contour Values', value=df, optional=False,
                                table_definition=table_def),
            self.coverage_argument(name='output_coverage', description='Output coverage',
                                   io_direction=IoDirection.OUTPUT)
        ]
        return arguments

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}
        self._raster = self.get_input_raster(arguments[ARG_INPUT_RASTER].value)
        z_min, z_max = self._raster.get_z_min_max()
        values = arguments[ARG_CONTOUR_VALUES].value
        for value in values[VALUE_HEADER]:
            if not z_min < value < z_max:
                errors['contour_values'] = 'The contour elevations need to be set to a value within the range ' \
                                           f'of elevations for the raster. {value} is not within the range of the ' \
                                           'raster elevations.'
                break
        return errors

    def run(self, arguments):
        """Runs the tool.

        Args:
            arguments (list): A list of the tool's arguments.
        """
        # shp_filename = 'c:/temp/contours.shp'
        shp_filename = '/vsimem/contours.shp'
        values = arguments[ARG_CONTOUR_VALUES].value
        ru.raster_to_shapefile_contours(self._raster, np.sort(values[VALUE_HEADER]).tolist(), shp_filename)
        vi = VectorInput(shp_filename)
        cov_geometry = vi.get_line_features()
        out_cov = arguments[ARG_OUTPUT_COVERAGE].value
        new_cov = convert_lines_to_coverage(cov_geometry, out_cov, self._raster.wkt, self.default_wkt)
        if new_cov is not None:
            self.set_output_coverage(new_cov, arguments[ARG_OUTPUT_COVERAGE])
