"""GeometryGradientTool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.tool_core import ALLOW_ONLY_SCALARS, IoDirection, Tool

# 4. Local modules
from xms.tool.algorithms.datasets.geometry_gradient import GeometryGradient

ARG_INPUT_DSET = 0
ARG_VECTOR = 1
ARG_MAGNITUDE = 2
ARG_DIRECTION = 3
ARG_VECTOR_DATASET = 4
ARG_MAGNITUDE_DATASET = 5
ARG_DIRECTION_DATASET = 6


class GeometryGradientTool(Tool):
    """Tool to compute geometry gradient datasets."""

    def __init__(self):
        """Initializes the class."""
        super().__init__(name='Geometry Gradient')
        self._dataset_reader = None
        self._gradient_builder = None
        self._gradient_mag_builder = None
        self._gradient_dir_builder = None
        self._dset_grid = None
        self._cell_plane_vals = []

        # Tin will be created from input dataset's geometry
        self._points = None
        # Output dataset values filled time step at a time
        self._ts_data = None
        self._vx = []
        self._vy = []

    def initial_arguments(self):
        """Get initial arguments for tool.

        Must override.

        Returns:
            (list): A list of the initial tool arguments.
        """
        arguments = [
            self.dataset_argument(name='input_dataset', description='Input dataset', filters=ALLOW_ONLY_SCALARS),
            self.bool_argument(name='gradient_vec_bool', description='Gradient vector', value=True),
            self.bool_argument(name='gradient_mag_bool', description='Gradient magnitude', value=True),
            self.bool_argument(name='gradient_dir_bool', description='Gradient direction', value=True),
            self.dataset_argument(name='vector_name', description='Output gradient vector dataset name',
                                  value='Gradient vector', io_direction=IoDirection.OUTPUT, optional=True),
            self.dataset_argument(name='magnitude_name', description='Output gradient magnitude dataset name',
                                  value='Gradient magnitude', io_direction=IoDirection.OUTPUT, optional=True),
            self.dataset_argument(name='direction_name', description='Output gradient direction dataset name',
                                  value='Gradient direction', io_direction=IoDirection.OUTPUT, optional=True),
        ]
        self.enable_arguments(arguments)
        return arguments

    def validate_arguments(self, arguments):
        """Called to determine if arguments are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}

        # Make sure at least one scalar output selected.
        vec_enabled = arguments[ARG_VECTOR].value
        mag_enabled = arguments[ARG_MAGNITUDE].value
        dir_enabled = arguments[ARG_DIRECTION].value
        if not vec_enabled and not mag_enabled and not dir_enabled:
            errors[arguments[ARG_VECTOR].name] = 'No output dataset selected.'
            errors[arguments[ARG_MAGNITUDE].name] = 'No output dataset selected.'
            errors[arguments[ARG_DIRECTION].name] = 'No output dataset selected.'

        # Validate input datasets
        self._dataset_reader = self._validate_input_dataset(arguments[ARG_INPUT_DSET], errors)
        if self._dataset_reader is not None and self._dataset_reader.location != 'points':
            errors[arguments[ARG_INPUT_DSET].name] = 'Only point datasets maybe used with this tool.'

        return errors

    def enable_arguments(self, arguments):
        """Called to show/hide arguments, change argument values and add new arguments.

        Args:
            arguments(list): The tool arguments.
        """
        # Show/hide the datasets as needed.
        arguments[ARG_VECTOR_DATASET].hide = not arguments[ARG_VECTOR].value
        arguments[ARG_MAGNITUDE_DATASET].hide = not arguments[ARG_MAGNITUDE].value
        arguments[ARG_DIRECTION_DATASET].hide = not arguments[ARG_DIRECTION].value

    def run(self, arguments):
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """
        cogrid = self.get_input_dataset_grid(arguments[ARG_INPUT_DSET].text_value)

        algorithm = GeometryGradient(arguments[ARG_VECTOR].value, arguments[ARG_MAGNITUDE].value,
                                     arguments[ARG_DIRECTION].value, arguments[ARG_VECTOR_DATASET].text_value,
                                     arguments[ARG_MAGNITUDE_DATASET].text_value,
                                     arguments[ARG_DIRECTION_DATASET].text_value, cogrid, self._dataset_reader,
                                     self.logger)

        try:
            self._gradient_builder, self._gradient_mag_builder, self._gradient_dir_builder = \
                algorithm.geometry_gradient()
        except RuntimeError as e:
            self.fail(str(e))
        else:
            self._add_output_datasets()

    def _add_output_datasets(self):
        """Add datasets created by the tool to be sent back to XMS."""
        if self._gradient_builder is not None:
            # Send the dataset back to XMS
            self.set_output_dataset(self._gradient_builder)
        if self._gradient_mag_builder is not None:
            # Send the dataset back to XMS
            self.set_output_dataset(self._gradient_mag_builder)
        if self._gradient_dir_builder is not None:
            # Send the dataset back to XMS
            self.set_output_dataset(self._gradient_dir_builder)

# def main():
#     """Main function, for testing."""
#     from xms.tool_gui.tool_dialog import ToolDialog
#     from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists
#
#     qapp = ensure_qapplication_exists()
#     tool = GeometryGradientTool()
#     tool.set_grid_uuid_for_testing('cimarron_gradient', '85755c32-a85a-4cd7-b7ec-c5903d7aabd8')
#     arguments = tool.initial_arguments()
#
#     arguments[GRADIENT_INPUT_DSET_IDX].value = 'cimarron_gradient'
#     # arguments[GRADIENT_VECTOR_IDX].value = True
#     # arguments[GRADIENT_MAG_IDX].value = True
#     # arguments[GRADIENT_DIR_IDX].value = True
#     arguments[GRADIENT_OUTPUT_PREFIX_IDX].value = 'cimarron_out_activity'
#     tool_dialog = ToolDialog(None, arguments, tool.name, tool=tool)
#     if tool_dialog.exec():
#         tool.run_tool(tool_dialog.tool_arguments)
#
#
# if __name__ == "__main__":
#     main()
