"""CoverageBuilder class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import uuid

# 2. Third party modules
from geopandas import GeoDataFrame
from shapely import LineString, Point, Polygon

# 3. Aquaveo modules

# 4. Local modules


class CoverageBuilder:
    """Class to construct a coverage as a GeoDataFrame."""

    def __init__(self, cov_wkt: str = None, cov_name: str = None, cov_uuid: str = None):
        """Construct a CoverageBuilder object.
        """
        self._geometry_list = []
        self._geometry_types = []
        self._polygon_arc_ids = []
        self._polygon_arc_directions = []
        self._interior_arc_ids = []
        self._interior_arc_directions = []
        self._start_node = []
        self._end_node = []
        self._geometry_ids = []
        self._cur_point_id = 1
        self._cur_arc_id = 1
        self._cur_poly_id = 1
        self._cov_wkt = cov_wkt
        if cov_name is None:
            self._cov_name = "new coverage"
        else:
            self._cov_name = cov_name
        if cov_uuid is None:
            self._cov_uuid = str(uuid.uuid4())
        else:
            self._cov_uuid = cov_uuid
        self._point_to_id = {}

        # Mappings of feature ids to indices in self._geometry_list and self._geometry_ids.
        self._node_id_to_idx = {}
        self._arc_id_to_idx = {}

    def add_arcs(self, coordinates_list: list):
        """Adds arcs to the coverage given the coordinate lists.

        Args:
            coordinates_list (list): Coordinates of the arcs to add to the coverage.
        """
        for coordinates in coordinates_list:
            self.add_arc(coordinates)

    def add_arc(self, arc_coordinates: list, arc_id: int = -1) -> int:
        """Adds an arc to the coverage given the arc's coordinates.

        Args:
            arc_coordinates (list): Coordinates of an arc to add to the coverage.
            arc_id (int): The optional arc ID if you want to set the arc's ID (not recommended).

        Return:
            The arc ID
        """
        start_node = self.get_node(arc_coordinates[0])
        for vertex in arc_coordinates[1:-1]:
            self.add_vertex(vertex)
        end_node = self.get_node(arc_coordinates[-1])
        self._geometry_ids.append(self._cur_arc_id if arc_id == -1 else arc_id)
        self._arc_id_to_idx[self._geometry_ids[-1]] = len(self._geometry_list)  # Haven't appended to _geometry_list yet
        self._cur_arc_id += 1
        self._geometry_types.append('Arc')
        self._geometry_list.append(LineString(arc_coordinates))
        self._polygon_arc_ids.append([])
        self._polygon_arc_directions.append([])
        self._interior_arc_ids.append([])
        self._interior_arc_directions.append([])
        self._start_node.append(start_node)
        self._end_node.append(end_node)
        return self._geometry_ids[-1]

    def add_polygon(self, outside_points: list, inside_points: list | None = None, outside_arcs: list | None = None,
                    interior_arcs: list | None = None, poly_id: int = -1) -> int:
        """Adds a polygon to the coverage given the polygon's outside and inside coordinates.

        Args:
            outside_points (list): List of lists containing the coordinates of the outside arcs defining the polygon.
            inside_points (list | None): List of lists of lists containing the coordinates of the inside polygon arcs.
            outside_arcs (list | None): List of arc IDs containing the IDs of the outside arcs defining the polygon.
            interior_arcs (list | None): List of lists of arc IDs containing the IDs of the inside polygon arcs.
            poly_id (int): The optional polygon ID if you want to set the polygon's ID (not recommended).

        Return:
            The polygon ID
        """
        polygon_arc_ids = []
        polygon_arc_directions = []
        interior_arc_ids = []
        interior_arc_directions = []
        poly_outside_pts = []
        poly_inside_pts = []
        for arc in outside_points:
            if outside_arcs is None:
                polygon_arc_ids.append(self._cur_arc_id)
                self.add_arc(arc)
            # Outside arcs have direction of "1" if not reversed
            polygon_arc_directions.append(1)
            poly_outside_pts.extend(arc)
        if inside_points is not None:
            for polygon in inside_points:
                poly_arc_ids = []
                poly_arc_directions = []
                poly_pts = []
                for arc in polygon:
                    if interior_arcs is None:
                        poly_arc_ids.append(self._cur_arc_id)
                        self.add_arc(arc)
                    # Inside arcs have direction of "1" if not reversed
                    poly_arc_directions.append(1)
                    poly_pts.extend(arc)
                if interior_arcs is None:
                    interior_arc_ids.append(poly_arc_ids)
                interior_arc_directions.append(poly_arc_directions)
                poly_inside_pts.append(poly_pts)
        self._geometry_ids.append(self._cur_poly_id if poly_id == -1 else poly_id)
        self._cur_poly_id += 1
        self._geometry_types.append('Polygon')
        self._geometry_list.append(Polygon(poly_outside_pts, poly_inside_pts))
        if outside_arcs is None:
            self._polygon_arc_ids.append(polygon_arc_ids)
        else:
            self._polygon_arc_ids.append(outside_arcs)
        self._polygon_arc_directions.append(polygon_arc_directions)
        if interior_arcs is None:
            self._interior_arc_ids.append(interior_arc_ids)
        else:
            self._interior_arc_ids.append(interior_arcs)
        self._interior_arc_directions.append(interior_arc_directions)
        self._start_node.append(-1)
        self._end_node.append(-1)
        return self._geometry_ids[-1]

    def get_node(self, node_coordinates: tuple) -> int:
        """Gets a node for the coverage given the coordinates.

        Converts a point to a node if a point already exists at the given coordinates.

        Args:
            node_coordinates (tuple): Coordinates of a node to add to the coverage.

        Return:
            The node ID
        """
        if node_coordinates not in self._point_to_id:
            self._point_to_id[node_coordinates] = [self._cur_point_id, True]
            self._node_id_to_idx[self._cur_point_id] = len(self._geometry_ids)
            self._geometry_ids.append(self._cur_point_id)
            self._cur_point_id += 1
            self._geometry_types.append('Node')
            self._add_point_info(node_coordinates)
        elif not self._point_to_id[node_coordinates][1]:
            # Not a node...this is a point that we need to convert to a node.
            index = self._point_index_from_id(self._point_to_id[node_coordinates][0])
            self._point_to_id[node_coordinates][1] = True
            self._geometry_types[index] = 'Node'
        return self._point_to_id[node_coordinates][0]

    def add_point(self, point_coordinates: tuple):
        """Adds a point to the coverage given the coordinates.

        Args:
            point_coordinates (tuple): Coordinates of a point to add to the coverage.

        Return:
            The point ID
        """
        if point_coordinates not in self._point_to_id:
            self._point_to_id[point_coordinates] = [self._cur_point_id, False]
            self._node_id_to_idx[self._cur_point_id] = len(self._geometry_ids)
            self._geometry_ids.append(self._cur_point_id)
            self._cur_point_id += 1
            self._geometry_types.append('Point')
            self._add_point_info(point_coordinates)
        return self._point_to_id[point_coordinates][0]

    def _add_point_info(self, coordinates):
        self._geometry_list.append(Point(coordinates[0], coordinates[1], coordinates[2]))
        self._polygon_arc_ids.append([])
        self._polygon_arc_directions.append([])
        self._interior_arc_ids.append([])
        self._interior_arc_directions.append([])
        self._start_node.append(-1)
        self._end_node.append(-1)

    def add_vertex(self, vertex_coordinates: tuple):
        """Adds a vertex to the coverage given the coordinates.

        Args:
            vertex_coordinates (tuple): Coordinates of a node to add to the coverage.
        """
        self._geometry_ids.append(-1)
        self._geometry_types.append('Vertex')
        self._add_point_info(vertex_coordinates)

    def get_node_geometry(self, node_id: int) -> Point | None:
        """Returns the geometry of the given node ID.

        Args:
            node_id (int): The node ID of the geometry to retrieve.

        Return:
            (Point): The geometry.
        """
        node_idx = self._node_id_to_idx.get(node_id)
        if node_idx is not None:
            return self._geometry_list[node_idx]
        return None

    def get_arc_geometry(self, arc_id: int) -> LineString | None:
        """Returns the geometry of the given arc ID.

        Args:
            arc_id (int): The arc ID of the geometry to retrieve.

        Return:
            (LineString): The geometry.
        """
        arc_idx = self._arc_id_to_idx.get(arc_id)
        if arc_idx is not None:
            return self._geometry_list[arc_idx]
        return None

    def _point_index_from_id(self, point_id: int) -> int:
        """Returns the point index from the given point ID.

        Args:
            point_id (int): The ID of the point.

        Return:
            (int): The point index in the list.
        """
        return self._node_id_to_idx.get(point_id)

    def build_coverage(self) -> GeoDataFrame:
        """Adds arcs to the coverage.

        Return:
            (GeoDataFrame): The coverage.
        """
        json_strings = [''] * len(self._geometry_types)
        gdf = GeoDataFrame({'id': self._geometry_ids,
                            'geometry_types': self._geometry_types,
                            'geometry': self._geometry_list,
                            'polygon_arc_ids': self._polygon_arc_ids,
                            'polygon_arc_directions': self._polygon_arc_directions,
                            'interior_arc_ids': self._interior_arc_ids,
                            'interior_arc_directions': self._interior_arc_directions,
                            'start_node': self._start_node, 'end_node': self._end_node,
                            'attributes': json_strings}, crs=self._cov_wkt if self._cov_wkt is not None else None)
        if self._cov_name is not None:
            gdf.attrs['name'] = self._cov_name
        gdf.attrs['uuid'] = self._cov_uuid
        return gdf
