"""IntegerArgument class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from .argument import Argument, IoDirection


class IntegerArgument(Argument):
    """Integer tool argument."""
    def __init__(
        self,
        name: str,
        description: str = '',
        io_direction: Optional[IoDirection] = IoDirection.INPUT,
        optional: Optional[bool] = False,
        value: Optional[int] = None,
        hide: Optional[bool] = False,
        min_value: Optional[int] = None,
        max_value: Optional[int] = None
    ):
        """Construct a dataset argument.

        Args:
            name (Optional[str]): Python friendly argument name.
            description (Optional[str]): User friendly description of the argument.
            io_direction (Optional[IoDirection]): IO Direction of the argument (input or output).
            optional (Optional[bool]): Is the argument optional?
            value (Optional[int]): Default value.
            hide (Optional[bool]): Should the argument be hidden (True) or visible?
            min_value (Optional[int]): The minimum value of the argument.
            max_value (Optional[int]): The maximum value of the argument.
        """
        super().__init__(name, description, io_direction, optional, value, hide)
        self._min_value = min_value
        self._max_value = max_value
        if self.io_direction == IoDirection.OUTPUT:
            self.optional = True  # output doesn't need to be specified

    def _get_type(self) -> str:
        """Get a string representing the argument type (int).

        Returns:
            (str): The argument type.
        """
        return 'integer'

    def _set_value(self, value: Optional[int]) -> None:
        """Set the argument value.

        Args:
            value (Optional[int]): The new argument value.
        """
        self._value = None if value is None else int(value)

    def get_interface_info(self) -> Optional[dict[str, object]]:
        """Get interface info for argument to be used in settings dialog.

        Returns:
            Dictionary of interface info.
        """
        if self.io_direction == IoDirection.OUTPUT:
            return None
        interface_info = {'type': 'Integer'}
        interface_info = interface_info | super().get_interface_info()
        if interface_info['value'] is None:
            interface_info['value'] = 0
        return interface_info

    def _is_required_satisfied(self) -> bool | None:
        """Is the "required" value satisfied?

        For an integer argument, the GUI interface always has a value.

        Returns:
              True/False if the required value is satisfied. None if not required.
        """
        return None

    def validate(self) -> str:
        """Validate the argument.

        Returns:
            (str): An error string if invalid or None.
        """
        if self.value is not None:
            integer_value = int(self.value)
            if self._min_value is not None and integer_value < self._min_value:
                return f'Value must be greater than or equal to {self._min_value}.'
            if self._max_value is not None and integer_value > self._max_value:
                return f'Value must be less than or equal to {self._max_value}.'
        return None

    def to_dict(self) -> dict:
        """Convert an argument to a dictionary.

        Returns:
            (dict): The object values as a dictionary.
        """
        values = {'__class__': 'IntegerArgument'}
        super_values = super(IntegerArgument, self).to_dict()
        values.update(super_values)
        self._add_key_value('min_value', self._min_value, values)
        self._add_key_value('max_value', self._max_value, values)
        return values
