"""Worker thread for the TUFLOWFV .fvc file importer."""
# 1. Standard python modules
import logging

# 2. Third party modules
from PySide2.QtCore import Signal

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg, ProcessFeedbackThread

# 4. Local modules
from xms.tuflowfv.file_io.geom_reader import TwoDMReader
from xms.tuflowfv.file_io.structure_manager import StructureManager


class GeometryImportWorkerThread(ProcessFeedbackThread):
    """Worker thread for importing a TUFLOWFV control input file."""
    processing_finished = Signal()

    def __init__(self, parent, filename, wkt):
        """Construct the worker.

        Args:
            parent (QWidget): The Qt parent dialog
            filename (str): Path to the .2dm file to import
            wkt (str): Projection to assign the imported data
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self.reader = TwoDMReader(filename=filename, manager=StructureManager(), wkt=wkt)

    def _do_work(self):
        """Thread runner that imports the .fvc file."""
        try:
            self.reader.read()
        except Exception as e:
            logging.getLogger('xms.tuflowfv').error(f'Error(s) encountered while reading TUFLOWFV geometry file: {str(e)}.')


def add_imported_data(query, reader):
    """Add the imported .2dm data to SMS.

    Args:
        query (Query): XMS interprocess communicator
        reader (GeomReader): The .2dm reader
    """
    if reader.do_ugrid:
        query.add_ugrid(reader.do_ugrid)
    if reader.material_coverage:
        query.add_coverage(reader.material_coverage, coverage_type='Materials', model_name='TUFLOWFV',
                           components=[reader.material_comp])
    if reader.bc_coverage:
        query.add_coverage(reader.bc_coverage, coverage_type='Boundary Conditions', model_name='TUFLOWFV',
                           components=[reader.bc_comp])


def import_geometry_with_feedback():
    """Driver for the .2dm import script with a feedback dialog."""
    xms_dlg.ensure_qapplication_exists()

    # Parse parent HWNDs and icon path from commandline arguments.
    if XmEnv.xms_environ_running_tests() != 'TRUE':
        parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
        qt_parent = xms_dlg.get_parent_window_container(parent_hwnd)
        # Create the timer that keeps our Python dialog in the foreground of XMS.
        _ = win_gui.create_and_connect_raise_timer(main_hwnd, qt_parent)  # Keep the timer in scope
    else:
        qt_parent = None
        main_hwnd = None

    query = Query()
    worker = GeometryImportWorkerThread(parent=qt_parent, filename=query.read_file,
                                        wkt=query.display_projection.well_known_text)
    display_text = {
        'title': 'Reading TUFLOWFV .2dm Geometry File',
        'working_prompt': 'Reading TUFLOW .2dm geometry file. Please wait...',
        'error_prompt': 'Error(s) encountered reading .2dm file. Review log output for more details.',
        'warning_prompt': 'Warning(s) encountered reading .2dm file. Review log output for more details.',
        'success_prompt': 'Successfully read TUFLOWFV .2dm geometry file.',
        'note': '',
        'auto_load': 'Auto load data into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms', worker, qt_parent)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec() and not LogEchoQSignalStream.logged_error:  # Only send if no error.
        add_imported_data(query, worker.reader)
        query.send()
    if qt_parent is not None:
        win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
