"""Class for switching the units labels in GUI labels when display projection is in imperial units."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


METRIC_TO_IMPERIAL = {
    '(m)': '(ft)',
    '(m/s)': '(ft/s)',
    '(m/day)': '(ft/day)',
    # This one I don't require the parenthesis around because they get used in labels that sometimes have more text
    # in the parenthesis, e.g. '(coefficient or m²/s)'. Think it is pretty safe to replace all text because of the
    # special superscript character. Since this is a special case, I'll leave the parenthesis requirement for other
    # units, even if it is unlikely to erroneously match for most of them.
    'm²/s': 'ft²/s',
    '(N/m²)': '(lbf/ft²)',
    # '(W/m²)': '(W/m²)',  # Not sure what the Imperial units for this would be, but NOAA uses W/m².
    '(kg/m³)': '(lb/ft³)',
    '(hPa)': '(psi)',
    '(Celsius°)': '(Fahrenheit°)',
}


class UnitsLabelSwitcher:
    """Class for switching the units labels in GUI labels when display projection is in imperial units."""

    def __init__(self, dlg):
        """Construct the widget.

        Args:
            dlg (QDialog): The dialog to switch labels for
        """
        self._dlg = dlg

    """
    Widget label switchers. Don't call these if current display projection vertical units are Metric.
    """
    def switch_to_imperial(self, widget_list):
        """Switch units text in labels for all widgets in a dialog from Metric to Imperial.

        Args:
            widget_list (list): Names of the Qt widget classes to switch label text for
        """
        for widget_type in widget_list:
            children = self._dlg.findChildren(widget_type)
            for child in children:
                self.switch_widget_to_imperial(child)

    @staticmethod
    def switch_widget_to_imperial(widget):
        """Switch a single widget's units label text from Metric to Imperial.

        Args:
            widget (QWidget): The widget to replace units label text
        """
        text = widget.text()
        for si_unit, imperial_unit in METRIC_TO_IMPERIAL.items():
            if si_unit in text:
                widget.setText(text.replace(si_unit, imperial_unit))
                return
