"""DisplaySettings Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.calculator.setting_group import SettingGroup
from xms.FhwaVariable.core_data.variables.variable import Variable


# @singleton
class DisplaySettings(SettingGroup):
    """Provides definitions for the themes."""

    def __init__(self, name: str = 'HydraulicToolbox', version: str = '0.1', agency: str = 'FHWA',
                 developed_by: str = 'Aquaveo'):
        """Initializes the constant values."""
        super().__init__(name, version, agency, developed_by)

        self.name = 'DisplaySettings'
        self.type = 'DisplaySettings'

        self.theme_names = ['dark', 'light', 'FHWA Classic', 'custom']

        self.input['Select theme'] = Variable(
            'Select visual theme', 'list', 0, self.theme_names,
            note='This setting follows your system theme setting. Change your system theme to change this setting.')
        self.input['Select theme'].read_only = True

# Theme Colors
        self.input['Theme colors'] = {}
        # Main Dialog Colors
        self.input['Theme colors']['Main dialog title text color'] = Variable(
            'Main dialog title text color', 'color', 'rgb', (235, 235, 235), complexity=2)
        self.input['Theme colors']['Main dialog title background color'] = Variable(
            'Main dialog title background color', 'color', 'rgb', (40, 42, 46), complexity=2)
        self.input['Theme colors']['Main dialog text color'] = Variable(
            'Main dialog text color', 'color', 'rgb', (255, 255, 255), complexity=2)
        self.input['Theme colors']['Main dialog background color'] = Variable(
            'Main dialog background color', 'color', 'rgb', (58, 61, 65), complexity=2)

        self.input['Theme colors']['Document background color'] = Variable(
            'Document background color', 'color', 'rgb', (50, 50, 50), complexity=2)
        self.input['Theme colors']['Main dialog item background color'] = Variable(
            'Main dialog item background color', 'color', 'rgb', (50, 50, 50), complexity=2)
        self.input['Theme colors']['Main dialog item text color'] = Variable(
            'Main dialog item text color', 'color', 'rgb', (255, 255, 255), complexity=2)

        # Dialog colors
        self.input['Theme colors']['Dialog title text color'] = Variable(
            'Dialog title text color', 'color', 'rgb', (235, 235, 235), complexity=2)
        self.input['Theme colors']['Dialog title background color'] = Variable(
            'Dialog title background color', 'color', 'rgb', (40, 42, 46), complexity=2)
        self.input['Theme colors']['Dialog text color'] = Variable(
            'Dialog text color', 'color', 'rgb', (255, 255, 255), complexity=2)
        self.input['Theme colors']['Dialog background color'] = Variable(
            'Dialog background color', 'color', 'rgb', (58, 61, 65), complexity=2)

        # Plot colors
        self.input['Theme colors']['Plot text color'] = Variable(
            'Plot text color', 'color', 'rgb', (255, 255, 255), complexity=2)
        self.input['Theme colors']['Plot tick color'] = Variable(
            'Plot tick color', 'color', 'rgb', (160, 160, 160), complexity=2)
        self.input['Theme colors']['Plot background color'] = Variable(
            'Plot background color', 'color', 'rgb', (10, 10, 10), complexity=2)
        self.input['Theme colors']['Plot structure color'] = Variable(
            'Plot structure color', 'color', 'rgb', (77, 77, 77), complexity=2)
        self.input['Theme colors']['Plot structure fill color'] = Variable(
            'Plot structure fill color', 'color', 'rgb', (191, 191, 191), complexity=2)
        self.input['Theme colors']['Plot embankment color'] = Variable(
            'Plot embankment color', 'color', 'rgb', (10, 10, 10), complexity=2)
        self.input['Theme colors']['Plot embankment fill color'] = Variable(
            'Plot embankment fill color', 'color', 'rgb', (10, 10, 10), complexity=2)
        self.input['Theme colors']['Plot soil color'] = Variable(
            'Plot soil color', 'color', 'rgb', (102, 51, 0), complexity=2)
        self.input['Theme colors']['Plot soil fill color'] = Variable(
            'Plot soil fill color', 'color', 'rgb', (215, 211, 199), complexity=2)
        self.input['Theme colors']['Plot WSP color'] = Variable(
            'Plot WSP color', 'color', 'rgb', (70, 118, 202), complexity=2)
        self.input['Theme colors']['Plot WSP fill color'] = Variable(
            'Plot WSP fill color', 'color', 'rgb', (0, 0, 255), complexity=2)
        self.input['Theme colors']['Plot normal depth color'] = Variable(
            'Plot normal depth color', 'color', 'rgb', (0, 255, 0), complexity=2)
        self.input['Theme colors']['Plot normal depth fill color'] = Variable(
            'Plot normal depth fill color', 'color', 'rgb', (64, 224, 208), complexity=2)
        self.input['Theme colors']['Plot critical depth color'] = Variable(
            'Plot critical depth color', 'color', 'rgb', (255, 0, 0), complexity=2)
        self.input['Theme colors']['Plot critical depth fill color'] = Variable(
            'Plot critical depth fill color', 'color', 'rgb', (255, 0, 0), complexity=2)
        self.input['Theme colors']['Plot hydraulic jump color'] = Variable(
            'Plot hydraulic jump color', 'color', 'rgb', (107, 16, 191), complexity=2)
        self.input['Theme colors']['Plot hydraulic jump fill color'] = Variable(
            'Plot hydraulic jump fill color', 'color', 'rgb', (107, 16, 191), complexity=2)
        self.input['Theme colors']['Plot roughness color'] = Variable(
            'Plot roughness color', 'color', 'rgb', (255, 215, 0), complexity=2)
        # self.input['Theme colors']['Plot roughness fill color'] = Variable(
        #     'Plot roughness fill color', 'color', 'rgb', (255, 239, 153), complexity=2)
        self.input['Theme colors']['Plot vena contracta color'] = Variable(
            'Plot vena contracta color', 'color', 'rgb', (46, 240, 160), complexity=2)
        self.input['Theme colors']['Plot velocity color'] = Variable(
            'Plot velocity color', 'color', 'rgb', (255, 140, 0), complexity=2)
        self.input['Theme colors']['Plot velocity fill color'] = Variable(
            'Plot velocity fill color', 'color', 'rgb', (255, 200, 120), complexity=2)

        # Scour plot colors
        self.input['Theme colors']['Plot channel bank line color'] = Variable(
            'Plot channel bank line color', 'color', 'rgb', (128, 64, 0), complexity=2)
        self.input['Theme colors']['Plot abutment toe line color'] = Variable(
            'Plot abutment toe line color', 'color', 'rgb', (130, 130, 130), complexity=2)
        self.input['Theme colors']['Plot long-term degradation color'] = Variable(
            'Plot long-term degradation color', 'color', 'rgb', (186, 12, 47), complexity=2)
        self.input['Theme colors']['Plot long-term degradation fill color'] = Variable(
            'Plot long-term degradation fill color', 'color', 'rgb', (227, 158, 172), complexity=2)
        self.input['Theme colors']['Plot contraction scour color'] = Variable(
            'Plot contraction scour color', 'color', 'rgb', (207, 127, 0), complexity=2)
        self.input['Theme colors']['Plot contraction scour fill color'] = Variable(
            'Plot contraction scour fill color', 'color', 'rgb', (236, 204, 153), complexity=2)
        self.input['Theme colors']['Plot pier scour color'] = Variable(
            'Plot pier scour color', 'color', 'rgb', (51, 146, 109), complexity=2)
        self.input['Theme colors']['Plot pier scour fill color'] = Variable(
            'Plot pier scour fill color', 'color', 'rgb', (153, 201, 182), complexity=2)
        self.input['Theme colors']['Plot abutment scour color'] = Variable(
            'Plot abutment scour color', 'color', 'rgb', (153, 201, 182), complexity=2)
        self.input['Theme colors']['Plot abutment scour fill color'] = Variable(
            'Plot abutment scour fill color', 'color', 'rgb', (51, 146, 109), complexity=2)

        # Table colors
        self.input['Theme colors']['Table title text color'] = Variable(
            'Table title text color', 'color', 'rgb', (255, 255, 255), complexity=2)
        self.input['Theme colors']['Table title background color'] = Variable(
            'Table title background color', 'color', 'rgb', (91, 115, 147), complexity=2)
        self.input['Theme colors']['Table header text color'] = Variable(
            'Table header text color', 'color', 'rgb', (255, 255, 255), complexity=2)
        self.input['Theme colors']['Table header background color'] = Variable(
            'Table header background color', 'color', 'rgb', (70, 99, 137), complexity=2)
        self.input['Theme colors']['Table subheader text color'] = Variable(
            'Table subheader text color', 'color', 'rgb', (255, 255, 255), complexity=2)
        self.input['Theme colors']['Table subheader background color'] = Variable(
            'Table subheader background color', 'color', 'rgb', (51, 63, 79), complexity=2)
        self.input['Theme colors']['Table inactive text color'] = Variable(
            'Table inactive text color', 'color', 'rgb', (255, 255, 255), complexity=2)
        self.input['Theme colors']['Table inactive background color'] = Variable(
            'Table inactive background color', 'color', 'rgb', (35, 35, 35), complexity=2)
        self.input['Theme colors']['Table read-only text color'] = Variable(
            'Table read-only text color', 'color', 'rgb', (255, 255, 255), complexity=2)
        self.input['Theme colors']['Table read-only background color'] = Variable(
            'Table read-only background color', 'color', 'rgb', (20, 20, 20), complexity=2)
        self.input['Theme colors']['Table editable text color'] = Variable(
            'Table editable text color', 'color', 'rgb', (255, 255, 255), complexity=2)
        self.input['Theme colors']['Table editable background color'] = Variable(
            'Table editable background color', 'color', 'rgb', (0, 0, 0), complexity=2)

        # Tab colors
        self.input['Theme colors']['Tab selected background color'] = Variable(
            'Tab selected background color', 'color', 'rgb', (15, 20, 23), complexity=2)
        self.input['Theme colors']['Tab selected text color'] = Variable(
            'Tab selected text color', 'color', 'rgb', (0, 0, 0), complexity=2)
        self.input['Theme colors']['Tab background color'] = Variable(
            'Tab background color', 'color', 'rgb', (1, 6, 9), complexity=2)
        self.input['Theme colors']['Tab text color'] = Variable(
            'Tab text color', 'color', 'rgb', (0, 0, 0), complexity=2)

        # general colors
        self.input['Theme colors']['Success background color'] = Variable(
            'Success background color', 'color', 'rgb', (79, 125, 62), complexity=2)
        self.input['Theme colors']['Success text color'] = Variable(
            'Success text color', 'color', 'rgb', (255, 255, 255), complexity=2)
        self.input['Theme colors']['Incomplete background color'] = Variable(
            'Incomplete background color', 'color', 'rgb', (143, 158, 35), complexity=2)
        self.input['Theme colors']['Incomplete text color'] = Variable(
            'Incomplete text color', 'color', 'rgb', (255, 255, 255), complexity=2)
        self.input['Theme colors']['Warning background color'] = Variable(
            'Warning background color', 'color', 'rgb', (148, 93, 14), complexity=2)
        self.input['Theme colors']['Warning text color'] = Variable(
            'Warning text color', 'color', 'rgb', (255, 255, 255), complexity=2)
        self.input['Theme colors']['Failure background color'] = Variable(
            'Failure background color', 'color', 'rgb', (200, 5, 5), complexity=2)
        self.input['Theme colors']['Failure text color'] = Variable(
            'Failure text color', 'color', 'rgb', (255, 255, 255), complexity=2)

# End of Theme Colors

        self.input['Display multiple values'] = Variable('Display tables vertically or horizontally', 'list', 0,
                                                         ['vertically', 'horizontally'])
        self.input['Display flat tables'] = Variable('Select visual theme', 'bool', False)
        self.input['Use 2 input tables when available'] = Variable("Use 2 input tables when available", 'bool', True,
                                                                   note="HY-8 interface style")

        # theme = self.get_theme()
        # for item in theme:
        #     self.input[item] = Variable(item, 'color', theme[item], complexity=2)

        self.average_text_color = 0.0
        self.average_background_color = 0.0

        self.isThemeDark = False
        # xms::XM_COLORREF ESSColors::m_whiteClr = XmsRGB(255, 255, 255);
        # xms::XM_COLORREF ESSColors::m_greyClr = XmsRGB(200, 200, 200);
        # xms::XM_COLORREF ESSColors::m_blackClr = XmsRGB(0, 0, 0);
        # xms::XM_COLORREF ESSColors::m_tanClr = XmsRGB(236, 233, 216);

        # xms::XM_COLORREF ESSColors::m_redLightClr = XmsRGB(255, 192, 192);
        # xms::XM_COLORREF ESSColors::m_redClr = XmsRGB(255, 5, 5);
        # xms::XM_COLORREF ESSColors::m_purpleLightClr = XmsRGB(255, 192, 255);
        # xms::XM_COLORREF ESSColors::m_purpleClr = XmsRGB(255, 0, 255);
        # xms::XM_COLORREF ESSColors::m_blueLightClr = XmsRGB(192, 192, 255);
        # xms::XM_COLORREF ESSColors::m_alternateblueLightClr = XmsRGB(210, 210, 255);
        # xms::XM_COLORREF ESSColors::m_blueClr = XmsRGB(5, 5, 255);
        # xms::XM_COLORREF ESSColors::m_greenLightClr = XmsRGB(192, 255, 192);
        # xms::XM_COLORREF ESSColors::m_greenClr = XmsRGB(5, 255, 5);
        # xms::XM_COLORREF ESSColors::m_yellowLightClr = XmsRGB(255, 255, 192);
        # xms::XM_COLORREF ESSColors::m_yellowClr = XmsRGB(251, 231, 5);
        # xms::XM_COLORREF ESSColors::m_orangeLightClr = XmsRGB(252, 213, 180);
        # xms::XM_COLORREF ESSColors::m_orangeClr = XmsRGB(251, 132, 5);

        # // Neutral theme from John Hut
        # 12 / 21 / 2018

        # xms::XM_COLORREF ESSColors::m_tableHeaderClr = XmsRGB(51, 63, 79);
        # xms::XM_COLORREF ESSColors::m_greyBackgroundClr = XmsRGB(240, 240, 240);
        # xms::XM_COLORREF ESSColors::m_headerClr = XmsRGB(172, 185, 202);
        # xms::XM_COLORREF ESSColors::m_subheaderClr = XmsRGB(214, 220, 228);
        # xms::XM_COLORREF ESSColors::m_inactiveTextClr = XmsRGB(240, 240, 240);
        # xms::XM_COLORREF ESSColors::m_readOnlyClr = XmsRGB(240, 240, 240);
        # xms::XM_COLORREF ESSColors::m_editableClr = XmsRGB(255, 255, 255);
        # xms::XM_COLORREF ESSColors::m_buttonClr = XmsRGB(0, 0, 0);
        # xms::XM_COLORREF ESSColors::m_tableBody1Clr = XmsRGB(255, 255, 255);
        # xms::XM_COLORREF ESSColors::m_tableBody2Clr = XmsRGB(240, 240, 240);
        # xms::XM_COLORREF ESSColors::m_optionalClr = XmsRGB(212, 212, 222);
        # xms::XM_COLORREF ESSColors::m_buttonTextClr = XmsRGB(93, 115, 145);

        # xms::XM_COLORREF ESSColors::m_warningClr = XmsRGB(200, 5, 5);
        # xms::XM_COLORREF ESSColors::m_warningLightClr = XmsRGB(254, 210, 210);
        # xms::XM_COLORREF ESSColors::m_warningSecondClr = XmsRGB(248, 228, 152);
        # xms::XM_COLORREF ESSColors::m_warningSecondLightClr = XmsRGB(254, 254, 214);
        # xms::XM_COLORREF ESSColors::m_approvalClr = XmsRGB(169, 220, 146);
        # xms::XM_COLORREF ESSColors::m_approvalLightClr = XmsRGB(198, 224, 180);

        # xms::XM_COLORREF ESSColors::m_purpleLightClr2 = XmsRGB(210, 210, 254);
        # xms::XM_COLORREF ESSColors::m_blueLightClr2 = XmsRGB(220, 228, 255);
        # xms::XM_COLORREF ESSColors::m_orangeLightClr2 = XmsRGB(254, 234, 222);

    def get_input_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        input_vars = {}

        input_vars = copy.deepcopy(self.input)
        if self.input['Select theme'].get_val() != 'custom':
            input_vars.pop('Theme colors')

        return input_vars

    def set_theme_colors(self):
        """Sets the visual theme."""
        if self.input['Select theme'].get_val() == 'neutral theme':
            self.set_light_theme_colors()
            self.set_nuetral_theme_colors()

        elif self.input['Select theme'].get_val() == 'dark':
            self.set_dark_theme_colors()

        elif self.input['Select theme'].get_val() == 'light':
            self.set_light_theme_colors()

        elif self.input['Select theme'].get_val() == 'FHWA Classic':
            self.set_light_theme_colors()
            self.set_fhwa_classic_theme_colors()

        elif self.input['Select theme'].get_val() == 'custom':
            pass

    def set_nuetral_theme_colors(self):
        """Sets the visual theme."""
        # Table colors
        self.input['Theme colors']['Table title text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Table title background color'].set_val((51, 63, 79))
        self.input['Theme colors']['Table header text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table header background color'].set_val((172, 185, 202))
        self.input['Theme colors']['Table subheader text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table subheader background color'].set_val((214, 220, 228))
        self.input['Theme colors']['Table inactive text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table inactive background color'].set_val((240, 240, 240))
        self.input['Theme colors']['Table read-only text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table read-only background color'].set_val((240, 240, 240))
        self.input['Theme colors']['Table editable text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table editable background color'].set_val((255, 255, 255))

    def set_dark_theme_colors(self):
        """Sets the visual theme."""
        # Main Dialog Colors
        self.input['Theme colors']['Main dialog title text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Main dialog title background color'].set_val((40, 40, 40))
        self.input['Theme colors']['Main dialog text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Main dialog background color'].set_val((65, 65, 65))

        self.input['Theme colors']['Document background color'].set_val((25, 25, 25))
        self.input['Theme colors']['Main dialog item background color'].set_val((30, 30, 30))
        self.input['Theme colors']['Main dialog item text color'].set_val((255, 255, 255))

        # Dialog colors
        self.input['Theme colors']['Dialog title text color'].set_val((235, 235, 235))
        self.input['Theme colors']['Dialog title background color'].set_val((40, 42, 46))
        self.input['Theme colors']['Dialog text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Dialog background color'].set_val((58, 61, 65))

        # Plot colors?
        self.input['Theme colors']['Plot text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Plot tick color'].set_val((160, 160, 160))
        self.input['Theme colors']['Plot background color'].set_val((10, 10, 10))
        self.input['Theme colors']['Plot structure color'].set_val((77, 77, 77))
        self.input['Theme colors']['Plot structure fill color'].set_val((191, 191, 191))
        self.input['Theme colors']['Plot embankment color'].set_val((10, 10, 10))
        self.input['Theme colors']['Plot embankment fill color'].set_val((10, 10, 10))
        self.input['Theme colors']['Plot soil color'].set_val((102, 51, 0))  # tan
        self.input['Theme colors']['Plot soil fill color'].set_val((215, 211, 199))  # light beige
        # self.input['Theme colors']['Plot soil fill color'].set_val((153, 102, 51)  # brown
        self.input['Theme colors']['Plot WSP color'].set_val((70, 118, 202))  # cornflower
        self.input['Theme colors']['Plot WSP fill color'].set_val((0, 0, 255))  # blue
        self.input['Theme colors']['Plot critical depth color'].set_val((255, 0, 0))  # red
        self.input['Theme colors']['Plot normal depth color'].set_val((0, 255, 0))  # green
        self.input['Theme colors']['Plot normal depth fill color'].set_val((64, 224, 208))  # aqua/turquoise
        self.input['Theme colors']['Plot hydraulic jump color'].set_val((107, 16, 191))  # blueviolet
        self.input['Theme colors']['Plot hydraulic jump fill color'].set_val((107, 16, 191))
        self.input['Theme colors']['Plot roughness color'].set_val((255, 215, 0))
        # self.input['Theme colors']['Plot roughness fill color'] = Variable(
        #     'Plot roughness fill color', 'color', 'rgb', (255, 239, 153), complexity=2)
        self.input['Theme colors']['Plot vena contracta color'].set_val((46, 240, 160))
        # Dark Orange color (another contrasting color): (255, 140, 0) and fill: (255, 200, 120)
        self.input['Theme colors']['Plot velocity color'].set_val((255, 140, 0))
        self.input['Theme colors']['Plot velocity fill color'].set_val((255, 200, 120))

        # Scour plot colors
        self.input['Theme colors']['Plot channel bank line color'].set_val((128, 64, 0))  # dark, dark red
        self.input['Theme colors']['Plot abutment toe line color'].set_val((130, 130, 130))  # grey
        self.input['Theme colors']['Plot long-term degradation color'].set_val((186, 12, 47))  # Dark red
        self.input['Theme colors']['Plot long-term degradation fill color'].set_val((227, 158, 172))  # light red
        self.input['Theme colors']['Plot contraction scour color'].set_val((207, 127, 0))  # Dark orange
        self.input['Theme colors']['Plot contraction scour fill color'].set_val((236, 204, 153))  # light orange
        self.input['Theme colors']['Plot pier scour color'].set_val((51, 146, 109))  # Dark green
        self.input['Theme colors']['Plot pier scour fill color'].set_val((153, 201, 182))  # light green
        self.input['Theme colors']['Plot abutment scour color'].set_val((153, 201, 182))  # light green
        self.input['Theme colors']['Plot abutment scour fill color'].set_val((51, 146, 109))  # Dark green

        # Table colors
        self.input['Theme colors']['Table title text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Table title background color'].set_val((91, 115, 147))
        self.input['Theme colors']['Table header text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Table header background color'].set_val((70, 99, 137))
        self.input['Theme colors']['Table subheader text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Table subheader background color'].set_val((51, 63, 79))
        self.input['Theme colors']['Table inactive text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Table inactive background color'].set_val((35, 35, 35))
        self.input['Theme colors']['Table read-only text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Table read-only background color'].set_val((20, 20, 20))
        self.input['Theme colors']['Table editable text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Table editable background color'].set_val((60, 60, 60))

        # Tab colors
        self.input['Theme colors']['Tab selected background color'].set_val((38, 41, 45))
        self.input['Theme colors']['Tab selected text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Tab background color'].set_val((18, 21, 25))
        self.input['Theme colors']['Tab text color'].set_val((255, 255, 255))

        # general colors
        self.input['Theme colors']['Success background color'].set_val((79, 125, 62))
        self.input['Theme colors']['Success text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Incomplete background color'].set_val((143, 158, 35))
        self.input['Theme colors']['Incomplete text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Warning background color'].set_val((148, 93, 14))
        self.input['Theme colors']['Warning text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Failure background color'].set_val((200, 5, 5))
        self.input['Theme colors']['Failure text color'].set_val((255, 255, 255))

    def set_light_theme_colors(self):
        """Sets the visual theme."""
        # Main Dialog colors
        self.input['Theme colors']['Main dialog title text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Main dialog title background color'].set_val((240, 240, 240))
        self.input['Theme colors']['Main dialog text color'].set_val((10, 10, 10))
        self.input['Theme colors']['Main dialog background color'].set_val((210, 220, 220))

        self.input['Theme colors']['Document background color'].set_val((150, 150, 150))
        self.input['Theme colors']['Main dialog item background color'].set_val((180, 180, 180))
        self.input['Theme colors']['Main dialog item text color'].set_val((25, 25, 25))

        # Dialog colors
        self.input['Theme colors']['Dialog title text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Dialog title background color'].set_val((255, 255, 255))
        self.input['Theme colors']['Dialog text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Dialog background color'].set_val((240, 240, 240))

        # Plot colors?
        self.input['Theme colors']['Plot text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Plot tick color'].set_val((160, 160, 160))
        self.input['Theme colors']['Plot background color'].set_val((255, 255, 255))
        self.input['Theme colors']['Plot structure color'].set_val((77, 77, 77))
        self.input['Theme colors']['Plot structure fill color'].set_val((191, 191, 191))
        self.input['Theme colors']['Plot embankment color'].set_val((10, 10, 10))
        self.input['Theme colors']['Plot embankment fill color'].set_val((10, 10, 10))
        self.input['Theme colors']['Plot soil color'].set_val((102, 51, 0))  # tan
        self.input['Theme colors']['Plot soil fill color'].set_val((215, 211, 199))  # light beige
        # theme['Plot soil fill color'].set_val((153, 102, 51)  # brown
        self.input['Theme colors']['Plot WSP color'].set_val((70, 118, 202))  # cornflower
        self.input['Theme colors']['Plot WSP fill color'].set_val((0, 0, 255))  # blue
        self.input['Theme colors']['Plot critical depth color'].set_val((255, 0, 0))  # red
        self.input['Theme colors']['Plot normal depth color'].set_val((0, 255, 0))  # green
        self.input['Theme colors']['Plot normal depth fill color'].set_val((64, 224, 208))  # aqua/turquoise
        self.input['Theme colors']['Plot hydraulic jump color'].set_val((107, 16, 191))  # blueviolet
        self.input['Theme colors']['Plot hydraulic jump fill color'].set_val((107, 16, 191))
        self.input['Theme colors']['Plot roughness color'].set_val((255, 215, 0))
        # self.input['Theme colors']['Plot roughness fill color'].set_val((255, 239, 153))
        self.input['Theme colors']['Plot vena contracta color'].set_val((46, 240, 160))
        self.input['Theme colors']['Plot velocity color'].set_val((255, 140, 0))
        self.input['Theme colors']['Plot velocity fill color'].set_val((255, 200, 120))

        # Scour plot colors
        self.input['Theme colors']['Plot channel bank line color'].set_val((128, 64, 0))  # dark, dark red
        self.input['Theme colors']['Plot abutment toe line color'].set_val((130, 130, 130))  # grey
        # theme['Plot long-term degradation color'].set_val((255, 0.0, 0.0)  # Red
        # theme['Plot long-term degradation fill color'].set_val((255, 153, 153)  # light red
        # theme['Plot contraction scour color'].set_val((0.0, 127, 0.0)  # Green
        # theme['Plot contraction scour fill color'].set_val((127, 204, 127)  # light green
        # theme['Plot pier scour color'].set_val((164, 19, 96)  # red-purple
        # theme['Plot pier scour fill color'].set_val((242, 28, 142)  # light red-purple
        # theme['Plot abutment scour color'].set_val((153, 51, 204)  # Purple
        # theme['Plot abutment scour fill color'].set_val((163, 73, 164)  # light purple
        self.input['Theme colors']['Plot long-term degradation color'].set_val((186, 12, 47))  # Dark red
        self.input['Theme colors']['Plot long-term degradation fill color'].set_val((227, 158, 172))  # light red
        self.input['Theme colors']['Plot contraction scour color'].set_val((207, 127, 0))  # Dark orange
        self.input['Theme colors']['Plot contraction scour fill color'].set_val((236, 204, 153))  # light orange
        self.input['Theme colors']['Plot pier scour color'].set_val((51, 146, 109))  # Dark green
        self.input['Theme colors']['Plot pier scour fill color'].set_val((153, 201, 182))  # light green
        self.input['Theme colors']['Plot abutment scour color'].set_val((153, 201, 182))  # light green
        self.input['Theme colors']['Plot abutment scour fill color'].set_val((51, 146, 109))  # Dark green

        # Table colors
        self.input['Theme colors']['Table title text color'].set_val((255, 255, 255))
        self.input['Theme colors']['Table title background color'].set_val((51, 63, 79))
        self.input['Theme colors']['Table header text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table header background color'].set_val((172, 185, 202))
        self.input['Theme colors']['Table subheader text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table subheader background color'].set_val((214, 220, 228))
        self.input['Theme colors']['Table inactive text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table inactive background color'].set_val((240, 240, 240))
        self.input['Theme colors']['Table read-only text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table read-only background color'].set_val((240, 240, 240))
        self.input['Theme colors']['Table editable text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table editable background color'].set_val((255, 255, 255))

        # Tab colors
        self.input['Theme colors']['Tab selected background color'].set_val((249, 249, 249))
        self.input['Theme colors']['Tab selected text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Tab background color'].set_val((243, 243, 243))
        self.input['Theme colors']['Tab text color'].set_val((0, 0, 0))

        # general colors
        self.input['Theme colors']['Success background color'].set_val((169, 220, 146))
        self.input['Theme colors']['Success text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Incomplete background color'].set_val((218, 229, 85))
        self.input['Theme colors']['Incomplete text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Warning background color'].set_val((248, 228, 152))
        self.input['Theme colors']['Warning text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Failure background color'].set_val((200, 5, 5))
        self.input['Theme colors']['Failure text color'].set_val((255, 255, 255))

    def set_fhwa_classic_theme_colors(self):
        """Sets the visual theme."""
        # Main Dialog colors
        self.input['Theme colors']['Main dialog title text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Main dialog title background color'].set_val((240, 243, 249))
        self.input['Theme colors']['Main dialog text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Main dialog background color'].set_val((144, 195, 236))

        # Dialog colors
        self.input['Theme colors']['Dialog title text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Dialog title background color'].set_val((255, 255, 255))
        self.input['Theme colors']['Dialog text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Dialog background color'].set_val((240, 240, 240))

        # Plot colors?
        self.input['Theme colors']['Plot text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Plot tick color'].set_val((160, 160, 160))
        self.input['Theme colors']['Plot background color'].set_val((255, 255, 255))
        self.input['Theme colors']['Plot structure color'].set_val((77, 77, 77))
        self.input['Theme colors']['Plot structure fill color'].set_val((191, 191, 191))
        self.input['Theme colors']['Plot embankment color'].set_val((10, 10, 10))
        self.input['Theme colors']['Plot embankment fill color'].set_val((10, 10, 10))
        self.input['Theme colors']['Plot soil color'].set_val((102, 51, 0))  # brown
        self.input['Theme colors']['Plot soil fill color'].set_val((153, 102, 51))  # tan
        self.input['Theme colors']['Plot WSP color'].set_val((0, 0, 255))  # blue
        self.input['Theme colors']['Plot WSP fill color'].set_val((70, 118, 202))  # cornflower
        self.input['Theme colors']['Plot critical depth color'].set_val((255, 0, 0))  # red
        self.input['Theme colors']['Plot normal depth color'].set_val((0, 255, 0))  # green
        self.input['Theme colors']['Plot hydraulic jump color'].set_val((107, 16, 191))  # blueviolet
        self.input['Theme colors']['Plot hydraulic jump fill color'].set_val((107, 16, 191))
        self.input['Theme colors']['Plot roughness color'].set_val((255, 215, 0))
        # self.input['Theme colors']['Plot roughness fill color'].set_val((255, 239, 153))
        self.input['Theme colors']['Plot vena contracta color'].set_val((46, 240, 160))
        self.input['Theme colors']['Plot velocity color'].set_val((255, 140, 0))
        self.input['Theme colors']['Plot velocity fill color'].set_val((255, 200, 120))

        # Scour plot colors
        self.input['Theme colors']['Plot channel bank line color'].set_val((128, 64, 0))  # dark, dark red
        self.input['Theme colors']['Plot abutment toe line color'].set_val((130, 130, 130))  # grey
        self.input['Theme colors']['Plot long-term degradation color'].set_val((255, 0.0, 0.0))  # Red
        self.input['Theme colors']['Plot long-term degradation fill color'].set_val((255, 153, 153))  # light red
        self.input['Theme colors']['Plot contraction scour color'].set_val((0.0, 127, 0.0))  # Green
        self.input['Theme colors']['Plot contraction scour fill color'].set_val((127, 204, 127))  # light green
        self.input['Theme colors']['Plot pier scour color'].set_val((164, 19, 96))  # red-purple
        self.input['Theme colors']['Plot pier scour fill color'].set_val((242, 28, 142))  # light red-purple
        self.input['Theme colors']['Plot abutment scour color'].set_val((153, 51, 204))  # Purple
        self.input['Theme colors']['Plot abutment scour fill color'].set_val((163, 73, 164))  # light purple

        # Table colors
        self.input['Theme colors']['Table title text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table title background color'].set_val((240, 240, 240))
        self.input['Theme colors']['Table header text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table header background color'].set_val((255, 255, 192))
        self.input['Theme colors']['Table subheader text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table subheader background color'].set_val((252, 213, 180))
        self.input['Theme colors']['Table inactive text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table inactive background color'].set_val((200, 200, 200))
        self.input['Theme colors']['Table read-only text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table read-only background color'].set_val((236, 233, 216))
        self.input['Theme colors']['Table editable text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Table editable background color'].set_val((255, 255, 255))

        # general colors
        self.input['Theme colors']['Success background color'].set_val((169, 220, 146))
        self.input['Theme colors']['Success text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Incomplete background color'].set_val((218, 229, 85))
        self.input['Theme colors']['Incomplete text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Warning background color'].set_val((248, 228, 152))
        self.input['Theme colors']['Warning text color'].set_val((0, 0, 0))
        self.input['Theme colors']['Failure background color'].set_val((200, 5, 5))
        self.input['Theme colors']['Failure text color'].set_val((255, 255, 255))

    def get_theme(self):
        """Returns the visual theme, based on the settings."""
        theme = {}
        self.set_theme_colors()

        for color_var in self.input['Theme colors']:
            theme[color_var] = self.input['Theme colors'][color_var].get_val()

        # # Used to determine whether the theme is dark or light
        number_of_colors = 6
        average_text_color = theme['Table title text color'][0] + \
            theme['Table title text color'][1] + \
            theme['Table title text color'][2]
        average_text_color += theme['Table header text color'][0] + \
            theme['Table header text color'][1] + \
            theme['Table header text color'][2]
        average_text_color += theme['Table subheader text color'][0] + \
            theme['Table subheader text color'][1] + \
            theme['Table subheader text color'][2]
        average_text_color += theme['Table inactive text color'][0] + \
            theme['Table inactive text color'][1] + \
            theme['Table inactive text color'][2]
        average_text_color += theme['Table read-only text color'][0] + \
            theme['Table read-only text color'][1] + \
            theme['Table read-only text color'][2]
        average_text_color += theme['Table editable text color'][0] + \
            theme['Table editable text color'][1] + \
            theme['Table editable text color'][2]
        average_text_color /= (3 * number_of_colors)

        average_background_color = theme['Table title background color'][0] + \
            theme['Table title background color'][1] + \
            theme['Table title background color'][2]
        average_background_color += theme['Table header background color'][0] + \
            theme['Table header background color'][1] + \
            theme['Table header background color'][2]
        average_background_color += theme['Table subheader background color'][0] + \
            theme['Table subheader background color'][1] + \
            theme['Table subheader background color'][2]
        average_background_color += theme['Table inactive background color'][0] + \
            theme['Table inactive background color'][1] + \
            theme['Table inactive text color'][2]
        average_background_color += theme['Table read-only background color'][0] + \
            theme['Table read-only background color'][1] + \
            theme['Table read-only background color'][2]
        average_background_color += theme['Table editable background color'][0] + \
            theme['Table editable background color'][1] + \
            theme['Table editable background color'][2]
        average_background_color /= (3 * number_of_colors)

        # # Determine if the selected theme is dark; white is 255 and black is 0; so if text is whiter than bg, dark
        self.is_theme_dark = False
        if average_text_color > average_background_color:
            self.is_theme_dark = True
        theme['Is theme dark'] = self.isThemeDark

        # inc = 20

        # # Tab colors
        # base_color = theme['Dialog background color']
        # theme['Tab selected background color'] = (base_color[0] - inc,
        #                                           base_color[1] - inc,
        #                                           base_color[2] - inc)
        # theme['Tab selected text color'] = theme['Dialog text color']
        # theme['Tab background color'] = (base_color[0] - inc * 2,
        #                                  base_color[1] - inc * 2,
        #                                  base_color[2] - inc * 2)
        # theme['Tab text color'] = theme['Dialog text color']

        # theme['Convert to rgbhex'] = self.convert_qcolor_to_rgb_hex
        theme['Convert to rgbhex'] = self.rgb_to_hex
        theme['Convert to QColor'] = None

        return theme

    def convert_qcolor_to_rgb_hex(self, color):
        """Converts a QColor to a RGB hec color."""
        if isinstance(color, Variable):
            color = color.get_val()
        return '#%02x%02x%02x' % (color[0], color[1], color[2])

    def rgb_to_hex(self, rgb_tuple):
        """Convert an RGB tuple to a hexadecimal color string."""
        if isinstance(rgb_tuple, (tuple, list)) and len(rgb_tuple) == 3:
            red = rgb_tuple[0]
            green = rgb_tuple[1]
            blue = rgb_tuple[2]
            if red < 0 or red > 255 or green < 0 or green > 255 or blue < 0 or blue > 255:
                print('RGB values must be between 0 and 255')
                if red < 0:
                    red = 0
                elif red > 255:
                    red = 255
                if green < 0:
                    green = 0
                elif green > 255:
                    green = 255
                if blue < 0:
                    blue = 0
                elif blue > 255:
                    blue = 255
            return "#{:02x}{:02x}{:02x}".format(red, green, blue)
        return None
