"""CalcData Class."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.calculator.variable_group import VariableGroup
from xms.FhwaVariable.core_data.model.commands.command import BatchCommandItem
from xms.FhwaVariable.core_data.variables.variable import Variable


class CalcData(VariableGroup):
    """A base class that defines the basics of a CalcData that controls a dialog."""

    def __init__(self, theme: dict = None, stand_alone_calc: bool = True, app_data=None,
                 model_name: str = None, project_uuid: str = None):
        """Initializes the CalcData.

        Args:
            theme (dict): the theme for the CalcData
            stand_alone_calc (bool): whether the CalcData is standalone
            app_data (AppData): the application data
            model_name (str): the name of the model
            project_uuid (str): the project UUID
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.class_name = ''  # Set by programmer to match name given in ModelBase

        # TODO: Review name, type, and class_name. See if I should combine at least two of them.
        # Provide better documentation to programmers on how to use these variables.

        # name and/or class_name is shown to the user?  name sets the dialog title?
        # class_name sets the name shown in menus/toolbar?
        # type is used when reading or programatically recognizing the CalcData?

        # self.name = "Gradually-Varied Flow (GVF) CalcData"
        # self.type = "GVFCalc"
        # self.class_name = 'Gradually-Varied Flow (GVF) CalcData'

        self.command_manager = None
        self.selected_unit_system = 'U.S. Customary Units'

        self.calc_support_dual_input = False  # Displays a second column of input (HY-8 Style)
        self.stack_input_tables = True  # Stack the input tables, False will put them side by side
        self.calc_support_warnings = True  # Displays the warnings text box
        self.calc_support_results = True  # Displays the Results table box
        self.calc_support_plot = True  # Displays the plot
        self.calc_save_on_accept = False  # Saves the CalcData to a file
        self.calc_filename = Path('')  # The filename to save or read

        self.dlg_width = None
        self.dlg_height = None

        self.min_y_size_for_plot_and_wiki = 1080

        self.show_wiki = True
        self.show_plot = True

        # Dual Input (second)
        self.input_name = 'Input'
        self.dual_input_name = 'Input (continued)'

        # self.input_dual = {}
        # if input_dual is not None:
        #     self.input_dual = input_dual

        # Open this wiki page when the help is clicked
        self.help_url = 'https://www.xmswiki.com/wiki/FHWA_Hydraulic_Toolbox'

        self.stand_alone_calc = stand_alone_calc
        self.setup_table = False

        self.theme = theme
        if self.theme is None and app_data is not None:
            self.theme = app_data.get_theme()

        self.unknown = None

        # 2D dict, Allow multiple plots
        self.plots = {}
        # self.plots['profile'] = {}
        # self.plots['profile']['Plot name'] = "Weir Profile"
        # self.plots['profile']['Legend'] = "upper right"

        # Define the following variable for each plot:
        # self.input['Plot options'] = {}
        # If I define it here, it will show up first and we usually want it to show up last

        # reference documents (two otions for use)
        # self.reference['pdf Title'] = document_path
        # self.reference['pdf Title'] = (document_path, page_number)
        self.reference_pdfs = {}

        self.show_change_for_types = ['float', 'int', 'list', 'bool',]
        self.show_change_for_types.extend(Variable.string_types)

    def get_filename(self):
        """The filename to write or read, if calc_save_on_accept is True."""
        return self.calc_filename

    def check_for_needed_data(self):
        """Checks for needed data for the CalcData to run.

        Else, show popup with warnings and do not run dialog.
        """
        return True

    def get_input_tab_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        input_vars = copy.deepcopy(self.get_input_group())

        # remove plot options that are too complex
        if 'Plot options' in input_vars:
            names_to_pop = []
            # sys_complexity = 0
            # result, sys_c_var = self.get_setting_var('Complexity')
            # result, null = self.get_setting('Null data', default=-9999.0)
            # if result:
            #     sys_complexity = sys_c_var.get_index(null)
            #     if sys_complexity is None:
            #         sys_complexity = 0
            sys_complexity = self.get_system_complexity_index()
            for plot_name in input_vars['Plot options']:
                if input_vars['Plot options'][plot_name].complexity > sys_complexity:
                    names_to_pop.append(plot_name)
            for name in names_to_pop:
                input_vars['Plot options'].pop(name)
            if not input_vars['Plot options']:
                input_vars.pop('Plot options')

        if 'Plot options' in input_vars:
            plot_options = copy.deepcopy(input_vars['Plot options'])
            input_vars.pop('Plot options')
            return {'Input': input_vars, 'Plot options': plot_options}

        return {'Input': input_vars}

    def get_input_dual_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        return None

    def get_input_dual_tab_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        input_vars = self.get_input_dual_group()

        if input_vars is None:
            return None

        return {'Input': input_vars}

    def get_tables_with_vertical_direction(self):
        """Returns the tables with vertical direction.

        Returns:
            list of str: tables with vertical direction
        """
        return []

    def get_results_tab_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
              results_vars (dictionary of variables): the input variables
        """
        results_vars = self.get_results_group()

        return {'Results': results_vars}

    def check_warnings(self):
        """Checks for warnings that are given during computations or a check if we can compute (get_can_compute).

        Returns:
            list of str: The warnings found (if any)
        """
        return self.warnings

    def set_val_by_uuid(self, row_uuid, new_value, index=None, is_unit_change=False, prev_value=None, add_undo=True,
                        uuid_index=None):
        """Set the value of a variable by its UUID.

        Args:
            row_uuid (str): the UUID of the variable
            new_value (str): the new value
            index (int): index of the variable
            is_unit_change (bool): whether the change is a unit change
            prev_value (str): the previous value
            add_undo (bool): whether to add to the undo stack
            uuid_index (int): the index of the UUID in the list
        """
        result, item = self.find_item_by_uuid(row_uuid)
        if not result:
            return False
        if self.command_manager and add_undo:
            if prev_value is None:
                if hasattr(item, 'get_val'):
                    prev_value = item.get_val()
                if uuid_index is not None:
                    prev_value = item.item_list[uuid_index]
            name = f"Changed '{item.name}'"
            if item.type in self.show_change_for_types:
                name = f"Set '{item.name}' to {new_value}"
            if is_unit_change:
                name = f"Changed '{item.name}' units to {new_value}"
            cmd = 'set_val_by_uuid'
            if is_unit_change:
                cmd = 'set_val_by_uuid_unit_change'
            new_cmd = BatchCommandItem(command=cmd, batch_uuid=row_uuid, new_value=new_value, prev_value=prev_value,
                                       index=index, name=name, uuid_index=uuid_index)
            self.command_manager.add_new_undo_command(new_cmd)
        if uuid_index is not None:
            if isinstance(new_value, str):
                if hasattr(item.item_list[uuid_index], 'name'):
                    item.item_list[uuid_index].name = new_value
                if 'Name' in item.item_list[uuid_index].input:
                    item.item_list[uuid_index].input['Name'].set_val(new_value)
            else:
                item.item_list[uuid_index] = new_value
        if isinstance(item, Variable):
            if not is_unit_change:
                item.set_val(new_value, self.app_data, index=index)
            else:
                item.set_selected_unit(self.selected_unit_system, new_value)
        else:
            item = new_value

        return True

    def operate_command(self, command_item, new_value, prev_value=None):
        """Operate the command.

        Args:
            command_item (CommandItem): the command item
            new_value (str): the new value
            prev_value (str): the previous value
        """
        if command_item.command == 'set_val_by_uuid':
            self.set_val_by_uuid(command_item.uuid, new_value, index=command_item.index, prev_value=prev_value,
                                 add_undo=False)
        elif command_item.command == 'set_val_by_uuid_unit_change':
            self.set_val_by_uuid(command_item.uuid, new_value, index=command_item.index, is_unit_change=True,
                                 prev_value=prev_value, add_undo=False)
