"""SeriesOptions Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules


class PointOptions(VariableGroup):
    """Class that defines the data for plotting options."""

    def __init__(self, input_list=None, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Plot Series Options.

        Args:
            input_list (dict): dictionary of input variables
            app_data (AppData): application data (for settings)
            model_name (str): name of the model
            project_uuid (uuid): project uuid
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'Plot line Options'
        self.type = 'LineOptions'

        self.input_list = input_list

        # Input
        self.marker_types = ['none', 'pixel', 'point', 'circle', 'triangle-down', 'triangle-up', 'triangle-left',
                             'triangle-right', 'triangle-down', 'triangle-up', 'triangle-left', 'triangle-right',
                             'triangle-down', 'triangle-up', 'triangle-left', 'triangle-right', 'octagon', 'square',
                             'pentagon', 'plus (filled)', 'plus', 'star', 'hexagon1', 'hexagon2', 'plus', 'x', 'x',
                             'diamond', 'thin-diamond', 'vline', 'hline', 'tickleft', 'tickright', 'tickup',
                             'tickdown', 'caretleft', 'caretright', 'caretup', 'caretdown', 'caretleft', 'caretright',
                             'caretup', 'caretdown']

        self.dataset_names = []
        self.input_names = {}
        if input_list is not None:
            for input in input_list:
                self.dataset_names.append(input_list[input].name)
                self.input_names[input] = input_list[input].name

        self.x_var = None
        self.y_var = None

        self.input['Name'] = Variable('Name', 'string', self.name)
        self.input['Display'] = Variable('Display', 'bool', True)
        self.input['X axis'] = Variable('X-Axis', 'list', 0, self.dataset_names, complexity=3)
        self.input['Y axis'] = Variable('Y-Axis', 'list', 0, self.dataset_names, complexity=3)
        self.input['Plot yy'] = Variable('Plot on YY-Axis', 'bool', False, [], complexity=3,
                                         note='Plot on the right axis')
        self.input['XX axis'] = Variable('XX-Axis', 'list', 0, self.dataset_names, complexity=3)
        self.input['YY axis'] = Variable('YY-Axis', 'list', 0, self.dataset_names, complexity=3)

        self.input['Point color'] = Variable('Point color', 'color', 'rgb', (256, 256, 256))
        self.input['Point marker'] = Variable('Point marker', 'list', 2, self.marker_types)
        self.input['Marker size'] = Variable('Marker size', 'float', 150, complexity=4)

        self.warnings = []
        self.results = {}
        # self.results['Results'] = Variable('Results', 'float_list', 0.0, [], precision=precision,
        #                                   unit_type=unit_type, native_unit=native_unit,
        #                                   us_units=us_units, si_units=si_units)

    def set_plot_point_options(self, name='', point_color=None, point_marker=None, marker_size=None):
        """Set the plot options.

        Args:
            name (str): name of the scatter set
            point_color (tuple): color of the point
            point_marker (str): marker type
            marker_size (float): size of the marker
        """
        if name is not None and name != '':
            self.name = name
            self.input['Name'].set_val(name)
        if point_color is not None:
            self.input['Point color'].set_val(point_color)
        if point_marker is not None:
            self.input['Point marker'].set_val(point_marker)
        if marker_size is not None:
            self.input['Marker size'].set_val(marker_size)

    def get_can_compute(self):
        """Determine whether we have enough data to compute.

        Returns:
            True, if we can compute; otherwise, False
        """
        return True, ''

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = {}

        input_vars = copy.deepcopy(self.input)

        return input_vars

    @staticmethod
    def convert_linetype_text_to_symbol(linetype):
        """Convert the line type text to a symbol.

        Args:
            linetype (str): line type text

        Returns:
            str: line type symbol
        """
        if linetype == 'none':
            return 'none'
        elif linetype == 'point':
            return '.'
        elif linetype == 'pixel':
            return ','
        elif linetype == 'circle':
            return 'o'
        elif linetype == 'triangle-down':
            return 'v'
        elif linetype == 'triangle-up':
            return '^'
        elif linetype == 'triangle-left':
            return '<'
        elif linetype == 'triangle-right':
            return '>'
        elif linetype == 'octagon':
            return '8'
        elif linetype == 'square':
            return 's'
        elif linetype == 'pentagon':
            return 'p'
        elif linetype == 'plus (filled)':
            return 'P'
        elif linetype == 'plus':
            return '+'
        elif linetype == 'star':
            return '*'
        elif linetype == 'hexagon1':
            return 'h'
        elif linetype == 'hexagon2':
            return 'H'
        elif linetype == 'x':
            return 'x'
        elif linetype == 'diamond':
            return 'D'
        elif linetype == 'thin-diamond':
            return 'd'
        elif linetype == 'vline':
            return '|'
        elif linetype == 'hline':
            return '_'
        elif linetype == 'tickleft':
            return '0'
        elif linetype == 'tickright':
            return '1'
        elif linetype == 'tickup':
            return '2'
        elif linetype == 'tickdown':
            return '3'
        elif linetype == 'caretleft':
            return '4'
        elif linetype == 'caretright':
            return '5'
        elif linetype == 'caretup':
            return '6'
        elif linetype == 'caretdown':
            return '7'
        elif linetype == 'caretleft':
            return '8'
        elif linetype == 'caretright':
            return '9'
        elif linetype == 'caretup':
            return '10'
        elif linetype == 'caretdown':
            return '11'
        else:
            return 'none'

    def get_key(self, axis):
        """Get the key for the axis.

        Args:
            axis (str): axis ['X axis', 'Y axis', 'XX axis', 'YY axis']
        """
        dataset_name = self.input[axis].get_val()
        for key in self.input_names:
            if self.input_names[key] == dataset_name:
                return key
        return dataset_name

    def get_plot_options_dict(self):
        """Get the plot options dictionary."""
        plot_options = {}

        x_key = self.get_key('X axis')
        y_key = self.get_key('Y axis')

        x_var = self.x_var
        y_var = self.y_var
        if x_var is None:
            if x_key in self.input_list:
                x_var = copy.deepcopy(self.input_list[x_key])
            if y_key in self.input_list:
                y_var = copy.deepcopy(self.input_list[y_key])

        plot_options['x var'] = x_var
        plot_options['y var'] = y_var

        plot_options['Name'] = self.input['Name'].get_val()
        plot_options['Point color'] = self.input['Point color'].get_val()
        plot_options['Point marker'] = self.convert_linetype_text_to_symbol(
            self.input['Point marker'].get_val())
        plot_options['Marker size'] = self.input['Marker size'].get_val()

        return plot_options
