"""TableCalc Class."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
from pathlib import Path
import sys

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.calculator.calcdata import CalcData
from xms.FhwaVariable.core_data.calculator.plot.plot_options import PlotOptions
from xms.FhwaVariable.core_data.calculator.table.table_options import TableOptions
from xms.FhwaVariable.core_data.calculator.table_calc import TableCalc
from xms.FhwaVariable.core_data.calculator.variable_group import VariableGroup
from xms.FhwaVariable.core_data.variables.variable import Variable


class TableData(CalcData):
    """Class that defines a table and plot of related data."""

    def __init__(self, theme, name, plot_names, app_data=None, model_name=None, project_uuid=None,
                 name_of_items='points', stand_alone_calc=True, input=None, default_number_of_items=1,
                 min_items=1, max_items=sys.maxsize, show_increment=False,):
        """Initializes the CalcData.

        Args:
            theme (dict): the theme
            name (str): the name of the CalcData
            plot_names (list of str): list of plot names
            app_data (AppData): the app data (for settings)
            name_of_items (str): name of the items
            stand_alone_calc (bool): whether the CalcData is a stand-alone CalcData
            input (dict): dictionary of input variables
            default_number_of_items (int): default Number of items
            min_items (int): minimum Number of items
            max_items (int): maximum Number of items
            show_increment (bool): whether to show the increment toggle
            model_name (str): the model name
            project_uuid (str): the project uuid
        """
        super().__init__(theme, app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = name

        self.calculator = TableCalc()

        self.calc_support_dual_input = True  # Displays a second column of input (HY-8 Style)
        self.calc_support_warnings = True  # Displays the warnings text box
        self.calc_support_results = False  # Displays the Results table box
        self.calc_support_plot = True  # Displays the plot
        self.calc_save_on_accept = False  # Saves the CalcData to a file

        self.dlg_width = None
        self.dlg_height = None

        # Open this wiki page when the help is clicked
        # self.self.help_url = 'https://www.xmswiki.com/wiki/SMS:SRH-2D_Channel_Calculator'

        self.stand_alone_calc = stand_alone_calc
        self.setup_table = True

        self.unknown = None

        # self.app_data = None
        self.theme = theme

        # self.input_options = {}
        # self.input_options['Number of items'] = Variable(f'number of {name_of_items}', 'int', min_items,
        #                                                  limits=(min_items, max_items))

        self.input['Table options'] = Variable('Table options', 'class',
                                               TableOptions(name_of_items, app_data=app_data, model_name=model_name,
                                                            project_uuid=project_uuid, min_items=min_items,
                                                            max_items=max_items, show_increment=show_increment))
        self.input['Table options'].get_val().input['Number of items'].set_val(default_number_of_items)
        self.input['Data input'] = Variable('Data input', 'class', VariableGroup(
            input=input, app_data=app_data, model_name=model_name, project_uuid=project_uuid))
        self.input['Plot options'] = {}

        # self.input_dual['Placeholder'] = Variable('Placeholder', 'string', 'placeholder')

        # Add plot options for each plot and the input included
        for plot_name in plot_names:
            input_dict = {}
            input_dict[plot_name] = input

            self.input['Plot options'][plot_name] = Variable(
                f'{plot_name} plot options', 'class', PlotOptions(plot_name, input_dict=input_dict, show_series=True,
                                                                  app_data=app_data, model_name=model_name,
                                                                  project_uuid=project_uuid),
                complexity=1)
            # self.input['{plot_name} plot options'].get_val().input[plot_name] = Variable(
            #     f'{plot_name} plot options', "class", PlotOptions(plot_name, input), )

        # 2D dict, Allow multiple plots
        if plot_names is None:
            plot_names = []
        self.plot_names = plot_names
        self.plots = {}
        for name in plot_names:
            self.plots[name] = {}
            self.plots[name]['Plot name'] = name
        # self.plots['profile'] = {}
        # self.plots['profile']['Plot name'] = "Weir Profile"
        # self.plots['profile']['Legend'] = "best"

        # reference documents (two otions for use)
        # self.reference['pdf Title'] = document_path
        # self.reference['pdf Title'] = (document_path, page_number)
        self.reference_pdfs = {}

    def check_list_length(self):
        """Checks the length of the list and adds items if necessary."""
        num_items = self.input['Table options'].get_val().input['Number of items'].get_val()

        for input in self.input['Data input'].get_val().input:
            while num_items > len(self.input['Data input'].get_val().input[input].value_options):
                self.input['Data input'].get_val().input[input].value_options.append(0.0)

    def get_filename(self):
        """The filename to write or read, if calc_save_on_accept is True."""
        return Path('')

    def set_plot_line_options(self, plot_name, related_index, index, x_axis, y_axis, name='', line_color=None,
                              linetype=None, line_width=None, fill_below_line=None, fill_color=None):
        """Set the plot options.

        Args:
            plot_name (str): name of the plot
            related_index (int): index of the data series
            index (int): index of the plot
            x_axis (str): name of dataset to use for the y-axis data
            y_axis (str): name of dataset to use for the y-axis data
            name (str): name of the series
            line_color (color tuple): line color
            linetype (str): line type ['solid', 'dashed', 'dotted', 'dash-dot']
            line_width (float): line width
            fill_below_line (bool): fill below line
            fill_color (color tuple): fill color
        """
        key_name = f'{plot_name} plot options'
        if key_name not in self.input:
            return

        self.input[key_name].get_val().set_plot_line_options(
            related_index=related_index, index=index, x_axis=x_axis, y_axis=y_axis, name=name, line_color=line_color,
            linetype=linetype, line_width=line_width, fill_below_line=fill_below_line, fill_color=fill_color)

    def set_name(self, name):
        """Set the name of the CalcData.

        Args:
            name (str): name of the CalcData
        """
        self.plots[name] = copy.deepcopy(self.plots[self.name])
        self.plots[name]['Plot name'] = name
        self.plots.pop(self.name)
        self.input['Plot options'][name] = copy.deepcopy(self.input['Plot options'][self.name])
        self.input['Plot options'].pop(self.name)
        self.name = name

    def set_show_num_items(self, show_num_items):
        """Set the show number of items.

        Args:
            show_num_items (bool): show number of items
        """
        self.input['Table options'].get_val().show_num_items = show_num_items

    def set_plot_series_options(self, plot_name, related_index, index, x_axis, y_axis, name='', line_color=None,
                                linetype=None, line_width=None, fill_below_line=None, fill_color=None, pattern=None,
                                density=None):
        """Set the plot options.

        Args:
            plot_name (str): name of the plot
            related_index (int): index of the related plot
            index (int): index of the plot
            x_axis (str): name of dataset to use for the y-axis data
            y_axis (str): name of dataset to use for the y-axis data
            name (str): name of the series
            line_color (color tuple): line color
            linetype (str): line type ['solid', 'dashed', 'dotted', 'dash-dot']
            line_width (float): line width
            fill_below_line (bool): fill below line
            fill_color (color tuple): fill color
            pattern (str): fill pattern
            density (int): fill pattern density
        """
        if plot_name not in self.input['Plot options']:
            return

        self.input['Plot options'][plot_name].get_val().set_plot_series_options(
            related_index=related_index, index=index, x_axis=x_axis, y_axis=y_axis, name=name, line_color=line_color,
            linetype=linetype, line_width=line_width, fill_below_line=fill_below_line, fill_color=fill_color,
            pattern=pattern, density=density)

        # self.input['Plot series'].get_val().item_list[index].set_plot_options(x_axis, y_axis, line_color, linetype,
        #                                                                       line_width, fill_below_line, fill_color)

    def set_plot_log_options(self, plot_name, x_axis_log, y_axis_log):
        """Set the plot log options.

        Args:
            plot_name (str): name of the plot
            x_axis_log (bool): log x-axis
            y_axis_log (bool): log y-axis
        """
        # self.input['Table options'].get_val().input[plot_name].get_val().set_plot_log_options(x_axis_log, y_axis_log)
        self.input['Plot options']['Default gradation'].get_val().set_plot_log_options(x_axis_log, y_axis_log)

    def get_input_tab_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        # Update the numbers; Do not remove any items. It may be useful to the user to keep the data, or may
        # accidentally shrink it. Use the default value for new items.
        number_of_items = self.input['Table options'].get_val().input['Number of items'].get_val()
        if number_of_items < self.input['Table options'].get_val().input['Number of items'].limits[0]:
            number_of_items = self.input['Table options'].get_val().input['Number of items'].limits[0]
            self.input['Table options'].get_val().input['Number of items'].set_val(number_of_items)
        for input in self.input['Data input'].get_val().input:
            self.input['Data input'].get_val().input[input].value = number_of_items
            if len(self.input['Data input'].get_val().input[input].value_options) < number_of_items:
                values = self.input['Data input'].get_val().input[input].value_options
                vals = [self.input['Data input'].get_val().input[input].default_value] * number_of_items
                self.input['Data input'].get_val().input[input].value_options = vals
                for i, value in enumerate(values):
                    self.input['Data input'].get_val().input[input].value_options[i] = value

        # sys_complexity = self.app_data.get_setting_var('Complexity')[1].value
        sys_complexity = self.get_system_complexity_index()
        input_vars = {}

        for input in self.input:
            if isinstance(self.input[input], dict):
                input_vars[input] = copy.deepcopy(self.input[input])
            else:
                input_vars[input] = copy.deepcopy(self.input[input].get_val().get_input_group())

        if self.app_data.get_setting('Dual column input'):
            # input_vars.extend(self.input_dual)
            input_vars.pop('Data input')

        # remove plot options that are too complex
        if 'Plot options' in input_vars:
            names_to_pop = []
            for plot_name in input_vars['Plot options']:
                if input_vars['Plot options'][plot_name].complexity > sys_complexity:
                    names_to_pop.append(plot_name)
            for name in names_to_pop:
                input_vars['Plot options'].pop(name)
            if not input_vars['Plot options']:
                input_vars.pop('Plot options')

        return input_vars

    def get_input_dual_tab_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        input_vars = self.get_input_dual_group()

        return {'Data table': input_vars['Data input'].value.input}

    def get_input_dual_group(self, unknown=None):
        """Returns the input group for the user interface.

        Args:
            unknown (string): variable that is unknown

        Returns:
            input_vars (list of variables): input group for the user interface's input table
        """
        input_vars = {}
        input_vars['Data input'] = copy.deepcopy(self.input['Data input'])

        return input_vars

    def get_tables_with_vertical_direction(self):
        """Returns the tables with vertical direction.

        Returns:
            list of str: tables with vertical direction
        """
        if self.input['Table options'].get_val().input['Table direction'].get_val() == 'vertical':
            return ['Data table']
        return []

    def get_results_tab_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
              results_vars (dictionary of variables): the input variables
        """
        results_vars = self.get_results_group()

        return {'Results': results_vars}

    def check_warnings(self):
        """Checks for warnings that are given during computations or a check if we can compute (get_can_compute).

        Returns:
            list of str: The warnings found (if any)
        """
        warning = None
        if warning:
            self.warnings.append(warning)
        return self.warnings

    def get_plot_options(self, plot_name):
        """Get the plot options.

        Args:
            plot_name (str): name of the plot

        Returns:
            dict: plot options
        """
        return self.input['Plot options'][plot_name].get_val()

    def get_plot_options_dict(self, plot_name):
        """Get the plot options.

        Args:
            plot_name (str): name of the plot

        Returns:
            dict: plot options
        """
        return self.get_plot_options(plot_name).get_plot_options_dict()

    def find_var_by_uuid(self, uuid):
        """Finds a variable by its uuid.

        Args:
            uuid (string): uuid of the variable

        Returns:
            variable: the variable
        """
        for item in self.item_list:
            result, item = item.find_var_by_uuid(uuid)
            if result:
                return result, item
        for var in self.input:
            result, item = var.find_var_by_uuid(uuid)
            if result:
                return result, item

        for var in self.results:
            result, item = var.find_var_by_uuid(uuid)
            if result:
                return result, item

        return False, None
