"""Classes to handle the GUI side of variables."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.interface_adapters.view_model.main.tool import ToolItem


class ToolManager():
    """Class to manage the tools."""

    def __init__(self, app_data):
        """Initialize the ToolManager class."""
        self.tools = {}

        # current_file_directory = os.path.dirname(os.path.abspath(__file__))
        # two_levels_up_directory = os.path.dirname(current_file_directory)
        # icon_folder = os.path.join(two_levels_up_directory, 'resources', 'icons')
        current_file_directory = Path(__file__).parent.parent
        two_levels_up_directory = current_file_directory.parent
        icon_folder = two_levels_up_directory / 'resources' / 'icons'

        new_icon = icon_folder / 'File_New.ico'
        add_icon = icon_folder / 'File_Add.ico'
        close_icon = icon_folder / 'File_Close.ico'
        open_icon = icon_folder / 'File_Open.ico'
        save_icon = icon_folder / 'File_Save.ico'
        settings_icon = icon_folder / 'GreyGear.ico'
        # project_settings_icon = icon_folder / 'GreyGear2.ico'
        support_window_icon = icon_folder / 'add_window.ico'
        show_wiki_panel_icon = icon_folder / 'question.png'
        undo_icon = icon_folder / 'undo.ico'
        redo_icon = icon_folder / 'redo.ico'
        edit_icon = icon_folder / 'Edit.ico'
        metadata_icon = icon_folder / 'EditMetadata.ico'
        delete_icon = icon_folder / 'Delete.ico'
        duplicate_icon = icon_folder / 'Duplicate.ico'
        profile_icon = icon_folder / 'profile.ico'
        delete_profile_icon = icon_folder / 'DeleteProfile.ico'
        # report_icon = icon_folder / 'report.ico'

        if app_data.graphics_view is not None:
            # orientation
            pan_icon = icon_folder / 'view' / 'pan.svg'
            zoom_icon = icon_folder / 'view' / 'zoom.svg'
            rotate_icon = icon_folder / 'view' / 'rotate.svg'
            # Display
            doc_window_icon = icon_folder / 'view' / 'new_View.png'
            close_doc_window_icon = icon_folder / 'view' / 'close_View.png'
            refresh_icon = icon_folder / 'view' / 'refresh.bmp'
            frame_icon = icon_folder / 'view' / 'frame_view.svg'
            # display_settings_icon = icon_folder / 'view' / 'display_settings.svg'
            plan_view_icon = icon_folder / 'view' / 'plan_view.svg'
            front_view_icon = icon_folder / 'view' / 'front_view.svg'
            side_view_icon = icon_folder / 'view' / 'side_view.svg'
            oblique_view_icon = icon_folder / 'view' / 'oblique_view.svg'
            prev_view_icon = icon_folder / 'view' / 'prev_view.ico'
            next_view_icon = icon_folder / 'view' / 'next_view.ico'

        # -- From SMS  --
        # Display: refresh, frame, display settings, plan view, front view, side view, oblique view
        # Optional macros: lighting, contour options, vector options, get module info, Plot wizard, Add online maps,
        #   map locator, build polygons, clean, toolbox
        # Modules: mesh, UGrid, cartesian grid, scatter set, map, GIS, 1-D, 1-D grid, particle, annotation
        # == Following are tools == (turned on and changes the cursor) that are not in the menu bar:
        # orientation; pan, zoom, rotate
        # ? measure, import from web
        # dynamic tool:
        #   UGrid: select point, create point, select cell, create tri elem, create quad elem, swap edges, edit plane,
        #       create cross-section
        #   Map: select objects, select feature points, create feature point, select feature vertex,
        #       create feature vertex, select feature arc, create feature arc, select feature arc group,
        #       select feature polygon, create 1-D grid frame, select 1-D grid frame, create 2-D grid frame,
        #       select 2-D grid frame, select compass plot

        # File
        self.tools['File'] = {}
        # New
        self.tools['File']['New'] = ToolItem('New', tool_tip='Clear contents to new workspace', icon=new_icon,
                                             shortcut='Ctrl+N')
        # Add project
        self.tools['File']['Add project'] = ToolItem('Add project', tool_tip='Create a new project', icon=add_icon)
        # Close project
        # self.tools['File']['Close project'] = ToolItem('Close project', tool_tip='Close selected project',
        #                                                icon=close_icon)
        # open
        self.tools['File']['Open'] = ToolItem('Open', tool_tip='Open existing project file', icon=open_icon,
                                              shortcut='Ctrl+O')
        # Save All
        self.tools['File']['Save all'] = ToolItem('Save all', tool_tip='Save all of the projects', icon=save_icon,
                                                  shortcut='Ctrl+Shift+S')
        # Save As
        self.tools['File']['Save as'] = ToolItem('Save as', tool_tip='Save the current project as a new file')
        self.tools['File']['Save as'].is_enabled = False
        # --
        self.tools['File']['S1'] = ToolItem(is_separator=True)
        # open recent
        self.tools['File']['Recent'] = {}
        # --
        self.tools['File']['S2'] = ToolItem(is_separator=True)
        # Settings
        self.tools['File']['Settings'] = ToolItem('Application Settings', tool_tip='Change the program settings',
                                                  icon=settings_icon, shortcut='Ctrl+P',
                                                  item_uuid=app_data.app_settings.uuid)
        # --
        self.tools['File']['S3'] = ToolItem(is_separator=True)
        # Exit
        self.tools['File']['Exit'] = ToolItem('Exit', tool_tip='Exit the program', shortcut='Ctrl+Q')

        # Edit
        self.tools['Edit'] = {}

        # self.tools['Edit']['S1'] = ToolItem(is_separator=True)

        # self.tools['Edit']['Edit project settings'] = ToolItem(
        #     'Edit project settings', icon=project_settings_icon, tool_tip='Edit the project settings', shortcut='')
        # self.tools['Edit']['Edit project metadata'] = ToolItem(
        #     'Edit project metadata', tool_tip='Edit the project metadata', shortcut='')

        self.tools['Edit']['S1'] = ToolItem(is_separator=True)
        self.tools['Edit']['Support window'] = ToolItem(
            'Create Support Window', icon=support_window_icon,
            tool_tip='Create a support window to display help and plot results', shortcut='Ctrl+alt+W')

        self.tools['Edit']['Show wiki panel'] = ToolItem(
            'Show wiki panel', icon=show_wiki_panel_icon,
            tool_tip='Show/Hide the wiki panel',)

        self.tools['Edit']['S2'] = ToolItem(is_separator=True)
        # Undo
        self.tools['Edit']['Undo'] = ToolItem('Undo', icon=undo_icon, tool_tip='Undo the last action',
                                              shortcut='Ctrl+Z')
        self.tools['Edit']['Undo history'] = {}
        # Redo
        self.tools['Edit']['Redo'] = ToolItem('Redo', icon=redo_icon, tool_tip='Redo the last action',
                                              shortcut='Ctrl+Y')
        self.tools['Edit']['Redo history'] = {}
        # --
        self.tools['Edit']['S3'] = ToolItem(is_separator=True)
        # Rename
        self.tools['Edit']['Rename'] = ToolItem('Rename', tool_tip='Rename', shortcut='F2')
        # Cut
        self.tools['Edit']['Cut'] = ToolItem('Cut', tool_tip='Cut', shortcut='Ctrl+X')
        # Copy
        self.tools['Edit']['Copy'] = ToolItem('Copy', tool_tip='Copy', shortcut='Ctrl+C')
        # Paste
        self.tools['Edit']['Paste'] = ToolItem('Paste', tool_tip='Paste', shortcut='Ctrl+V')
        # Delete  - Rescind in favor of CalcData delete
        # self.tools['Edit']['Delete'] = ToolItem('Delete', tool_tip='Delete', shortcut='Del')
        # --
        self.tools['Edit']['S4'] = ToolItem(is_separator=True)
        # Select all
        self.tools['Edit']['Select all'] = ToolItem('Select all', tool_tip='Select all', shortcut='Ctrl+A')

        # --Graphics--
        if app_data.graphics_view is not None:
            # Orientation (tools): pan, zoom, rotate
            self.tools['Orientation'] = {}
            self.tools['Orientation']['Pan'] = ToolItem('Pan', icon=pan_icon, tool_tip='Pan the view', menu_group=None)
            self.tools['Orientation']['Pan'].is_checked = False
            self.tools['Orientation']['Zoom'] = ToolItem('Zoom', icon=zoom_icon, tool_tip='Zoom the view',
                                                         menu_group=None)
            self.tools['Orientation']['Zoom'].is_checked = False
            self.tools['Orientation']['Rotate'] = ToolItem('Rotate', icon=rotate_icon, tool_tip='Rotate the view',
                                                           menu_group=None)
            self.tools['Orientation']['Rotate'].is_checked = False

            # Display: refresh, frame, display settings, plan view, front view, side view, oblique view
            self.tools['Display'] = {}
            self.tools['Display']['Document window'] = ToolItem('Document window', icon=doc_window_icon,
                                                                tool_tip='Create a new document window',
                                                                shortcut='Ctrl+Shift+W')
            self.tools['Display']['Close document window'] = ToolItem('Close document window',
                                                                      icon=close_doc_window_icon,
                                                                      tool_tip='Close the active document window',
                                                                      shortcut='Ctrl+Shift+Q')
            self.tools['Display']['Refresh'] = ToolItem('Refresh', icon=refresh_icon, tool_tip='Refresh the view',
                                                        shortcut='Shift+R')
            # self.tools['Display']['Display settings'] = ToolItem('Display settings', icon=display_settings_icon,
            #                                                    tool_tip='Display settings', shortcut='Shift+D')
            self.tools['Display']['Frame'] = ToolItem('Frame', icon=frame_icon, tool_tip='Frame the view',
                                                      shortcut='F', menu_group='view')
            self.tools['Display']['Plan view'] = ToolItem('Plan view', icon=plan_view_icon,
                                                          tool_tip='Plan view', shortcut='Shift+P', menu_group='view')
            self.tools['Display']['Front view'] = ToolItem('Front view', icon=front_view_icon,
                                                           tool_tip='Front view', shortcut='Shift+F', menu_group='view')
            self.tools['Display']['Side view'] = ToolItem('Side view', icon=side_view_icon,
                                                          tool_tip='Side view', shortcut='Shift+S', menu_group='view')
            self.tools['Display']['Oblique view'] = ToolItem('Oblique view', icon=oblique_view_icon,
                                                             tool_tip='Oblique view', shortcut='Shift+O',
                                                             menu_group='view')
            self.tools['Display']['Previous view'] = ToolItem('Previous view', icon=prev_view_icon,
                                                              tool_tip='Previous view', shortcut='Shift+Z',
                                                              menu_group='view')
            self.tools['Display']['Next view'] = ToolItem('Next view', icon=next_view_icon,
                                                          tool_tip='Next view', shortcut='Shift+X', menu_group='view')

        # Calculators
        self.tools['Calculators'] = {}
        # Edit project settings

        # self.tools['Calculators']['S1'] = ToolItem(is_separator=True)

        # Edit Calculator
        self.tools['Calculators']['Edit Calculator'] = ToolItem(
            'Edit Calculator', icon=edit_icon, tool_tip='Edit the calculator', shortcut='')
        self.tools['Calculators']['Edit Calculator'].is_enabled = False
        # Notes CalcData
        self.tools['Calculators']['Edit metadata'] = ToolItem('Edit metadata', icon=metadata_icon,
                                                              tool_tip='Edit the metadata for the calculator',
                                                              shortcut='')
        self.tools['Calculators']['Edit metadata'].is_enabled = False
        # Delete Calculator
        self.tools['Calculators']['Delete Calculator'] = ToolItem('Delete Calculator',
                                                                  tool_tip='Delete the calculator', icon=delete_icon,
                                                                  shortcut='Delete')
        self.tools['Calculators']['Delete Calculator'].is_enabled = False
        # Duplicate Calculator
        self.tools['Calculators']['Duplicate Calculator'] = ToolItem('Duplicate Calculator',
                                                                     tool_tip='Duplicate the calculator',
                                                                     icon=duplicate_icon,
                                                                     shortcut='')
        self.tools['Calculators']['Duplicate Calculator'].is_enabled = False
        # Create CalcData
        self.tools['Calculators']['S2'] = ToolItem(is_separator=True)

        # Profiles
        self.tools['Profiles'] = {}
        # profile_names = app_data.get_profile_names()
        # app_data.profile_var = Variable('Profiles', 'list', app_data.selected_profile_settings, profile_names)
        # app_data.profile_var.set_val = app_data.set_profile_val
        app_data.update_profile_names()
        self.tools['Profiles']['Profiles'] = ToolItem(
            'Profiles', tool_tip='Profile selection', shortcut='', item_uuid=app_data.profile_var.uuid,
            var=app_data.profile_var)
        self.tools['Edit']['S1'] = ToolItem(is_separator=True)
        self.tools['Profiles']['Edit'] = ToolItem('Edit Profile', tool_tip='Edit the selected profile',
                                                  icon=profile_icon, shortcut='')
        app_data.profile_icon = profile_icon
        self.tools['Profiles']['Delete'] = ToolItem('Delete Profile', tool_tip='Delete the selected profile',
                                                    icon=delete_profile_icon, shortcut='')
        self.tools['Edit']['S2'] = ToolItem(is_separator=True)
        self.tools['Profiles']['Import profile file'] = ToolItem(
            'Import profile file', tool_tip='Import profile from file', shortcut='')
        self.tools['Profiles']['Export profile file'] = ToolItem(
            'Export profile file', tool_tip='Export the selected profile to a file', shortcut='')

        # Units
        self.tools['Units'] = {}
        var = app_data.get_setting_var('Selected unit system')[1]
        self.tools['Units']['Units'] = ToolItem(
            'Units', tool_tip='Unit system selection', shortcut='', item_uuid=var.uuid,
            var=var)

        # Map
        self.tools['Map'] = {}

        # Reports
        self.tools['Reports'] = {}

        # Windows
        if app_data.graphics_view is not None:
            self.tools['Windows'] = {}
            self.tools['Windows']['main'] = {}
            self.tools['Windows']['main']['tile'] = ToolItem('Tile', tool_tip='Tile Windows')
            self.tools['Windows']['main']['cascade'] = ToolItem('Cascade', tool_tip='Cascade Windows')
            self.tools['Windows']['main']['minimize all'] = ToolItem('Minimize all', tool_tip='Minimize all Windows')
            self.tools['Windows']['main']['maximize all'] = ToolItem('Maximize all', tool_tip='Maximize all Windows')
            self.tools['Windows']['main']['restore all'] = ToolItem('Restore all', tool_tip='Restore all Windows')

            self.tools['Windows'][2] = {}
            self.tools['Windows'][2]['horizontal'] = ToolItem('Horizontal', tool_tip='2 windows horizontal',
                                                              menu_group='layouts')
            self.tools['Windows'][2]['vertical'] = ToolItem('Vertical', tool_tip='2 windows vertical',
                                                            menu_group='layouts')

            self.tools['Windows'][3] = {}
            self.tools['Windows'][3]['horizontal'] = ToolItem('Horizontal', tool_tip='3 windows horizontal',
                                                              menu_group='layouts')
            self.tools['Windows'][3]['vertical'] = ToolItem('Vertical', tool_tip='3 windows vertical',
                                                            menu_group='layouts')
            self.tools['Windows'][3]['primary and vertical'] = ToolItem(
                'Primary and vertical', tool_tip='primary window and 2 vertical windows on the right',
                menu_group='layouts')
            self.tools['Windows'][3]['primary and horizontal'] = ToolItem(
                'Primary and horizontal', tool_tip='primary window and 2 horizontal windows below',
                menu_group='layouts')

            self.tools['Windows'][4] = {}
            self.tools['Windows'][4]['horizontal'] = ToolItem('Horizontal', tool_tip='4 windows horizontal',
                                                              menu_group='layouts')
            self.tools['Windows'][4]['vertical'] = ToolItem('Vertical', tool_tip='4 windows vertical',
                                                            menu_group='layouts')
            self.tools['Windows'][4]['primary and vertical'] = ToolItem(
                'Primary and vertical', tool_tip='Primary window and 3 vertical on the right', menu_group='layouts')
            self.tools['Windows'][4]['primary and horizontal'] = ToolItem(
                'Primary and horizontal', tool_tip='Primary window and 3 horizontal below', menu_group='layouts')
            self.tools['Windows'][4]['grid'] = ToolItem('Grid', tool_tip='4 windows displayed in a grid',
                                                        menu_group='layouts')

            self.tools['Windows'][5] = {}
            self.tools['Windows'][5]['horizontal'] = ToolItem('Horizontal', tool_tip='5 windows horizontal',
                                                              menu_group='layouts')
            self.tools['Windows'][5]['vertical'] = ToolItem('Vertical', tool_tip='5 windows vertical',
                                                            menu_group='layouts')
            self.tools['Windows'][5]['primary and vertical'] = ToolItem(
                'Primary and four right', tool_tip='Primary window and four vertical windows on the right',
                menu_group='layouts')
            self.tools['Windows'][5]['primary and horizontal'] = ToolItem(
                'Primary and four below', tool_tip='Primary window and four horizontal windows below',
                menu_group='layouts')
            self.tools['Windows'][5]['two vertical and three vertical right'] = ToolItem(
                'Two vertical and three vertical right',
                tool_tip='Two vertical windows on the left and three vertical windows on the right',
                menu_group='layouts')
            self.tools['Windows'][5]['two horizontal and three horizontal below'] = ToolItem(
                'Two horizontal and three horizontal below',
                tool_tip='Two horizontal windows on top and three horizontal windows below', menu_group='layouts')
            self.tools['Windows'][5]['primary, two vertical center, and two vertical right'] = ToolItem(
                'Primary, two vertical center, and two vertical right',
                tool_tip='Primary window on the left, two vertical windows in the center, and two vertical windows '
                'on the right', menu_group='layouts')
            self.tools['Windows'][5]['primary center with four surrounding'] = ToolItem(
                'Primary Center with Four Surrounding',
                tool_tip='One primary window in the center, with four smaller windows surrounding it (top, bottom, '
                'left, and right)', menu_group='layouts')
            # self.tools['Windows'][5]['diagonal layout'] = ToolItem(
            #     'Diagonal layout',
            #     tool_tip='Two windows stacked diagonally on the left (top-left and bottom-left) and three windows '
            #     'stacked diagonally on the right (top-right, center-right, and bottom-right)', menu_group='layouts')
            self.tools['Windows'][5]['two vertical left, primary center, and two vertical right'] = ToolItem(
                'Two vertical left, primary center, and two vertical right',
                tool_tip='Two windows on the left (stacked vertically), one in the center, and two on the right '
                '(stacked vertically)', menu_group='layouts')
            # self.tools['Windows'][5]['grid layout (2x3 with one larger)'] = ToolItem(
            #     'Grid layout (2x3 with one larger)',
            #     tool_tip='A grid layout with two rows and three columns, where one window spans two rows or two '
            #     'columns.', menu_group='layouts')

            self.tools['Windows'][6] = {}
            self.tools['Windows'][6]['horizontal'] = ToolItem('Horizontal', tool_tip='6 windows horizontal',
                                                              menu_group='layouts')
            self.tools['Windows'][6]['vertical'] = ToolItem('Vertical', tool_tip='6 windows vertical',
                                                            menu_group='layouts')
            self.tools['Windows'][6]['primary and vertical'] = ToolItem(
                'Primary and five right', tool_tip='Primary window left and five vertical windows on the right',
                menu_group='layouts')
            self.tools['Windows'][6]['primary and horizontal'] = ToolItem(
                'Primary and five below', tool_tip='Primary window left and five horizontal windows below',
                menu_group='layouts')
            self.tools['Windows'][6]['primary and two vertical center and three vertical right'] = ToolItem(
                'Primary and two vertical center and three vertical right',
                tool_tip='Primary window, two vertical windows in the center, and three vertical windows on the right',
                menu_group='layouts')
            self.tools['Windows'][6]['primary and two horizontal center and three horizontal below'] = ToolItem(
                'Primary and two horizontal center and three horizontal below',
                tool_tip='Primary window, two horizontal windows in the center and three horizontal windows below',
                menu_group='layouts')
            self.tools['Windows'][6]['primary and two right, the upper with two horizontal, and the lower with three '
                                     'horizontal'] = ToolItem(
                'Primary and two right, the upper with two horizontal, and the lower with three horizontal',
                tool_tip='Primary and two right, the upper with two horizontal, and the lower with three horizontal',
                menu_group='layouts')
            self.tools['Windows'][6]['primary with two vertical right and three below'] = ToolItem(
                'Primary with two vertical right and three below',
                tool_tip='Primary window with two vertical windows to the right and three horizontal windows below',
                menu_group='layouts')
            self.tools['Windows'][6]['two vertical left, two vertical center, and two vertical right'] = ToolItem(
                'Two vertical left, two vertical center, and two vertical right',
                tool_tip='Two vertical windows on the left, two vertical windows in the center, and two vertical '
                'windows on the right', menu_group='layouts')
            self.tools['Windows'][6]['two vertical left, two horizontal center, and two vertical right'] = ToolItem(
                'Two vertical left, two horizontal center, and two vertical right',
                tool_tip='Two vertical windows on the left, two horizontal windows in the center, and two vertical '
                'windows on the right', menu_group='layouts')
            self.tools['Windows'][6]['three vertical left, three vertical right'] = ToolItem(
                'Three vertical left, three vertical right',
                tool_tip='Three vertical windows on the left, three vertical windows on the right',
                menu_group='layouts')
            # self.tools['Windows'][6]['three horizontal, three horizontal below'] = ToolItem(
            #     'Three horizontal, three horizontal below',
            #     tool_tip='Three horizontal windows, three horizontal windows below', menu_group='layouts')
            self.tools['Windows'][6]['primary center with five surrounding'] = ToolItem(
                'Primary center with five surrounding',
                tool_tip='One primary in the center, five surrounding (top, bottom, left, right, diagonal)',
                menu_group='layouts')
            # self.tools['Windows'][6]['grid layout (2x3)'] = ToolItem(
            #     'Grid layout (2x3)',
            #     tool_tip='Two rows and three columns, all windows of equal size.', menu_group='layouts')
            # self.tools['Windows'][6]['diagonal split layout'] = ToolItem(
            #     'Diagonal Split Layout',
            #     tool_tip='Three diagonally stacked on the left, three diagonally stacked on the right',
            #     menu_group='layouts')
            # self.tools['Windows'][6]['zigzag layout'] = ToolItem(
            #     'Zigzag layout',
            #     tool_tip='Three stacked vertically on the left, one in the center, two stacked vertically on the '
            #     'right', menu_group='layouts')
            # self.tools['Windows'][6]['primary with two center and four surrounding'] = ToolItem(
            #     'Primary with two center and four surrounding',
            #     tool_tip='One primary in the center, two stacked vertically in the center, and three surrounding',
            #     menu_group='layouts')
            self.tools['Windows'][6]["two vertical left, primary center, and three vertical right"] = ToolItem(
                "Two vertical left, primary center, and three vertical right",
                tool_tip='Two stacked vertically on the left, one large center window, three stacked vertically on '
                'the right', menu_group='layouts')

        # Help
        self.tools['Help'] = {}
        self.tools['Help']['User Manual'] = ToolItem('User Manual', tool_tip='User Manual', shortcut='CTRL+H')
        # Reference Documents
        self.tools['Help']['Reference Documents'] = {}
        # --
        self.tools['Help']['S1'] = ToolItem(is_separator=True)
        # Wiki
        self.tools['Help']['Wiki'] = {}
        # --
        self.tools['Help']['S2'] = ToolItem(is_separator=True)
        # Register Software
        self.tools['Help']['Register'] = ToolItem('Register', tool_tip='Register the software', shortcut='')
        # --
        self.tools['Help']['S3'] = ToolItem(is_separator=True)
        # About
        self.tools['Help']['About'] = ToolItem('About', tool_tip='About the software', shortcut='')

        # Project
        self.tools['Project'] = {}
        self.tools['Project']['Edit project settings'] = ToolItem(
            'Edit project settings', icon=settings_icon, tool_tip='Edit the project settings', shortcut='')
        self.tools['Project']['Close project'] = ToolItem(
            'Close project', tool_tip='Close project', icon=close_icon)

    def get_tool_by_uuid(self, uuid):
        """Returns a tool from the project.

        Args:
            uuid (string): The unique identifier of the tool

        Returns:
            tool (ToolItem): The tool with the given uuid
        """
        for category in self.tools:
            result, item = self._get_tool_by_uuid_recursive(self.tools[category], uuid)
            if result:
                return True, item
        return False, None

    def _get_tool_by_uuid_recursive(self, tool_item, uuid):
        """Returns a tool from the project.

        Args:
            tool_item (ToolItem): The tool to search
            uuid (string): The unique identifier of the tool

        Returns:
            tool (ToolItem): The tool with the given uuid
        """
        if isinstance(tool_item, dict):
            for key in tool_item:
                result, item = self._get_tool_by_uuid_recursive(tool_item[key], uuid)
                if result:
                    return True, item
            return False, None
        if tool_item.uuid == uuid:
            return True, tool_item
        return False, None

    def get_selected_gui_tool(self):
        """Get the selected tool.

        Returns:
            ToolItem: ToolItem object
        """
        tool = 'select'  # Default state
        if 'Orientation' in self.tools:
            for key in self.tools['Orientation']:
                if self.tools['Orientation'][key].is_checked:
                    tool = key.lower()
                    break
        return tool

    def verify_layout(self, layout, num_docs):
        """Verify the layout of the tools."""
        if 'Windows' in self.tools:
            if layout in self.tools['Windows']['main']:
                if 0 < num_docs <= 6:
                    for key in self.tools['Windows']['main']:
                        self.tools['Windows']['main'][key].is_checked = None
                        self.tools['Windows']['main'][key].is_enabled = True
                return True, layout
            elif 1 < num_docs <= 6 and layout in self.tools['Windows'][num_docs]:
                if not self.tools['Windows'][num_docs][layout].is_checked:
                    for key in self.tools['Windows'][num_docs]:
                        self.tools['Windows'][num_docs][key].is_checked = None
                        self.tools['Windows'][num_docs][key].is_enabled = True
                    self.tools['Windows'][num_docs][layout].is_checked = True
                    self.tools['Windows'][num_docs][layout].is_enabled = False
                return True, layout
            else:
                # The layout is invalid, and we need to sete a default
                return False, 'tile'
        return False, 'maximize all'

    def update_tools(self, projects_manager, command_manager, file_manager, undo_max_history, open_file_func, app_data):
        """Update the tool manager.

        Args:
            projects_manager (ProjectsManager): The projects manager
            command_manager (CommandManager): The command manager
            file_manager (FileManager): The file manager
            undo_max_history (int): The maximum number of undo history items to display
            open_file_func (function): Function to open a file
            app_data (AppData): The application data object
        """
        # This is run every time the GUI is updated
        selected_items = projects_manager.get_selected_items_uuid_list()
        # tm = self.tool_manager

        if len(selected_items) == 0:
            self.tools['File']['Save as'].is_enabled = False
            # edit (calc and metadata)
            self.tools['Calculators']['Edit Calculator'].is_enabled = False
            self.tools['Calculators']['Edit metadata'].is_enabled = False
            # Rename
            self.tools['Edit']['Rename'].is_enabled = False
            # delete
            self.tools['Calculators']['Delete Calculator'].is_enabled = False
            # tm.tools['Edit']['Delete'].is_enabled = False
            # copy/duplicate
            self.tools['Calculators']['Duplicate Calculator'].is_enabled = False
        elif len(selected_items) == 1:
            sel_items, _ = projects_manager.get_items_with_uuids([selected_items[0]])
            sel_item = sel_items[0]
            enable_delete = sel_item.class_name != 'Project'
            self.tools['File']['Save as'].is_enabled = True
            # edit (calc and metadata)
            self.tools['Calculators']['Edit Calculator'].is_enabled = True
            self.tools['Calculators']['Edit metadata'].is_enabled = True
            self.tools['Calculators']['Edit Calculator'].item_uuid = selected_items[0]
            self.tools['Calculators']['Edit metadata'].item_uuid = selected_items[0]
            # Rename
            self.tools['Edit']['Rename'].is_enabled = True
            self.tools['Edit']['Rename'].item_uuid = selected_items[0]
            # delete
            self.tools['Calculators']['Delete Calculator'].is_enabled = enable_delete
            self.tools['Calculators']['Delete Calculator'].name = 'Delete Calculator'
            self.tools['Calculators']['Delete Calculator'].item_uuid = selected_items[0]
            # tm.tools['Edit']['Delete'].is_enabled = True
            # copy/duplicate
            self.tools['Calculators']['Duplicate Calculator'].is_enabled = True
            self.tools['Calculators']['Duplicate Calculator'].name = 'Duplicate Calculator'
            self.tools['Calculators']['Duplicate Calculator'].item_uuid = selected_items[0]
        else:  # Multiple selections
            self.tools['File']['Save as'].is_enabled = False
            # edit (calc and metadata)
            self.tools['Calculators']['Edit Calculator'].is_enabled = False
            self.tools['Calculators']['Edit metadata'].is_enabled = False
            # Rename
            self.tools['Edit']['Rename'].is_enabled = False
            # delete
            self.tools['Calculators']['Delete Calculator'].is_enabled = True
            self.tools['Calculators']['Delete Calculator'].name = 'Delete calculators'
            self.tools['Calculators']['Delete Calculator'].item_uuid = selected_items
            # tm.tools['Edit']['Delete'].is_enabled = True
            # copy/duplicate
            self.tools['Calculators']['Duplicate Calculator'].is_enabled = True
            self.tools['Calculators']['Duplicate Calculator'].name = 'Duplicate calculators'
            self.tools['Calculators']['Duplicate Calculator'].item_uuid = selected_items

        # If there is only one project, we can assume those tools to use the only project
        if len(projects_manager.get_projects_list()) == 1:
            self.tools['File']['Save as'].is_enabled = True

        # Windows
        if app_data.graphics_view is not None:
            if len(app_data.doc_windows) <= 1:
                self.tools['Display']['Close document window'].is_enabled = False
            else:
                self.tools['Display']['Close document window'].is_enabled = True

        # Profiles
        # profile_names = app_data.get_profile_names()
        app_data.update_profile_names()
        # self.tools['Profiles']['Profiles'].value_options = profile_names
        # self.tools['Profiles']['Profiles'].var.value = app_data.selected_profile_settings
        self.tools['Profiles']['Profiles'].var = app_data.profile_var

        if len(app_data.profile_settings_list) == 0:
            raise ValueError('No profiles found.')
        if app_data.profile_settings_list[app_data.selected_profile_settings].read_only:
            self.tools['Profiles']['Delete'].is_enabled = False
            self.tools['Profiles']['Export profile file'].is_enabled = False
        else:
            self.tools['Profiles']['Delete'].is_enabled = True
            self.tools['Profiles']['Export profile file'].is_enabled = True
        # self.tools['Profiles']['Edit'].item_uuid = app_data.profile_settings_list[
        #     app_data.selected_profile_settings].uuid

        # paste (if something is in the clipboard)
        # Handled in the GUI, as it depends on QApplication

        # Undo / Redo based on history...
        undo_list = command_manager.get_undo_list()
        redo_list = command_manager.get_redo_list()
        if len(undo_list) > 0:
            self.tools['Edit']['Undo'].is_enabled = True
            self.tools['Edit']['Undo'].name = f'Undo ({len(undo_list)})'

            self.tools['Edit']['Undo history'] = {}
            for index, undo in enumerate(undo_list):
                if index >= undo_max_history:
                    break
                self.tools['Edit']['Undo history'][index] = copy.copy(self.tools['Edit']['Undo'])
                self.tools['Edit']['Undo history'][index].name = undo.name
                self.tools['Edit']['Undo history'][index].uuid = undo.uuid
        else:
            self.tools['Edit']['Undo'].is_enabled = False
            self.tools['Edit']['Undo'].name = 'Undo'
            self.tools['Edit']['Undo history'] = {}

        if len(redo_list) > 0:
            self.tools['Edit']['Redo'].is_enabled = True
            self.tools['Edit']['Redo'].name = f'Redo ({len(redo_list)})'

            self.tools['Edit']['Redo history'] = {}
            for index, redo in enumerate(redo_list):
                if index >= undo_max_history:
                    break
                self.tools['Edit']['Redo history'][index] = copy.copy(self.tools['Edit']['Undo'])
                self.tools['Edit']['Redo history'][index].name = redo.name
                self.tools['Edit']['Redo history'][index].uuid = redo.uuid
        else:
            self.tools['Edit']['Redo'].is_enabled = False
            self.tools['Edit']['Redo'].name = 'Redo'
            self.tools['Edit']['Redo history'] = {}

        # Recent files
        file_list = file_manager.get_recent_file_dict()
        for file, file_uuid in file_list.items():
            self.tools['File']['Recent'][file] = copy.copy(self.tools['File']['Open'])
            self.tools['File']['Recent'][file].name = file
            self.tools['File']['Recent'][file].command = open_file_func
            # self.tools['File']['Recent'][file].args = (file,)
            self.tools['File']['Recent'][file].set_args_and_kwargs(file)
            self.tools['File']['Recent'][file].icon = None
            self.tools['File']['Recent'][file].shortcut = None
            self.tools['File']['Recent'][file].tool_tip = file
            self.tools['File']['Recent'][file].uuid = file_uuid
