"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionConcentrationCalc:
    """Perform concentration unit conversion computations."""
    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Concentration
    # ----------------------------------------------------------------------
    ppthousand = ['parts per thousand', 'ppt']
    ppm = ['parts per million', 'ppm']

    def __init__(self):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.ppthousand:
            return True, 'ppt'
        elif from_unit in self.ppm:
            return True, 'ppm'
        return False, ''

    # ----------------------------------------------------------------------
    # Concentration
    # ----------------------------------------------------------------------
    def convert_units(self, from_unit, to_unit, value):
        """Convert concentration.

        Args:
            from_unit (string): unit that value is in currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if from_unit in self.ppthousand:
            return self.convert_from_ppthousand(to_unit, value)
        elif from_unit in self.ppm:
            return self.convert_from_ppm(to_unit, value)

        return False, value

    def convert_from_ppthousand(self, to_unit, value):
        """Convert from the ppt unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.ppthousand:
            return True, value
        elif to_unit in self.ppm:
            return True, value * 1000.0

        else:
            return False, value

    def convert_from_ppm(self, to_unit, value):
        """Convert from the ppm unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.ppthousand:
            return True, value / 1000.0
        elif to_unit in self.ppm:
            return True, value

        else:
            return False, value
