"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionForceCalc:
    """Perform force unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Force
    # ----------------------------------------------------------------------
    lbf = ['lb', 'lbf', 'pound-force', 'pound force', 'pound_force', 'pound', 'POUND', 'pounds', 'POUNDS', 'poundal',
           'POUNDAL', 'poundals', 'POUNDALS']
    newton = ['N', 'Newton', 'NEWTON', 'Newtons', 'NEWTONS']

    def __init__(self):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.lbf:
            return True, 'N'
        return False, ''

    # ----------------------------------------------------------------------
    # Energy
    # ----------------------------------------------------------------------
    def convert_units(self, from_unit, to_unit, value):
        """Convert energy.

        Args:
            from_unit (string): unit that value is in currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if from_unit in self.lbf:
            return self.convert_from_lbf(to_unit, value)
        elif from_unit in self.newton:
            return self.convert_from_newton(to_unit, value)

        return False, value

    def convert_from_lbf(self, to_unit, value):
        """Convert from the pound-force unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.lbf:
            return True, value
        elif to_unit in self.newton:
            return True, value * 4.44822

        else:
            return False, value

    def convert_from_newton(self, to_unit, value):
        """Convert from the newton unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.lbf:
            return True, value * 0.224809
        elif to_unit in self.newton:
            return True, value

        else:
            return False, value
