"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionPressureCalc:
    """Perform pressure unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Pressure
    # ----------------------------------------------------------------------
    psi = ['psi', 'Psi', 'PSI', 'lb/in^2', 'pounds per square inch', 'Pounds Per Square Inch']
    psf = ['psf', 'Psf', 'PSF', 'lb/ft^2', 'pounds per square foot', 'Pounds Per Square Foot']
    atm = ['atm', 'Atm', 'ATM', 'atmosphere', 'Atmosphere', 'ATMOSPHERE', 'standard atmosphere', 'Standard Atmosphere']
    bar = ['bar', 'Bar', 'BAR']
    pa = ['pa', 'pascal', 'Pascal', 'PASCAL']

    def __init__(self):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.bar:
            return True, 'bar'
        elif from_unit in self.psi or from_unit in self.psf:
            return True, 'pa'
        elif from_unit in self.atm:
            return True, 'atm'
        return False, ''

    # ----------------------------------------------------------------------
    # Pressure
    # ----------------------------------------------------------------------
    # US Units
    def convert_units(self, from_unit, to_unit, value):
        """Convert pressure.

        Args:
            from_unit (string): unit that value is in currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if from_unit in self.pa:
            return self.convert_from_pa(to_unit, value)
        elif from_unit in self.bar:
            return self.convert_from_bar(to_unit, value)
        elif from_unit in self.psi:
            return self.convert_from_psi(to_unit, value)
        elif from_unit in self.psf:
            return self.convert_from_psf(to_unit, value)
        elif from_unit in self.atm:
            return self.convert_from_atm(to_unit, value)

        return False, value

    def convert_from_pa(self, to_unit, value):
        """Convert from the pascal unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.pa:
            return True, value
        elif to_unit in self.bar:
            return True, value * 1e-5
        elif to_unit in self.psi:
            return True, value * 0.000145038
        elif to_unit in self.psf:
            return True, value * 0.020885401
        elif to_unit in self.atm:
            return True, value * 9.869250513319517e-6

        else:
            return False, value

    def convert_from_bar(self, to_unit, value):
        """Convert from the bar unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.pa:
            return True, value * 100000.18082621002395
        elif to_unit in self.bar:
            return True, value
        elif to_unit in self.psi:
            return True, value * 14.503799999999511527
        elif to_unit in self.psf:
            return True, value * 2088.543423315753
        elif to_unit in self.atm:
            return True, value * 0.98692505133195185252

        else:
            return False, value

    def convert_from_psi(self, to_unit, value):
        """Convert from the psi unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.pa:
            return True, value * 6894.76
        elif to_unit in self.bar:
            return True, value / 14.503773773
        elif to_unit in self.psi:
            return True, value
        elif to_unit in self.psf:
            return True, value * 144.0
        elif to_unit in self.atm:
            return True, value / 14.695948775513

        else:
            return False, value

    def convert_from_psf(self, to_unit, value):
        """Convert from the psf unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.pa:
            return True, value * 47.880258888889
        elif to_unit in self.bar:
            return True, value / 2088.543423315753
        elif to_unit in self.psi:
            return True, value / 144.0
        elif to_unit in self.psf:
            return True, value
        elif to_unit in self.atm:
            return True, value / 2116.21662367394

        else:
            return False, value

    def convert_from_atm(self, to_unit, value):
        """Convert from the atmosphere unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.pa:
            return True, value * 101325.0
        elif to_unit in self.bar:
            return True, value * 1.01325
        elif to_unit in self.psi:
            return True, value * 14.695948775513
        elif to_unit in self.psf:
            return True, value * 2116.21662367394
        elif to_unit in self.atm:
            return True, value

        else:
            return False, value
