"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.units.length import ConversionLengthCalc


class ConversionWeirCalc():
    """Perform weir unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Weir coefficient
    # ----------------------------------------------------------------------
    weir_us = ['weir US Units', 'weir US']
    weir_si = ['weir SI Units', 'weir SI']

    def __init__(self, g=32.2):
        """Initialize the ConversionCalc Class.
        """
        self.g_us = g

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.weir_us:
            return True, 'weir SI'
        return False, ''

    # ----------------------------------------------------------------------
    # Weir coefficient
    # ----------------------------------------------------------------------
    def convert_units(self, from_unit, to_unit, value):
        """Convert weir units.

        Args:
            from_unit (string): unit that value is in currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        # This conversion is given in HDS 4 "Introduction to Highway Hydraulics on page 40 (54 of pdf).  The coefficient
        # has units of (g)^(1/2). The conversion factor is (9.81)^(0.5)/(32.2)^(0.5) = 0.552. Previously used values are
        # still given here, but updated (more precision) values are now used.

        # TODO: Consider getting the gravity from the settings
        # if self.app_data is not None:
        #     _, g_us = self.app_data.get_setting('Gravity', 32.2)
        # Actual Units: 'ft/s^2', 'm/s^2'
        length_convert = ConversionLengthCalc()
        _, g_si = length_convert.convert_from_foot('m', self.g_us)

        if from_unit in self.weir_us:
            return self.convert_from_weir_us(to_unit, value, self.g_us, g_si)
        elif from_unit in self.weir_si:
            return self.convert_from_weir_si(to_unit, value, self.g_us, g_si)

        return False, value

    def convert_from_weir_us(self, to_unit, value, g_us, g_si):
        """Convert from the weir US unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert
            g_us (float): g with foot reference
            g_si (float): g with meter reference

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.weir_us:
            return True, value
        elif to_unit in self.weir_si:
            return True, value * g_si ** 0.5 / (g_us ** 0.5)

        else:
            return False, value

    def convert_from_weir_si(self, to_unit, value, g_us, g_si):
        """Convert from the weir US unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert
            g_us (float): g with foot reference
            g_si (float): g with meter reference

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.weir_us:
            return True, value * g_us ** 0.5 / (g_si ** 0.5)
        elif to_unit in self.weir_si:
            return True, value

        else:
            return False, value
