"""Classes to handle the GUI side of variables."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import sys

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.app_data.app_data import AppData
from xms.FhwaVariable.core_data.units.unit_conversion import ConversionCalc
from xms.FhwaVariable.core_data.variables.variable import Variable
from xms.FhwaVariable.interface_adapters.view_model.dialog.table import Cell, Table


col_name = 0
col_val = 1
col_unit = 2
col_note = 3
col_type = 4
col_key = 5
col_uuid = 6


class TablePresenter:
    """Provides a class that will store a variable, handle interface, and file IO."""

    def __init__(self, app_data):
        """Initialize the VariableGui Class.

        Args:
            app_data: Settings CalcData
        """
        self.app_data = AppData()
        self.theme = app_data.get_theme()

        self.current_list_index = 0
        self.current_row_is_header = False

        _, self.use_sig_figures = app_data.get_setting('Use significant figures', True)

    def setup_table(self, tab_name, table_is_input, calcdata, variable_list, table_is_vertical=False):
        """Set up the controls for the dialog.

        Args:
            tab_name (string): the name of the tab
            table_is_input (bool): identifies if we are making an input table or result table
            calcdata: the CalcData object
            variable_list (dict): list of variables to define the table controls
            table_is_vertical (bool): identifies if the table is vertical

        Returns:
            Table: the table object
        """
        _, self.use_sig_figures = self.app_data.get_setting('Use significant figures', True)

        table = Table()
        key_string = tab_name.replace(" ", "_")

        row = 0

        # found, complexity_var = self.app_data.get_setting_var('Complexity')
        # if found:
        #     sys_complexity = complexity_var.value
        # else:
        #     sys_complexity = 0
        sys_complexity = calcdata.get_system_complexity_index()
        table.sys_complexity = sys_complexity

        # Determine our number of columns and setup the index to the columns
        num_cols = self._determine_num_columns(variable_list, sys_complexity)

        new_cols = num_cols - (col_uuid + 1)

        table.table_is_vertical = table_is_vertical
        table.col_name = col_name
        table.col_val = col_val
        table.col_unit = col_unit + new_cols
        table.col_note = col_note + new_cols
        table.col_type = col_type + new_cols
        table.col_key = col_key + new_cols
        table.col_uuid = col_uuid + new_cols
        if table_is_vertical:  # We want to display the units first
            table.col_unit = col_val
            table.col_val = col_unit

        num_rows = 0
        table.set_num_rows(num_rows)
        table.set_num_cols(num_cols)

        # Setup the units converter
        unit_converter = ConversionCalc(self.app_data, model_name=calcdata.model_name,
                                        project_uuid=calcdata.project_uuid)
        _, unit_system = self.app_data.get_setting('Selected unit system')

        name = ""
        recursion_level = 0

        # Set the table with data in variable list
        if len(variable_list) > 0:
            _, table = self._setup_table_with_variable_list_recursive(
                table_is_input, table, variable_list, sys_complexity, unit_converter, unit_system, row, name,
                key_string, recursion_level, calcdata.unknown)

        # If there is no data and this is a result table, show a message that we are waiting for data for computation
        elif table_is_input is False:
            # table.setRowCount(1)
            self.set_row_as_header_or_subheader(table, 0, "No data available for results", recursion_level)

        if table.max_cols > table.num_columns:
            text_color = self.theme["Table inactive text color"]
            bg_color = self.theme["Table inactive background color"]
            cell = Cell(cell_type='string', value='', is_read_only=True, bg_color=bg_color, text_color=text_color)
            val_cells = [cell for _ in range(table.max_cols - table.num_columns)]
            change = table.max_cols - table.num_columns
            for i in range(len(table.rows)):
                if len(table.rows[i].value_cells) == 0:
                    table.rows[i].value_cells = val_cells
                elif len(table.rows[i].value_cells) < change:
                    for _ in range(len(table.rows[i].value_cells), change):
                        table.rows[i].value_cells.append(cell)

        # Set the header labels
        header_labels = ['Name']
        if table_is_vertical:
            header_labels.append('Unit')
        if new_cols > 0:
            for i in range(0, new_cols + 1):
                header_labels.append(f'Value {i + 1}')
        else:
            header_labels.append("Value")
        if not table_is_vertical:
            header_labels.append('Unit')
        header_labels.append('Note')
        header_labels.append('Type')
        header_labels.append('Key')
        header_labels.append('UUID')

        table.header_labels = header_labels
        # if not table_is_vertical:
        #     table.setHorizontalHeaderLabels(header_labels)
        # else:
        #     table.setVerticalHeaderLabels(header_labels)

        # Set table colors, based on theme
        table.bg_color = self.theme["Table title background color"]
        table.text_color = self.theme["Table title text color"]
        # bg_color = self.rgb_to_hex(self.theme["Table title background color"])
        # color = self.rgb_to_hex(self.theme["Table title text color"])
        # table.setStyleSheet(
        #     f'QHeaderView::section {{ background-color: {bg_color}; '
        #     f'color: {color} }}'
        # )

        # Hide the key column (only used for internal reasons (to identify the variable on a change event))
        # if not table_is_vertical:
        #     table.hideColumn(table.columnCount() - 2)  # Key column
        #     table.hideColumn(table.columnCount() - 1)  # UUID column  # col_uuid
        # else:
        #     table.hideRow(table.rowCount() - 2)  # Key column
        #     table.hideRow(table.rowCount() - 1)  # UUID column  # col_uuid

        # self._expand_table_to_fill_space(table)

        return table

    def _determine_num_columns(self, variable_list: dict, sys_complexity: int) -> int:
        """Determine the number of columns in the table.

        Args:
            variable_list (dict): list of variables to define the table controls
            sys_complexity (int): the complexity of the system
        """
        num_cols = col_uuid + 1
        num_cols = self._determine_num_columns_recursive(variable_list, sys_complexity)
        return num_cols

    def _determine_num_columns_recursive(self, variable_list: dict, sys_complexity: int) -> int:
        """Determine the number of columns in the table.

        Args:
            variable_list (dict): list of variables to define the table controls
            sys_complexity (int): the complexity of the system
        """
        num_cols = col_uuid + 1
        for _, cur_variable in variable_list.items():
            if isinstance(cur_variable, dict):
                num_cols = self._determine_num_columns_recursive(cur_variable, sys_complexity)
            elif cur_variable.complexity > sys_complexity:
                continue
            elif cur_variable.type in ['float_list']:
                if num_cols < cur_variable.value + col_uuid:
                    num_cols = cur_variable.value + col_uuid
            elif cur_variable.type in ['class', 'UserArray', 'calc_list']:
                cols = self._determine_num_columns_recursive(cur_variable.value.get_input_group(), sys_complexity)
                if cols > num_cols:
                    num_cols = cols

        return num_cols

    def _setup_table_with_variable_list_recursive(self, table_is_input, table, variable_list, sys_complexity,
                                                  unit_converter, unit_system, row, name, key, recursion_level,
                                                  unknown):
        """Set up the table with a given variable list, calls itself as needed.

        Args:
            table_is_input (bool): identifies if we are making an input table or result table
            table (Table): table object that we modify and return
            variable_list (dict): list of variables to define the table controls
            sys_complexity: the complexity of the system
            unit_converter: class used to provide unit conversion for the interface
            unit_system (string): Tells the row which unit system to provide for the user
            row (int): current row for the function (needed for recursion)
            name (int): used in recursion to set headers
            key (string): used in recursion to determine variables that are modified
            recursion_level (int): number of indentations for row; each recursion is an indentation
            unknown (string or None): unknown variable for CalcData (used by CalcData, but accessed recursively)

        Returns:
            row (int): the updated row number
            table (Table): the updated table object
        """
        num_rows = table.get_num_rows()
        num_vars = 0

        for _, cur_variable in variable_list.items():
            if isinstance(cur_variable, dict):
                num_vars += 1
            elif cur_variable.complexity <= sys_complexity:
                # skip
                num_vars += 1

        num_rows += num_vars
        table.set_num_rows(num_rows)

        _, unit_system = self.app_data.get_setting('Selected unit system')

        current_list_index = 0
        self.current_row_is_header = False

        if recursion_level > 0:
            # Set Header and Sub-headers
            row = self.set_row_as_header_or_subheader(table, row, name, recursion_level)

        # Determine max columns
        cols = self.determine_max_columns(variable_list, sys_complexity)
        table.set_max_cols(cols + 6)  # 7 is the normal number of columns, but we use one col (value)

        for var_key, cur_variable in variable_list.items():
            # Determine Key
            key_string = var_key.replace(" ", "_")
            new_key = key
            if len(new_key) > 0:
                new_key += " "
            new_key += key_string

            if isinstance(cur_variable, dict):
                # This is a sub-group
                name = var_key
                tokens = var_key.split(' ', 3)
                if tokens[0] == 'CalcOrVarlist':
                    name = tokens[-1]
                row, table = self._setup_table_with_variable_list_recursive(
                    table_is_input, table, cur_variable, sys_complexity, unit_converter, unit_system, row, name,
                    new_key, recursion_level + 1, unknown)
                continue

            # check if the complexity means we should skip this item
            if cur_variable.complexity > sys_complexity:
                continue

            current_list_index = self.determine_if_header(var_key, cur_variable, current_list_index)

            # Get the precision and units
            _, additional_zeros = self.app_data.get_setting('Additional significant figures')
            precision = cur_variable.precision + additional_zeros
            selected_unit = cur_variable.get_selected_unit(unit_system)

            # Determine limits
            limit_txt, lower_limit, upper_limit = self._determine_limits(cur_variable, unit_converter, unit_system,
                                                                         table_is_input, precision)

            # Name
            self._setup_name_column(cur_variable, table, row, recursion_level, limit_txt)

            # Variable
            row, continue_ = self._setup_value_column(
                table, row, cur_variable, sys_complexity, unit_converter, unit_system, selected_unit, precision,
                recursion_level, unknown, table_is_input, lower_limit, upper_limit, new_key)
            if continue_:
                continue

            # Units
            self._setup_units_column(cur_variable, table, row, unit_system, table_is_input,
                                     recursion_level)

            # Note
            self._setup_notes_column(table, row, cur_variable, recursion_level)

            # Type
            self._setup_type_column(table, row, cur_variable, recursion_level)

            # Key
            self._setup_key_column(table, row, cur_variable, new_key, recursion_level)

            # Uuid
            self._setup_uuid_column(table, row, cur_variable, recursion_level)

            row += 1
        return row, table

    def determine_max_columns(self, variable_list: dict, sys_complexity: int) -> int:
        """Determine the maximum number of columns in the table.

        Args:
            variable_list (dict): list of variables to define the table controls
            sys_complexity (int): the complexity of the system
        """
        max_cols = col_uuid + 1
        for _, cur_dict_var in variable_list.items():
            if isinstance(cur_dict_var, dict):
                cols = self.determine_max_columns(cur_dict_var, sys_complexity)
                if cols > max_cols:
                    max_cols = cols
            elif cur_dict_var.complexity > sys_complexity:
                pass
            elif cur_dict_var.type in ['float_list']:
                if max_cols < cur_dict_var.value + col_uuid:
                    max_cols = cur_dict_var.value + col_uuid
            elif cur_dict_var.type in ['class', 'UserArray', 'calc_list']:
                cols = self.determine_max_columns(cur_dict_var.value.get_input_group(), sys_complexity)
                if cols > max_cols:
                    max_cols = cols

        return max_cols

    def determine_if_header(self, var_key, cur_variable, current_list_index):
        """Determine if the current variable is a header or sub-header.

        Args:
            var_key: the key of the variable
            cur_variable: the variable that we are setting up
            current_list_index: the current index of the list
        """
        self.current_row_is_header = False
        if hasattr(cur_variable, 'set_row_as_header') and cur_variable.set_row_as_header:
            self.current_row_is_header = True
        else:
            tokens = var_key.split(' ')
            if tokens[0] == 'CalcOrVarlist':
                try:
                    index = int(tokens[2])
                    if index == current_list_index:
                        self.current_row_is_header = True
                        current_list_index += 1
                except ValueError:
                    pass
            else:
                current_list_index = 0
        return current_list_index

    def _determine_limits(self, cur_variable, unit_converter, unit_system, table_is_input, precision):
        """Determine the limits for a given variable.

        Args:
            cur_variable: the variable that we are setting up
            unit_converter: class used to provide unit conversion for the interface
            unit_system (string): Tells the row which unit system to provide for the user
            table_is_input (bool): identifies if we are making an input table or result table
            precision: the precision to use for the value

        Returns:
            limit_txt (string): the text to display for the limits
            lower_limit: the lower limit of the variable
            upper_limit: the upper limit of the variable
        """
        limit_txt = ''
        lower_limit = None
        upper_limit = None
        if table_is_input and cur_variable.type in ['float', 'float_list', 'int']:
            lower_limit = cur_variable.limits[0]
            upper_limit = cur_variable.limits[-1]

            # Handle precision for rounding function
            limit_precision = precision
            if limit_precision == 0:
                limit_precision = None

            if lower_limit <= -sys.float_info.max:
                lower_limit = None
            if cur_variable.type in ['float', 'float_list'] and upper_limit >= sys.float_info.max:
                upper_limit = None
            if cur_variable.type in ['int'] and upper_limit >= sys.maxsize:
                upper_limit = None
            if lower_limit is not None or upper_limit is not None:
                _, lower_limit_converted = unit_converter.convert_units(cur_variable.native_unit,
                                                                        cur_variable.get_selected_unit(unit_system),
                                                                        lower_limit)
                _, upper_limit_converted = unit_converter.convert_units(cur_variable.native_unit,
                                                                        cur_variable.get_selected_unit(unit_system),
                                                                        upper_limit)

                if lower_limit is not None and upper_limit is not None:
                    limit_txt = f' [{round(lower_limit_converted, limit_precision)} - ' \
                        f'{round(upper_limit_converted, limit_precision)}]'
                elif lower_limit is not None:
                    limit_txt = f' [ ≥ {round(lower_limit_converted, limit_precision)}]'
                elif upper_limit is not None:
                    limit_txt = f' [ ≤ {round(upper_limit_converted, limit_precision)}]'
        return limit_txt, lower_limit, upper_limit

    # def rgb_to_hex(self, rgb_tuple):
    #     """Convert an RGB tuple to a hexadecimal color string."""
    #     if not isinstance(rgb_tuple, tuple) or len(rgb_tuple) != 3:
    #         return None
    #     return "#{:02x}{:02x}{:02x}".format(rgb_tuple[0], rgb_tuple[1], rgb_tuple[2])

    def _get_background_text_color(self, cur_variable, recursion_level, is_editable_data=False, table_is_input=True):
        """Get the background and text color for a given variable.

        Args:
            cur_variable: the variable that we are setting up
            recursion_level (int): number of indentations for row; each recursion is an indentation
            is_editable_data (bool): identifies if the data is editable
            table_is_input (bool): identifies if we are making an input table or result table
        """
        background_color = self.theme['Table inactive background color']
        text_color = self.theme['Table inactive text color']
        if self.current_row_is_header:
            if recursion_level == 0:
                background_color = self.theme['Table header background color']
                text_color = self.theme['Table header text color']
            else:
                background_color = self.theme['Table subheader background color']
                text_color = self.theme['Table subheader text color']
        elif is_editable_data:
            if not table_is_input:
                background_color = self.theme['Table inactive background color']
                text_color = self.theme['Table inactive text color']
            elif cur_variable.read_only:
                background_color = self.theme['Table read-only background color']
                text_color = self.theme['Table read-only text color']
            else:
                background_color = self.theme['Table editable background color']
                text_color = self.theme['Table editable text color']

        if cur_variable.data_status == 'approved':
            background_color = self.theme['Success background color']
            text_color = self.theme['Success text color']
        elif cur_variable.data_status == 'incomplete':
            background_color = self.theme['Incomplete background color']
            text_color = self.theme['Incomplete text color']
        elif cur_variable.data_status == 'warning':
            background_color = self.theme['Warning background color']
            text_color = self.theme['Warning text color']
        elif cur_variable.data_status == 'error':
            background_color = self.theme['Failure background color']
            text_color = self.theme['Failure text color']

        return background_color, text_color

    def _setup_value_column(self, table, row, cur_variable, sys_complexity, unit_converter, unit_system,
                            selected_unit, precision, recursion_level=0, unknown=None,
                            table_is_input=True, lower_limit=None, upper_limit=None, new_key=""):
        """Set up the value column for a given variable.

        Args:
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            cur_variable: the variable that we are setting up
            unit_converter: class used to provide unit conversion for the interface
            unit_system (string): Tells the row which unit system to provide for the user
            selected_unit: the unit that is currently selected
            precision: the precision to use for the value

        Returns:
            row (int): the updated row number
            continue_ (bool): should we continue to the next variable
        """
        val = cur_variable.get_val()
        if cur_variable.type in ['class', 'UserArray', 'calc_list']:
            row = self._setup_value_with_class(cur_variable, table_is_input, table, sys_complexity, unit_converter,
                                               unit_system, row, new_key, recursion_level, unknown)
            return row, True

        elif cur_variable.type in ['calc', 'table']:
            self._setup_value_with_calc(cur_variable, table, row, table_is_input, recursion_level)

        elif cur_variable.type in ['float', 'int']:
            self._setup_value_with_number(cur_variable, table, row, table_is_input, upper_limit,
                                          lower_limit, val, unit_converter, selected_unit,
                                          precision, recursion_level=recursion_level)

        elif cur_variable.type in Variable.string_types or cur_variable.type == 'file':
            self._setup_value_with_str(cur_variable, table, row, table_is_input,
                                       val, recursion_level)

        elif cur_variable.type in ['list', 'uuid_dict']:
            self._setup_value_with_list(cur_variable, table, row, table_is_input,
                                        recursion_level=recursion_level)

        elif cur_variable.type == 'bool':
            self._setup_value_with_bool(cur_variable, table, row, table_is_input, val)

        elif cur_variable.type == 'float_list':
            self._setup_value_with_float_list(cur_variable, table, row, table_is_input,
                                              unit_converter, selected_unit, precision,
                                              lower_limit, upper_limit, recursion_level=recursion_level)

        elif cur_variable.type == 'string_list':
            self._setup_value_with_string_list(cur_variable, table, row, table_is_input, val,
                                               recursion_level)

        elif cur_variable.type in ['action', 'image']:
            self._setup_value_with_button(cur_variable, table, row, table_is_input, recursion_level)

        elif cur_variable.type == 'color':
            self._setup_value_with_color(cur_variable, table, row, table_is_input, recursion_level)

        else:  # 'date'
            self._setup_value_with_value(cur_variable, table, row, table_is_input, recursion_level)

        return row, False

    def _setup_value_with_class(self, cur_variable, table_is_input, table, sys_complexity, unit_converter, unit_system,
                                row, new_key, recursion_level, unknown):
        """Set up the value column for a given variable.

        Args:
            cur_variable: the variable that we are setting up
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            table_is_input (bool): identifies if we are making an input table or result table
            sys_complexity: the complexity of the system
            unit_converter: class used to provide unit conversion for the interface
            unit_system (string): Tells the row which unit system to provide for the user
            new_key: the key for the variable
            recursion_level (int): number of indentations for row; each recursion is an indentation
            unknown (string or None): unknown variable for CalcData (used by CalcData, but accessed recursively)
        """
        name = cur_variable.name
        new_level = recursion_level + 1
        new_var_list = cur_variable.get_val().get_input_group(unknown)

        if new_var_list is {}:
            return row

        row, table = self._setup_table_with_variable_list_recursive(
            table_is_input, table, new_var_list, sys_complexity, unit_converter, unit_system, row, name, new_key,
            new_level, unknown)
        return row

    def _setup_value_with_float_list(self, cur_variable, table, row, table_is_input, unit_converter, selected_unit,
                                     precision, lower_limit=None, upper_limit=None, recursion_level=0):
        """Set up the value column for a given variable.

        Args:
            cur_variable: the variable that we are setting up
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            table_is_input (bool): identifies if we are making an input table or result table
            unit_converter: class used to provide unit conversion for the interface
            selected_unit: the unit that is currently selected
            precision: the precision to use for the value
            lower_limit: the lower limit for the variable
            upper_limit: the upper limit for the variable
            recursion_level (int): number of indentations for row; each recursion is an indentation
        """
        size = min(len(cur_variable.value_options), cur_variable.value)
        # table.set_max_cols(size + 6)  # 7 is the normal number of columns, but we use one col (value)
        for index in range(size):
            value = cur_variable.value_options[index]
            self._setup_value_with_number(cur_variable, table, row, table_is_input, upper_limit,
                                          lower_limit, value, unit_converter, selected_unit,
                                          precision, index, recursion_level=recursion_level)

        # If our data is smaller than the max columns, fill in the rest with empty strings
        table_max = table.max_cols - 6
        if size > 0:
            background_color, text_color = self._get_background_text_color(
                cur_variable=cur_variable, recursion_level=recursion_level, is_editable_data=False,
                table_is_input=table_is_input)
            for index in range(size, table_max):
                cell = Cell(cell_type='string', value=str(''), is_read_only=True, bg_color=background_color,
                            text_color=text_color, variable_type=cur_variable.type)
                table.set_item(row, table.col_val + index, cell)

    # @staticmethod
    # def display_image(image):
    #     """Handle the click on Browse.

    #     Args:
    #         image (np.ndarray): the image to display
    #     """
    #     if image is None:
    #         return
    #     if not isinstance(image, np.ndarray):
    #         return

    #     cv2.imshow('Image', image)
    #     cv2.waitKey(0)
    #     cv2.destroyAllWindows()

    #     return

    # def _split_uuid_and_index(self, input_string):
    #     """Check for and split a string containing a UUID and an optional index.

    #     Args:
    #         input_string (str): The input string to check and split.

    #     Returns:
    #         tuple: A tuple containing the UUID and the index (or None if not present).
    #     """
    #     if ',' in input_string:
    #         uuid_str, index_str = input_string.split(',', 1)
    #         parsed_index = int(index_str)
    #         return uuid_str, parsed_index
    #     else:
    #         return input_string, None

    # def _get_uuid_from_table(self, table, row, col):
    #     """Get the var key from the table (from a hidden row).

    #     Args:
    #         table: QT table (so we can make changes on that table)
    #         row (int): current row that had a cell that was changed
    #         col (int): current col that had a cell that was changed

    #     Returns:
    #         row_uuid (UUID): the UUID of the variable that set the row
    #     """
    #     if row is None or col is None:
    #         return None
    #     if row >= table.rowCount() or col >= table.columnCount():
    #         return None
    #     if not table.table_is_vertical:
    #         uuid_col = table.columnCount() - 1
    #         uuid_str = table.item(row, uuid_col).text()
    #     else:
    #         uuid_col = table.rowCount() - 1
    #         uuid_str = table.item(uuid_col, col).text()
    #     if uuid_str == "":
    #         return None
    #     row_uuid, index = self._split_uuid_and_index(uuid_str)
    #     row_uuid = uuid.UUID(row_uuid)
    #     return row_uuid, index

    # def table_changed(self, CalcData, table, row, col, new_val=None, add_undo=True):
    #     """Handle a table cell change event. a dialog should have a function that will call this function.

    #     Args:
    #         CalcData: a python class that has: get_input_group, get_results_group, check_warnings, get_can_compute,
    #             clear_results, compute_data, setup_plot_data. This class will manage lists of variables.
    #         table: QT table (so we can make changes on that table)
    #         row (int): current row that had a cell that was changed
    #         col (int): current col that had a cell that was changed
    #         new_val: the new value that was modified in the table
    #         add_undo (bool): Says whether we should add this to the undo history (False when we perform undo)
    #     """
    #     if row >= table.rowCount() or col >= table.columnCount():
    #         return False
    #     if new_val is None:
    #         widget = table.cellWidget(row, col)
    #         if not widget:
    #             widget = table.item(row, col)
    #         if hasattr(widget, 'text'):
    #             new_val = widget.text()
    #         elif hasattr(widget, 'currentText'):
    #             new_val = widget.currentText()
    #     row_uuid, uuid_index = self._get_uuid_from_table(table, row, col)
    #     if row_uuid is None:
    #         return False
    #     index = col - table.col_val
    #     is_unit_change = col == table.col_unit
    #     if table.table_is_vertical:
    #         index = row - table.col_val
    #         is_unit_change = row == table.col_unit
    #     CalcData.set_val_by_uuid(row_uuid, new_val, index=index, is_unit_change=is_unit_change,
    #                                add_undo=add_undo, uuid_index=uuid_index)
    #     if CalcData.type == 'AppSettingsData':
    #         self.app_data = CalcData
    #     return True

    def set_row_as_header_or_subheader(self, table, row, name, recursion_level):
        """Sets the format of a row to be a header or sub-header.

        Args:
            table: QT table
            row (int): the row that should be formatted as a header or sub-header
            name (string): The text that we want to show on the header/sub-header
            recursion_level (int): number of indentations for header
        """
        bg_color = self.theme['Table header background color']
        text_color = self.theme['Table header text color']
        if recursion_level > 1:
            bg_color = self.theme['Table subheader background color']
            text_color = self.theme['Table subheader text color']
        txt = "  " * (recursion_level - 1) + name
        table.add_row(row_index=row, name=txt, bg_color=bg_color, text_color=text_color, value_is_read_only=True)
        row += 1
        # # Set Header and Sub-headers
        # for col_index in range(table.col_name, table.col_note + 1):
        #     cell = QTableWidgetItem(txt)
        #     if recursion_level <= 1:
        #         cell.setBackground(QColor(self.rgb_to_hex(self.theme['Table header background color'])))
        #         cell.setForeground(QColor(self.rgb_to_hex(self.theme['Table header text color'])))
        #     else:
        #         cell.setBackground(QColor(self.rgb_to_hex(self.theme['Table subheader background color'])))
        #         cell.setForeground(QColor(self.rgb_to_hex(self.theme['Table subheader text color'])))
        #     cell.setFlags(cell.flags() ^ Qt.ItemIsEditable)
        #     if not table_is_vertical:
        #         table.setItem(row, col_index, cell)
        #     else:
        #         table.setItem(col_index, row, cell)
        #     txt = ""
        # row += 1
        return row

    # @staticmethod
    # def _expand_table_to_fill_space(table):
    #     """Expands the table to fill the space available.

    #     Args:
    #         table: QT table
    #     """
    #     table.resizeColumnsToContents()
    #     width_available = table.width()
    #     # Add the vertical header
    #     width_used = table.verticalHeader().width()
    #     # Add scroll bars
    #     if table.verticalScrollBar().isVisible():
    #         width_used += table.verticalScrollBar().width() + 1
    #     num_col = table.columnCount()
    #     for col_index in range(num_col):
    #         width_used += table.columnWidth(col_index)
    #     # If the table is less than 60% of the table, adjust all columns to be wider
    #     if width_used < width_available * 0.5:
    #         extra_width = (width_available * 0.5) / num_col
    #         for col_index in range(num_col):
    #             table.setColumnWidth(col_index, table.columnWidth(col_index) + extra_width)
    #         width_used += width_available * 0.4
    #     if width_used < width_available:
    #         # The last column is 'key'; we want to size the last VISIBLE column, notes
    #         note_index = num_col - 2
    #         current_width = table.columnWidth(note_index)
    #         table.setColumnWidth(note_index, current_width + width_available - width_used - 1)

    def _setup_type_column(self, table, row, cur_variable, recursion_level):
        """Set up the type column for a given variable.

        Args:
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            cur_variable: the variable that we are setting up
            recursion_level (int): number of indentations for row; each recursion is an indentation
        """
        background_color, text_color = self._get_background_text_color(cur_variable, recursion_level)
        cell = Cell(cell_type='string', value=cur_variable.type, is_read_only=True, bg_color=background_color,
                    text_color=text_color, variable_type=cur_variable.type)
        table.set_item(row, table.col_type, cell)

    def _setup_key_column(self, table, row, cur_variable, new_key, recursion_level):
        """Set up the key column for a given variable.

        Args:
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            cur_variable: the variable that we are setting up
            new_key: the key for the variable
            recursion_level (int): number of indentations for row; each recursion is an indentation
        """
        background_color, text_color = self._get_background_text_color(cur_variable, recursion_level)

        # key_cell = QTableWidgetItem(new_key)
        # key_cell.setFlags(key_cell.flags() ^ Qt.ItemIsEditable)
        # if not table_is_vertical:
        #     table.setItem(row, table.col_key, QTableWidgetItem(key_cell))
        # else:
        #     table.setItem(table.col_key, row, QTableWidgetItem(key_cell))

        cell = Cell(cell_type='string', value=new_key, is_read_only=True, bg_color=background_color,
                    text_color=text_color, variable_type=cur_variable.type)
        table.set_item(row, table.col_key, cell)

    def _setup_uuid_column(self, table, row, cur_variable, recursion_level):
        """Set up the UUID column for a given variable.

        Args:
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            cur_variable: the variable that we are setting up
            recursion_level (int): number of indentations for row; each recursion is an indentation
        """
        background_color, text_color = self._get_background_text_color(cur_variable, recursion_level)

        # uuid_cell = QTableWidgetItem(str(cur_variable.uuid))
        # uuid_cell.setFlags(uuid_cell.flags() ^ Qt.ItemIsEditable)
        # if not table_is_vertical:
        #     table.setItem(row, table.col_uuid, QTableWidgetItem(uuid_cell))
        # else:
        #     table.setItem(table.col_uuid, row, QTableWidgetItem(uuid_cell))

        cell = Cell(cell_type='string', value=str(cur_variable.uuid), is_read_only=True, bg_color=background_color,
                    text_color=text_color, variable_type=cur_variable.type)
        table.set_item(row, table.col_uuid, cell)

    def _setup_name_column(self, cur_variable, table, row, recursion_level, limit_txt):
        """Set up the name column for a given variable.

        Args:
            cur_variable: the variable that we are setting up
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            recursion_level (int): number of indentations for row; each recursion is an indentation
            limit_txt (string): the text to display for the limits

        Returns:
            name_cell: the cell that was created for the
        """
        background_color, text_color = self._get_background_text_color(cur_variable, recursion_level)

        name = "  " * recursion_level + cur_variable.name + limit_txt
        # name_cell = QTableWidgetItem(name)
        # name_cell.setBackground(QColor(background_color))
        # name_cell.setForeground(QColor(text_color))
        # name_cell.setFlags(name_cell.flags() ^ Qt.ItemIsEditable)
        # if not table_is_vertical:
        #     table.setItem(row, table.col_name, name_cell)
        # else:
        #     table.setItem(table.col_name, row, name_cell)
        cell = Cell(cell_type='string', value=name, is_read_only=True, bg_color=background_color,
                    text_color=text_color, variable_type=cur_variable.type)
        table.set_item(row, table.col_name, cell)

    def _setup_notes_column(self, table, row, cur_variable, recursion_level):
        """Set up the notes column for a given variable.

        Args:
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            cur_variable: the variable that we are setting up
            recursion_level (int): number of indentations for row; each recursion is an indentation
        """
        background_color, text_color = self._get_background_text_color(cur_variable, recursion_level)

        # note_cell = QTableWidgetItem(cur_variable.note)
        # note_cell.setBackground(QColor(background_color))
        # note_cell.setForeground(QColor(text_color))
        # note_cell.setFlags(note_cell.flags() ^ Qt.ItemIsEditable)
        # if not table_is_vertical:
        #     table.setItem(row, cur_col_note, note_cell)
        # else:
        #     table.setItem(cur_col_note, row, note_cell)

        cell = Cell(cell_type='string', value=cur_variable.note, is_read_only=True, bg_color=background_color,
                    text_color=text_color, variable_type=cur_variable.type)
        table.set_item(row, table.col_note, cell)

    def _setup_units_column(self, cur_variable, table, row, unit_system, table_is_input, recursion_level):
        """Set up the units column for a given variable.

        Args:
            cur_variable: the variable that we are setting up
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            unit_system (string): Tells the row which unit system to provide for the user
            table_is_input (bool): identifies if we are making an input table or result table
            recursion_level (int): number of indentations for row; each recursion is an indentation
        """
        background_color, text_color = self._get_background_text_color(cur_variable, recursion_level, False,
                                                                       table_is_input)

        units_list, index = cur_variable.get_units_list_and_index(unit_system)
        if len(units_list) and index is not None and index >= 0:
            # units_combo = QComboBox(table)
            # units_combo.addItems(units_list)
            # units_combo.setCurrentIndex(index)
            # if not table_is_vertical:
            #     units_combo.setProperty("row", row)
            #     units_combo.setProperty("col", table.col_unit)
            #     units_combo.currentTextChanged.connect(lambda txt, cur_row=row, cur_col=table.col_unit:
            #                                            combo_change_slot(cur_row, cur_col, txt))
            #     units_combo.installEventFilter(table)
            #     table.setCellWidget(row, cur_col_unit, units_combo)
            # else:
            #     units_combo.setProperty("row", table.col_unit)
            #     units_combo.setProperty("col", row)
            #     units_combo.currentTextChanged.connect(lambda txt, cur_row=table.col_unit, cur_col=row:
            #                                            combo_change_slot(cur_row, cur_col, txt))
            #     units_combo.installEventFilter(table)
            #     table.setCellWidget(cur_col_unit, row, units_combo)

            cell = Cell(cell_type='list', value=index, value_options=units_list, is_read_only=False,
                        bg_color=background_color, text_color=text_color, variable_type=cur_variable.type)
            table.set_item(row, table.col_unit, cell)
        else:
            if cur_variable.type in ['file', 'image']:  # Add a matplotlib hatch selection for images
                # button = QPushButton('Browse....')
                # # button.clicked.connect(partial(self.select_file_or_folder, cur_variable.file_mode,
                # # cur_variable.value_options, row, table.col_unit))
                # if not table_is_vertical:
                #     button.clicked.connect(lambda btn_bool=True, cur_row=row, cur_col=table.col_unit:
                #                            button_click_slot(btn_bool, cur_row, cur_col))
                #     table.setCellWidget(row, cur_col_unit, button)
                # else:
                #     button.clicked.connect(lambda btn_bool=True, cur_row=table.col_unit, cur_col=row:
                #                            button_click_slot(btn_bool, cur_row, cur_col))
                #     table.setCellWidget(cur_col_unit, row, button)
                is_read_only = cur_variable.read_only
                cell = Cell(cell_type='button', value='Browse...', is_read_only=is_read_only,
                            bg_color=background_color, text_color=text_color, variable_type=cur_variable.type)
                table.set_item(row, table.col_unit, cell)
            else:
                # blank_cell = QTableWidgetItem("")
                # blank_cell.setBackground(QColor(background_color))
                # blank_cell.setForeground(QColor(text_color))
                # blank_cell.setFlags(blank_cell.flags() ^ Qt.ItemIsEditable)
                # if not table_is_vertical:
                #     table.setItem(row, cur_col_unit, blank_cell)
                # else:
                #     table.setItem(cur_col_unit, row, blank_cell)

                cell = Cell(cell_type='string', value='', is_read_only=True, bg_color=background_color,
                            text_color=text_color, variable_type=cur_variable.type)
                table.set_item(row, table.col_unit, cell)

    def _setup_value_with_calc(self, cur_variable, table, row, table_is_input, recursion_level):
        """Set up the value column for a given variable.

        Args:
            cur_variable: the variable that we are setting up
            table: view model table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            table_is_input (bool): identifies if we are making an input table or result table
            recursion_level (int): number of indentations for row; each recursion is an indentation
        """
        background_color, text_color = self._get_background_text_color(cur_variable, recursion_level, False,
                                                                       table_is_input)

        # button = QPushButton('Define....')
        # if cur_variable.value is None:
        #     button.setDisabled(True)
        #     # button.setBackground(self.theme['Table read-only background color'])
        #     # button.setForeground(self.theme['Table read-only text color'])
        #     button.setStyleSheet(f'background-color: {background_color};'
        #                          f'color: {text_color} }}')
        # if not table_is_vertical:
        #     button.clicked.connect(lambda btn_bool=True, cur_row=row, cur_col=table.col_val:
        #                            button_click_slot(btn_bool, cur_row, cur_col))
        #     table.setCellWidget(row, table.col_val, button)
        # else:
        #     button.clicked.connect(lambda btn_bool=True, cur_row=table.col_val, cur_col=row:
        #                            button_click_slot(btn_bool, cur_row, cur_col))
        #     table.setCellWidget(table.col_val, row, button)

        is_read_only = False
        if cur_variable.value is None or cur_variable.read_only:
            is_read_only = True

        cell = Cell(cell_type='button', value='Define...', is_read_only=is_read_only, bg_color=background_color,
                    text_color=text_color, variable_type=cur_variable.type)
        table.set_item(row, table.col_val, cell)

    # @staticmethod
    # def _get_validator(cur_variable, table, lower_limit=None, upper_limit=None):
    #     """Get the validator for the variable.

    #     Args:
    #         cur_variable: the variable that we are setting up
    #         table: QT table (so we can make changes on that table)
    #         lower_limit: the lower limit for the variable
    #         upper_limit: the upper limit for the variable
    #     """
    #     # Setup the validator
    #     validator = None
    #     int_list = ['int']
    #     dbl_list = ['float', 'float_list']
    #     num_str_list = ['zip', 'phone']
    #     if cur_variable.type in int_list:
    #         validator = QIntValidator(table)
    #     elif cur_variable.type in dbl_list:
    #         validator = QDoubleValidator(table)
    #         validator.setNotation(QDoubleValidator.StandardNotation)
    #     if cur_variable.type in int_list and cur_variable.type in dbl_list:
    #         if lower_limit is not None:
    #             validator.setBottom(lower_limit)
    #         if upper_limit is not None:
    #             validator.setTop(upper_limit)
    #     if cur_variable.type in num_str_list:
    #         validator = DigitPunctuationValidator(table)
    #     return validator

    def _setup_value_with_number(self, cur_variable, table, row, table_is_input, upper_limit,
                                 lower_limit, val, unit_converter, selected_unit, precision,
                                 index=0, recursion_level=0):
        """Set up the value column for a given variable.

        Args:
            cur_variable: the variable that we are setting up
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            table_is_input (bool): identifies if we are making an input table or result table
            upper_limit: the upper limit for the variable
            lower_limit: the lower limit for the variable
            val: the value of the variable
            unit_converter: class used to provide unit conversion for the interface
            selected_unit: the unit that is currently selected
            precision: the precision to use for the value
            index (int): the index of the value in the list (if applicable)
            recursion_level (int): number of indentations for row; each recursion is an indentation
        """
        background_color, text_color = self._get_background_text_color(cur_variable, recursion_level, True,
                                                                       table_is_input)

        # Convert to display
        if cur_variable.type in ['float', 'float_list']:
            _, val = unit_converter.convert_units(cur_variable.native_unit, selected_unit, val)
            cell_type = 'float'
        else:  # int
            precision = 0
            cell_type = 'int'
        txt = f'{val}'
        if self.use_sig_figures or cur_variable.type == 'int':
            txt = f'{val:.{precision}f}'
            # TODO: Consider implementing scientific notation
            # txt = f'{val:.{precision}g}'

        # item = QLineEdit(table)
        # validator = self._get_validator(cur_variable, table, lower_limit, upper_limit)
        # if validator is not None:
        #     item.setValidator(validator)
        # item.setText(f'{val:.{precision}f}')
        # item.setStyleSheet(
        #     'QLineEdit::section { '
        #     f'background-color: {background_color};'
        #     f'color: {text_color} }}')

        # if not table_is_vertical:
        #     item.editingFinished.connect(lambda cur_row=row, cur_col=table.col_val + index:
        #                                  edit_change_slot(cur_row, cur_col))
        #     # Set the value
        #     table.setCellWidget(row, table.col_val + index, item)
        # else:
        #     item.editingFinished.connect(lambda cur_row=table.col_val + index, cur_col=row:
        #                                  edit_change_slot(cur_row, cur_col))
        #     # Set the value
        #     table.setCellWidget(table.col_val + index, row, item)

        is_read_only = not table_is_input or cur_variable.read_only
        cell = Cell(cell_type=cell_type, value=txt, is_read_only=is_read_only, bg_color=background_color,
                    text_color=text_color, variable_type=cur_variable.type)
        cell.lower_limit = lower_limit
        cell.upper_limit = upper_limit
        table.set_item(row, table.col_val + index, cell)

    def _setup_value_with_str(self, cur_variable, table, row, table_is_input, val, recursion_level):
        """Set up the value column for a given variable.

        Args:
            cur_variable: the variable that we are setting up
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            table_is_input (bool): identifies if we are making an input table or result table
            val: the value of the variable
            recursion_level (int): number of indentations for row; each recursion is an indentation
        """
        background_color, text_color = self._get_background_text_color(cur_variable, recursion_level, True,
                                                                       table_is_input)

        # # string_item = QTableWidgetItem(str(val))
        # string_item = QLineEdit()
        # string_item.setText(str(val))

        # validator = self._get_validator(cur_variable, table, None, None)
        # if validator is not None:
        #     string_item.setValidator(validator)

        # # if not table_is_input:
        # #     string_item.setBackground(self.theme['Table inactive background color'])
        # #     string_item.setForeground(self.theme['Table inactive text color'])
        # #     string_item.setFlags(string_item.flags() ^ Qt.ItemIsEditable)
        # # elif cur_variable.read_only:
        # #     string_item.setBackground(self.theme['Table read-only background color'])
        # #     string_item.setForeground(self.theme['Table read-only text color'])
        # #     string_item.setFlags(string_item.flags() ^ Qt.ItemIsEditable)
        # # else:
        # #     string_item.setBackground(self.theme['Table editable background color'])
        # #     string_item.setForeground(self.theme['Table editable text color'])
        # string_item.setStyleSheet(
        #     'QLineEdit::section { '
        #     f'background-color: {background_color};'
        #     f'color: {text_color} }}')
        # # if cur_variable.type == 'zip':
        # #     string_item.setValidator(QIntValidator(QRegularExpressionValidator(QRegularExpression("\\d{5}"))))
        # # elif cur_variable.type == 'phone':
        # #     string_item.setValidator(QIntValidator())
        # # elif cur_variable.type == 'email':
        # #     string_item.setValidator(QRegularExpressionValidator(QRegularExpression(
        # #         "[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\\.[A-Za-z]{2,4}")))
        # # elif cur_variable.type == 'url':
        # #     string_item.setValidator(QRegularExpressionValidator(QRegularExpression(
        # #         "(http|https)://[A-Za-z0-9.-]+\\.[A-Za-z]{2,4}(/\\S*)?")))
        # if not table_is_vertical:
        #     string_item.editingFinished.connect(lambda cur_row=row, cur_col=table.col_val:
        #                                         edit_change_slot(cur_row, cur_col))
        #     # table.setItem(row, table.col_val, string_item)
        #     table.setCellWidget(row, table.col_val, string_item)
        # else:
        #     string_item.editingFinished.connect(lambda cur_row=table.col_val, cur_col=row:
        #                                         edit_change_slot(cur_row, cur_col))
        #     table.setCellWidget(table.col_val, row, string_item)

        is_read_only = not table_is_input or cur_variable.read_only
        cell = Cell(cell_type='string', value=str(val), is_read_only=is_read_only, bg_color=background_color,
                    text_color=text_color, variable_type=cur_variable.type)
        table.set_item(row, table.col_val, cell)

    def _setup_value_with_list(self, cur_variable, table, row, table_is_input,
                               recursion_level):
        """Set up the value column for a given variable.

        Args:
            cur_variable: the variable that we are setting up
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            table_is_input (bool): identifies if we are making an input table or result table
            recursion_level (int): number of indentations for row; each recursion is an indentation
        """
        background_color, text_color = self._get_background_text_color(cur_variable, recursion_level, True,
                                                                       table_is_input)

        # val_combo = QComboBox(table)
        index = cur_variable.value
        val_list = []
        if cur_variable.type == 'uuid_dict':
            keys_as_list = list(cur_variable.value_options.keys())
            names_as_list = list(cur_variable.value_options.values())
            # val_combo.addItems(names_as_list)
            val_list = names_as_list
            if len(cur_variable.value_options) == 0 or cur_variable.value not in keys_as_list:
                index = -1
            else:
                index = keys_as_list.index(cur_variable.value)
        else:
            # val_combo.addItems(cur_variable.value_options)
            val_list = cur_variable.value_options
        # val_combo.installEventFilter(table)
        # val_combo.setCurrentIndex(index)
        # if not table_is_input:
        #     pass
        # if not table_is_vertical:
        #     val_combo.currentTextChanged.connect(lambda txt, cur_row=row, cur_col=table.col_val:
        #                                          combo_change_slot(cur_row, cur_col, txt))
        #     table.setCellWidget(row, table.col_val, val_combo)
        # else:
        #     val_combo.currentTextChanged.connect(lambda txt, cur_row=table.col_val, cur_col=row:
        #                                          combo_change_slot(cur_row, cur_col, txt))
        #     table.setCellWidget(table.col_val, row, val_combo)
        is_read_only = not table_is_input or cur_variable.read_only
        cell = Cell(cell_type='list', value=index, value_options=val_list, is_read_only=is_read_only,
                    bg_color=background_color, text_color=text_color, variable_type=cur_variable.type)
        table.set_item(row, table.col_val, cell)

    def _setup_value_with_bool(self, cur_variable, table, row, table_is_input,
                               val):
        """Set up the value column for a given variable.

        Args:
            cur_variable: the variable that we are setting up
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            table_is_input (bool): identifies if we are making an input table or result table
            val: the value of the variable
        """
        background_color = self.theme['Table inactive background color']
        text_color = self.theme['Table inactive text color']

        is_read_only = not table_is_input or cur_variable.read_only
        if not table_is_input:
            if val:
                bool_string = 'True'
            else:
                bool_string = 'False'
            # string_item = QTableWidgetItem(bool_string)
            # string_item.setBackground(QColor(background_color))
            # string_item.setForeground(QColor(text_color))
            # string_item.setFlags(string_item.flags() ^ Qt.ItemIsEditable)
            # if not table_is_vertical:
            #     table.setItem(row, table.col_val, string_item)
            # else:
            #     table.setItem(table.col_val, row, string_item)
            cell = Cell(cell_type='string', value=bool_string, is_read_only=is_read_only,
                        bg_color=background_color, text_color=text_color, variable_type=cur_variable.type)
        else:
            # check_box = QCheckBox(table)
            # check_box.setChecked(cur_variable.get_val())
            # if not table_is_vertical:
            #     check_box.stateChanged.connect(lambda txt, cur_row=row, cur_col=table.col_val:
            #                                    combo_change_slot(cur_row, cur_col, txt))
            #     table.setCellWidget(row, table.col_val, check_box)
            # else:
            #     check_box.stateChanged.connect(lambda txt, cur_row=table.col_val, cur_col=row:
            #                                    combo_change_slot(cur_row, cur_col, txt))
            #     table.setCellWidget(table.col_val, row, check_box)

            cell = Cell(cell_type='bool', value=cur_variable.get_val(), is_read_only=is_read_only,
                        bg_color=background_color, text_color=text_color, variable_type=cur_variable.type)
        table.set_item(row, table.col_val, cell)

    def _setup_value_with_string_list(self, cur_variable, table, row, table_is_input, val,
                                      recursion_level):
        """Set up the value column for a given variable.

        Args:
            cur_variable: the variable that we are setting up
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            table_is_input (bool): identifies if we are making an input table or result table
            val: the value of the variable
            recursion_level (int): number of indentations for row; each recursion is an indentation
        """
        background_color, text_color = self._get_background_text_color(cur_variable, recursion_level, True,
                                                                       table_is_input)

        is_read_only = not table_is_input or cur_variable.read_only

        if not table_is_input:
            for index, value in enumerate(val):
                # string_item = QTableWidgetItem(value)
                # if not table_is_input or cur_variable.read_only:
                #     string_item.setBackground(QColor(background_color))
                #     string_item.setForeground(QColor(text_color))
                #     string_item.setFlags(string_item.flags() ^ Qt.ItemIsEditable)
                # else:
                #     string_item.setBackground(QColor(background_color))
                #     string_item.setForeground(QColor(text_color))
                # if not table_is_vertical:
                #     table.setItem(row, table.col_val + index, string_item)
                # else:
                #     table.setItem(table.col_val + index, row, string_item)

                cell = Cell(cell_type='string', value=value, is_read_only=is_read_only,
                            bg_color=background_color, text_color=text_color, variable_type=cur_variable.type)
                table.set_item(row, table.col_val + index, cell)

    def _setup_value_with_button(self, cur_variable, table, row, table_is_input, recursion_level,):
        """Set up the value column for a given variable.

        Args:
            cur_variable: the variable that we are setting up
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            table_is_input (bool): identifies if we are making an input table or result table
            recursion_level (int): number of indentations for row; each recursion is an indentation
        """
        background_color, text_color = self._get_background_text_color(cur_variable, recursion_level, True,
                                                                       table_is_input)

        text = 'View...'
        if cur_variable.type == 'action':
            text = cur_variable.value_options

        # button = QPushButton('View....')
        # if cur_variable.type == 'action':
        #     button.setText(cur_variable.value_options)
        # # button.clicked.connect(partial(self.select_file_or_folder, cur_variable.file_mode,
        # # cur_variable.value_options, row, table.col_unit))
        # if not table_is_vertical:
        #     button.clicked.connect(lambda btn_bool=True, cur_row=row, cur_col=table.col_val:
        #                            button_click_slot(btn_bool, cur_row, cur_col))
        # else:
        #     button.clicked.connect(lambda btn_bool=True, cur_row=table.col_val, cur_col=row:
        #                            button_click_slot(btn_bool, cur_row, cur_col))
        # # if cur_variable.value is None:
        #     # button.setDisabled(True)
        #     # button.setBackground(self.theme['Table read-only background color'])
        #     # button.setForeground(self.theme['Table read-only text color'])
        #     # button.setStyleSheet(
        #     #     f'background-color: {self.theme["Table read-only background color"] };'
        #     #     f'color: {self.theme["Table read-only text color"]} }}')
        # if not table_is_vertical:
        #     table.setCellWidget(row, table.col_val, button)
        # else:
        #     table.setCellWidget(table.col_val, row, button)

        is_read_only = not table_is_input or cur_variable.read_only
        cell = Cell(cell_type='button', value=text, is_read_only=is_read_only,
                    bg_color=background_color, text_color=text_color, variable_type=cur_variable.type)
        table.set_item(row, table.col_val, cell)

    def _setup_value_with_color(self, cur_variable, table, row, table_is_input, recursion_level):
        """Set up the value column for a given variable.

        Args:
            cur_variable: the variable that we are setting up
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            table_is_input (bool): identifies if we are making an input table or result table
            recursion_level (int): number of indentations for row; each recursion is an indentation
        """
        background_color, text_color = self._get_background_text_color(cur_variable, recursion_level, True,
                                                                       table_is_input)

        color = cur_variable.get_val()
        if color is None:
            color = (0, 0, 0)

        # color = QColor(self.rgb_to_hex(cur_variable.get_val()))
        # if color is None:
        #     cur_variable.set_val((0, 0, 0))
        #     color = QColor(self.rgb_to_hex(cur_variable.get_val()))
        # button = FHWAColorButton(color, row=row, col=table.col_val, action=button_click_slot)
        # # button = QPushButton('View....')
        # # FHWAColorButton
        # # button.clicked.connect(partial(self.select_file_or_folder, cur_variable.file_mode,
        # # cur_variable.value_options, row, table.col_unit))
        # # self.setAutoFillBackground(True)
        # # self.setStyleSheet(f'background-color: {table.color}')
        # if not table_is_vertical:
        #     # button.clicked.connect(lambda cur_row=row, cur_col=table.col_val:
        #     #                            button_click_slot(cur_row, cur_col))
        #     table.setCellWidget(row, table.col_val, button)
        # else:
        #     button.row = table.col_val
        #     button.col = row
        #     # button.clicked.connect(lambda cur_row=table.col_val, cur_col=row:
        #     #                            button_click_slot(cur_row, cur_col))
        #     table.setCellWidget(table.col_val, row, button)

        is_read_only = not table_is_input or cur_variable.read_only
        cell = Cell(cell_type='color', value=color, is_read_only=is_read_only,
                    bg_color=background_color, text_color=text_color, variable_type=cur_variable.type)
        table.set_item(row, table.col_val, cell)

    def _setup_value_with_value(self, cur_variable, table, row, table_is_input, recursion_level):
        """Set up the value column for a given variable.

        Args:
            cur_variable: the variable that we are setting up
            table: QT table (so we can make changes on that table)
            row (int): current row for the function (needed for recursion)
            table_is_input (bool): identifies if we are making an input table or result table
            recursion_level (int): number of indentations for row; each recursion is an indentation
        """
        background_color, text_color = self._get_background_text_color(cur_variable, recursion_level, True,
                                                                       table_is_input)

        is_read_only = not table_is_input or cur_variable.read_only
        cell = Cell(cell_type=cur_variable.type, value=cur_variable.get_val(),
                    is_read_only=is_read_only, bg_color=background_color, text_color=text_color,
                    variable_type=cur_variable.type)
        table.set_item(row, table.col_val, cell)

    # @staticmethod
    # def select_file_or_folder(starting_file, file_mode, filter):
    #     """Handle the click on Browse.

    #     Args:
    #         starting_file (string): the starting file for the dialog
    #         file_mode (string): Determines select behavior for File Dialog
    #         filter (string): filter text string for File Dialog

    #     Returns:
    #         selected_file (string): the selected file
    #     """
    #     file_dialog = QFileDialog(directory=starting_file)
    #     if file_mode == 'folder':
    #         file_dialog.setFileMode(QFileDialog.Directory)
    #     elif file_mode == 'existing file':
    #         file_dialog.setFileMode(QFileDialog.ExistingFile)
    #     elif file_mode == 'new file':
    #         file_dialog.setFileMode(QFileDialog.AcceptSave)
    #     elif file_mode == 'any':
    #         file_dialog.setFileMode(QFileDialog.AnyFile)
    #     if filter != '':
    #         file_dialog.setNameFilter(filter)  # Set the filter
    #     result = file_dialog.exec_()

    #     # selected_file, _ = file_dialog.getOpenFileName()
    #     selected_file = None
    #     if result:
    #         selected_file = file_dialog.selectedFiles()
    #         if isinstance(selected_file, list) and len(selected_file) > 0:
    #             selected_file = selected_file[0]

    #     return selected_file

    # @staticmethod
    # def select_color(color):
    #     """Handle the click on Browse.

    #     Args:
    #         file_mode (string): Determines select behavior for File Dialog
    #         filter (string): filter text string for File Dialog

    #     Returns:
    #         color (QColor): the user-selected color
    #     """
    #     new_color = QColorDialog.getColor(color)

    #     if new_color.isValid():
    #         return new_color
    #     return None

    # def table_button_clicked(self, CalcData, table, row, col, launch_dlg,
    #                          add_undo=True, icon=None):
    #     """Get the var key from the table (from a hidden row).

    #     Args:
    #         CalcData: a python class that has: get_input_group, get_results_group, check_warnings, get_can_compute,
    #             clear_results, compute_data, setup_plot_data. This class will manage
    #             lists of variables.
    #         table: QT table (so we can make changes on that table)
    #         row (int): current row that had a cell that was changed
    #         col (int): current col that had a cell that was changed
    #         launch_dlg: a function that will launch a dialog to get a new value
    #         add_undo (bool): Says whether we should add this to the undo history (False when we perform undo)
    #         prev_value: the value that we want to set as the previous value
    #         icon: the icon to use for the launched Dialog
    #     """
    #     result = False
    #     if row >= table.rowCount() or col >= table.columnCount():
    #         return result
    #     row_uuid, uuid_index = self._get_uuid_from_table(table, row, col)
    #     if row_uuid is None:
    #         return result
    #     is_unit_col = col == table.col_unit
    #     if table.table_is_vertical:
    #         is_unit_col = row == table.col_unit
    #     result, item = CalcData.find_item_by_uuid(row_uuid, uuid_index)
    #     if not result:
    #         return result
    #     # Handle different button types:
    #     if uuid_index is not None:
    #         # This is a calc/var list item
    #         # Determine if the user clicked duplicate, delete, or define
    #         button_txt = table.cellWidget(row, col).text()[0:5]
    #         if button_txt == 'Dupli':
    #             result, calc_list_item = CalcData.find_item_by_uuid(row_uuid)
    #             if result:
    #                 calc_list_item.duplicate_item(uuid_index)
    #         elif button_txt == 'Delet':
    #             result, calc_list_item = CalcData.find_item_by_uuid(row_uuid)
    #             if result:
    #                 calc_list_item.delete_item(uuid_index)
    #         elif button_txt == 'Defin' or button_txt == 'View.' or button_txt == 'Edit ':
    #             result, calc = launch_dlg(item, icon=icon)
    #             if result:
    #                 # result, calc_list_item = CalcData.find_item_by_uuid(row_uuid)
    #                 # calc_list_item.set_item_by_index(calc, uuid_index)
    #                 CalcData.set_val_by_uuid(row_uuid, calc, prev_value=item, add_undo=add_undo,
    #                                            uuid_index=uuid_index)
    #                 result = True

    #     elif not is_unit_col:
    #         if item.type in ['calc', 'table',]:
    #             result, calc = launch_dlg(item.value, icon=icon)
    #             if result:
    #                 CalcData.set_val_by_uuid(row_uuid, calc, prev_value=item, add_undo=add_undo)
    #                 result = True
    #         elif item.type in ['image']:
    #             self.display_image(item.get_val())
    #         elif item.type in ['color']:
    #             old_color = item.get_val()
    #             if not isinstance(old_color, QColor):
    #                 old_color = QColor(self.rgb_to_hex(old_color))
    #             color = self.select_color(old_color)
    #             if color:
    #                 rgb_color = (color.red(), color.green(), color.blue())
    #                 CalcData.set_val_by_uuid(row_uuid, rgb_color, prev_value=item.get_val(), add_undo=add_undo)
    #                 result = True
    #         else:
    #             raise ValueError(f'Button clicked on an unsupported type: {item.type}')
    #     elif is_unit_col:  # Buttons in the unit column
    #         if item.type in ['file', 'image']:
    #             new_val = None
    #             old_file = item.get_val()
    #             file = self.select_file_or_folder(item.get_val(), item.file_mode, item.value_options)
    #             if file and item.type in ['image']:
    #                 new_val = cv2.imread(file)
    #             elif file and item.type in ['file']:
    #                 new_val = file
    #             if new_val is not None:
    #                 CalcData.set_val_by_uuid(row_uuid, new_val, prev_value=old_file, add_undo=add_undo)
    #             result = True
    #         else:
    #             raise ValueError(f'Button clicked on an unsupported type: {item.type}')

    #     if result and CalcData.type == 'SettingsCalc':
    #         AppData.settings_calc = CalcData
    #     return result
