"""Presenter that creates a view-model of the about dialog."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.interface_adapters.view_model.main.about_dlg import AboutDlgData
from xms.FhwaVariable.interface_adapters.view_model.main.about_dlg import ApplicationInfo
from xms.FhwaVariable.interface_adapters.view_model.main.about_dlg import CreditInfo
from xms.FhwaVariable.interface_adapters.view_model.main.about_dlg import DeveloperInfo
from xms.FhwaVariable.interface_adapters.view_model.main.about_dlg import ModelInfo
from xms.FhwaVariable.interface_adapters.view_model.main.about_dlg import SponsoringAgencyInfo


class AboutPresenter:
    """Presenter that creates a view-model of the about dialog."""
    def __init__(self, app_name: str, view: object):
        """Initialize the about presenter class.

        Args:
            app_name (str): The name of the application.
            view (object): The view object.
        """
        self.app_name = app_name
        self.view = view

    def show_about_dialog(self, model_dict):
        """Show the about dialog."""
        help_dialog_data = self.create_view_model(model_dict)
        self.view.show_about_dialog(help_dialog_data)

    def create_view_model(self, model_dict):
        """Create the view model."""
        # Application Information
        model_base = model_dict[self.app_name]
        app_info = ApplicationInfo(
            app_name=model_base.about_dict['model_name'],
            app_icon=str(Path(model_base.about_dict['model_icon'])) if model_base.about_dict['model_icon'] else "",
            version=model_base.about_dict['version'],
            build_message=model_base.about_dict['build_message'],
            build_date=model_base.about_dict['build_date'],
            website=model_base.about_dict['model_website'],
            wiki=model_base.about_dict['wiki_url'],
            description=model_base.about_dict['description'] or "No description available."
        )

        # Sponsoring Agency Information
        sponsoring_agency_info = SponsoringAgencyInfo(
            name=model_base.about_dict['agency'],
            icon=str(Path(model_base.about_dict['agency_icon'])) if model_base.about_dict['agency_icon'] else "",
            website=model_base.about_dict['agency_website'],
            description=model_base.about_dict['agency_description']
        )

        # Interface Developer Information
        model_developer_info = DeveloperInfo(
            name=model_base.about_dict['developed_by'],
            icon=str(Path(
                model_base.about_dict['developed_by_icon'])) if model_base.about_dict['developed_by_icon'] else "",
            website=model_base.about_dict['developed_by_website'],
            description=model_base.about_dict['developed_by_description']
        )

        # Interface Developer Information
        interface_developer_info = DeveloperInfo(
            name=model_base.about_dict['interface_developed_by'],
            icon=str(Path(model_base.about_dict['interface_developed_by_icon'])) if model_base.about_dict[
                'interface_developed_by_icon'] else "",
            website=model_base.about_dict['interface_developed_by_website'],
            description=model_base.about_dict['interface_developed_by_description']
        )

        # Populate AboutDlgData
        help_dialog_data = AboutDlgData(
            application_info=app_info,
            sponsoring_agency_info=sponsoring_agency_info,
            model_developer_info=model_developer_info,
            interface_developer_info=interface_developer_info,
            credits=self.get_credits(),  # me & others
            included_models=self.get_models(model_dict),  # List of models included in the application
            available_models=[]  # List of models available for purchase
        )

        return help_dialog_data

    def get_models(self, model_dict):
        """Get the models."""
        models = []
        for model in model_dict.values():
            if model.model_name == self.app_name:
                continue
            models.append(ModelInfo(
                model_name=model.model_name,
                model_icon=str(Path(model.model_icon)) if model.model_icon else "",
                version=model.version,
                sponsoring_agency=model.agency,
                model_website=model.model_website,
                wiki_url=model.wiki_url,
                description=model.description or "No description available."
            ))
        return models

    def get_credits(self):
        """Get the credits."""
        credits = []
        # Add Aquaveo
        credits.append(CreditInfo(
            name="Aquaveo",
            image=Path(
                "C:/Users/username/PycharmProjects/xms/FhwaVariable/interface_adapters/view_model/main/Aquaveo.png"),
            description="The company that developed the application."
        ))
        # Add me
        credits.append(CreditInfo(
            name="Eric Joseph Jones (Regneva Wilhelm Tone)",
            image=Path(
                "C:/Users/username/PycharmProjects/xms/FhwaVariable/interface_adapters/view_model/main/Regneva.png"),
            description="I did all the work. :)   (ok, to be technical, I did most of the work.)"
        ))
        # Add others
        credits.append(CreditInfo(
            name="Others",
            image=Path(
                "C:/Users/username/PycharmProjects/xms/FhwaVariable/interface_adapters/view_model/main/others.png"),
            description="Other developers."
        ))
        return credits
