"""Classes to handle the GUI side of variables."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
import uuid

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.interface_adapters.view_model.main.tree_data import TreeData
# from xms.FhwaVariable.core_data.calculator.metadata import Metadata


class _TreeItem(TreeData):
    """Data class to hold commands that are executed. Used to keep an undo/redo history."""
    current_file_directory = Path(__file__).parent
    two_levels_up_directory = current_file_directory.parent.parent.parent
    icon_folder = two_levels_up_directory / 'core_data' / 'resources' / 'icons'

    def __init__(self, name, tree_item_uuid=None, item_class=None, parent=None, parent_uuid=None, icon=None,
                 is_checked=False, is_partially_checked=False, is_enabled=True, is_expanded=False, is_selected=False,
                 tool_tip=None, item_uuid=None, project_uuid=None):
        """Initialize the _TreeItem class.

        Args:
            name (string): Tells which function is called to undo/redo command
            tree_item_uuid (UUID): Unique identifier for the tree item
            item_class (string): The class to create the object
            parent (string): Unique identifier for the command
            parent_uuid (string): Unique identifier for the command
            icon (string): Unique identifier for the command
            is_checked (bool): Whether the tree item is is_checked
            is_partially_checked (bool): Whether the tree item is is_partially_checked
            is_enabled (bool): Whether the tree item is is_enabled
            is_expanded (bool): Whether the tree item is is_expanded
            is_selected (bool): Whether the tree item is is_selected
            tool_tip (string): Text shown to the user in a display of undo/redo history
            item_uuid (string): Unique identifier for the command
            project_uuid (string): Unique identifier for the command
        """
        super().__init__(icon=icon, is_checked=is_checked, is_expanded=is_expanded, is_selected=is_selected,
                         parent_uuid=parent_uuid)

        self.uuid = tree_item_uuid
        if self.uuid is None:
            if item_uuid is not None:
                self.uuid = item_uuid
            else:
                self.uuid = uuid.uuid4()
        self.name = name
        self.item_class = item_class
        self.is_partially_checked = is_partially_checked
        self.is_enabled = is_enabled
        self.tool_tip = tool_tip
        self.parent = parent
        self.children = []
        self.item_uuid = item_uuid
        self.accepted_drop_types = []
        self.project_uuid = project_uuid
        self.is_visible = True


class FolderItem(_TreeItem):
    """Data class to organize the data in the project data tree in folders."""
    file_icon = _TreeItem.icon_folder / 'File_Save.ico'
    settings_icon = _TreeItem.icon_folder / 'GreyGear.ico'
    root_icon = _TreeItem.icon_folder / 'root.ico'
    folder_icon = _TreeItem.icon_folder / 'Folder.ico'
    proposed_icon = _TreeItem.icon_folder / 'BlueFolder.ico'
    existing_icon = _TreeItem.icon_folder / 'RedFolder.ico'

    folder_list = ['root', 'Model', 'Project', 'Folder', 'Proposed', 'Existing', 'File', 'Settings',
                   'Geometry Folder', 'Coverage Folder', 'GIS Folder', 'Model Folder']

    def __init__(self, name, class_name='Folder', item_class=None, icon=None, is_checked=False,
                 is_partially_checked=False, is_enabled=True, is_expanded=True, is_selected=False, tool_tip=None,
                 item_uuid=None, parent=None, parent_uuid=None, project_uuid=None):
        """Initialize the FolderItem class.

        Args:
            name (string): Tells which function is called to undo/redo command
            class_name (string): The class that will reference to the calcdata_dict
            item_class (string): The class to create the object
            icon (string): Unique identifier for the command
            is_checked (bool): Whether the tree item is is_checked
            is_partially_checked (bool): Whether the tree item is is_partially_checked
            is_enabled (bool): Whether the tree item is is_enabled
            is_expanded (bool): Whether the tree item is is_expanded
            is_selected (bool): Whether the tree item is is_selected
            tool_tip (string): Text shown to the user in a display of undo/redo history
            item_uuid (string): Unique identifier for the command
            parent (string): Unique identifier for the command
            parent_uuid (string): Unique identifier for the command
            project_uuid (string): Unique identifier for the command
        """
        self.uuid = uuid.uuid4()
        if item_uuid is None:
            item_uuid = self.uuid
        super().__init__(name, item_class=item_class, icon=icon, is_checked=is_checked,
                         is_partially_checked=is_partially_checked, is_enabled=is_enabled, is_expanded=is_expanded,
                         is_selected=is_selected, tool_tip=tool_tip, item_uuid=item_uuid, parent=parent,
                         parent_uuid=parent_uuid, project_uuid=project_uuid)

        self.is_drop_enabled = True

        self.class_name = class_name
        self.type = 'FolderItem'

        # self.metadata = Metadata()

        # current_file_directory = os.path.dirname(os.path.abspath(__file__))
        # two_levels_up_directory = os.path.dirname(current_file_directory)
        # icon_folder = os.path.join(two_levels_up_directory, 'resources', 'icons')

        # TODO: Set the icons in the app_data, then get them here
        # TODO: Set the model_icon with the model interactor
        self.model_icon = FolderItem.root_icon
        self.project_icon = FolderItem.root_icon
        self.folder_icon = FolderItem.folder_icon
        self.proposed_icon = FolderItem.proposed_icon
        self.existing_icon = FolderItem.existing_icon

        self.set_standard_icon(False)

    def set_standard_icon(self, override=True):
        """Sets the icon for the folder."""
        if override is False and self.icon is not None:
            return
        if self.class_name == FolderItem.folder_list[0]:
            self.icon = self.root_icon
        elif self.class_name == FolderItem.folder_list[1]:
            self.icon = self.model_icon
        elif self.class_name == FolderItem.folder_list[2]:
            self.icon = self.project_icon
        elif self.class_name == FolderItem.folder_list[3]:
            self.icon = self.folder_icon
        elif self.class_name == FolderItem.folder_list[4]:
            self.icon = self.proposed_icon
        elif self.class_name == FolderItem.folder_list[5]:
            self.icon = self.existing_icon
        elif self.class_name == FolderItem.folder_list[6]:
            self.icon = self.file_icon
        elif self.class_name == FolderItem.folder_list[7]:
            self.icon = self.settings_icon
        elif self.class_name == FolderItem.folder_list[8:11]:  # Main project folders
            self.icon = self.folder_icon


class Ugrid(_TreeItem):
    """Data class to hold the Ugrid data."""

    def __init__(self, name, item_class=None, icon=None, is_checked=False,
                 is_partially_checked=False, is_enabled=True, is_expanded=True, is_selected=False, tool_tip=None,
                 item_uuid=None, parent=None, parent_uuid=None, project_uuid=None):
        """Initialize the Ugrid class.

        Args:
            name (string): Tells which function is called to undo/redo command
            item_class (string): The class to create the object
            icon (string): Unique identifier for the command
            is_checked (bool): Whether the tree item is is_checked
            is_partially_checked (bool): Whether the tree item is is_partially_checked
            is_enabled (bool): Whether the tree item is is_enabled
            is_expanded (bool): Whether the tree item is is_expanded
            is_selected (bool): Whether the tree item is is_selected
            tool_tip (string): Text shown to the user in a display of undo/redo history
            item_uuid (string): Unique identifier for the command
            parent (string): Unique identifier for the command
            parent_uuid (string): Unique identifier for the command
            project_uuid (string): Unique identifier for the command
        """
        super().__init__(name, item_class=item_class, icon=icon, is_checked=is_checked,
                         is_partially_checked=is_partially_checked, is_enabled=is_enabled, is_expanded=is_expanded,
                         is_selected=is_selected, tool_tip=tool_tip, item_uuid=item_uuid, parent=parent,
                         parent_uuid=parent_uuid, project_uuid=project_uuid)

        self.icon = _TreeItem.icon_folder / 'UGrid_Module_Icon.svg'

        self.ugrid = None
        self.filename = None

        self.class_name = 'Geometry'

        self.is_drop_enabled = False
        self.is_drag_enabled = True

        # self.metadata = Metadata()
