"""Classes to Define the model-view of a window (or document)."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass, field
import uuid

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.interface_adapters.view_model.window_base import WindowBase


@dataclass
class Window(WindowBase):
    """A class to define the model-view of a window (or document)."""
    window_uuid: uuid.UUID = field(default_factory=uuid.uuid4)  # Generate a new UUID for each instance
    window_index: int = 0
    window_type: str = "graphics"  # Possible window types: graphics, image, plot, table

    # # Docking information
    # docked: bool = False
    # dock_area: Optional[str] = None  # Possible values: 'left', 'right', 'top', 'bottom'


@dataclass
class GraphicsImage():
    """Resulting image to give to the view for display."""
    # neutral_blue: tuple[int, int, int] = (128, 128, 255)  # Instance-level field
    image_count: int = 0
    image_width: int = 800
    image_height: int = 600
    image_bytes: bytes = field(default=None)  # Computed in __post_init__


@dataclass
class Orientation():
    """Camera orientation to give to the graphics to render the view."""
    bearing: float = 0.0
    dip: float = 90.0
    at: tuple[float, float, float] = (0.0, 0.0, 0.0)
    width: float = 0.0
    height: float = 0.0


@dataclass
class GraphicsData:
    """Data that will be given to Graphics to generate the display image."""
    display_settings = None
    geom_uuid_list: list = field(default_factory=list)
    dataset_uuid_list: list = field(default_factory=list)
    coverages_uuid_list: list = field(default_factory=list)
    image_uuid_list: list = field(default_factory=list)


@dataclass
class GraphicsWindow(Window):
    """A graphics view view model."""
    is_active: bool = False
    resulting_image: GraphicsImage = field(default_factory=GraphicsImage)
    camera_orientation: Orientation = field(default_factory=Orientation)
    graphics_data: GraphicsData = field(default_factory=GraphicsData)


@dataclass
class ImageWindow(Window):
    """An image view."""
    image = None


@dataclass
class PlotWindow(Window):
    """A plot view."""
    plot_dict = {}


@dataclass
class TableWindow(Window):
    """A table view."""
    table_dict = {}
