"""CalcData for performing Gradually Varied Flow operations."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import math
import statistics
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator import Calculator

# 4. Local modules
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation
from xms.HydraulicToolboxCalc.util.intersection import do_intersect_xy, find_intersection_xy


class GVFCalc(Calculator):
    """A class that defines a channel and performs Manning's n computations."""

    def _get_can_compute(self, unknown=None):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Args:
            unknown (string): variable that is unknown and being calculated.

        Returns:
            bool: True if can compute
        """
        result = False
        _, self.zero_tol = self.get_data('Zero tolerance', 1e-6)
        _, self.null_data = self.get_data('Null data', -9999.0)

        if unknown is None:
            unknown = 'Head'
        self.manning_n_calc.unknown = unknown

        self.input_dict['calc_data']['Geometry']['calculator'].input_dict = copy.copy(self.input_dict)
        self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data'] = copy.copy(
            self.input_dict['calc_data']['Geometry'])
        result = self.input_dict['calc_data']['Geometry']['calculator']._get_can_compute(unknown)
        self.warnings.update(self.input_dict['calc_data']['Geometry']['calculator'].warnings)

        found = False
        flows = self.input_dict['calc_data']['Flows']
        if hasattr(flows, '__len__') and len(flows) < 1:
            self.warnings['flow'] = "Please enter a flow"
            result = False
        else:
            for flow in flows:
                if flow > 0.0:
                    found = True
                    break
            if not found:
                self.warnings['flow'] = "Please enter a positive, non-zero flow"
                result = False

        # if self.input_dict['Composite n'] < 0.0001:
        #     self.warnings.append("Please enter the composite Manning's n value")
        #     result = False

        self.manning_n_calc.input_dict = self.input_dict['calc_data']['Geometry']['calculator'].input_dict
        self.manning_n_calc.input_dict['calc_data']['Flows'] = self.input_dict['calc_data']['Flows']

        if self.input_dict['calc_data']['Upstream invert station'] == self.input_dict['calc_data'][
                'Downstream invert station']:
            self.warnings['invert_stations'] = "Please enter the invert stations"
            result = False

        self.input_dict['calc_data']['Upstream water depth']['calculator'].input_dict = copy.copy(self.input_dict)
        self.input_dict['calc_data']['Upstream water depth']['calculator'].input_dict['calc_data'] = copy.copy(
            self.input_dict['calc_data']['Upstream water depth'])
        self.input_dict['calc_data']['Upstream water depth']['calculator'].manning_n_calc.input_dict = copy.copy(
            self.manning_n_calc.input_dict)

        self.input_dict['calc_data']['Downstream water depth']['calculator'].input_dict = copy.copy(self.input_dict)
        self.input_dict['calc_data']['Downstream water depth']['calculator'].input_dict['calc_data'] = copy.copy(
            self.input_dict['calc_data']['Downstream water depth'])
        self.input_dict['calc_data']['Downstream water depth']['calculator'].manning_n_calc.input_dict = copy.copy(
            self.manning_n_calc.input_dict)

        return result

    def clear_results(self):
        """Clears the results and those of subclasses to prepare for computation."""
        self.warnings = {}
        self.input_dict['calc_data']['Geometry']['calculator'].clear_results()

        self.normal_depths = []
        self.results['Normal depth'] = []

        self.critical_depths = []
        self.results['Critical depth'] = []

        self.results['Downstream depth'] = []
        self.results['Downstream velocity'] = []
        self.results['Length flowing full'] = []
        self.results['Length flowing free'] = []

        # Intermediate Results
        # self.high_elev = 0.0
        # self.low_elev = 0.0
        self.results['Station'] = []
        self.results['WSE'] = []
        self.results['Distance from inlet'] = []
        self.results['Depth'] = []
        self.results['Flow area'] = []
        self.results['Wetted perimeter'] = []
        self.results['Top width'] = []
        self.results['Hydraulic radius'] = []
        self.results['Manning n'] = []
        self.results['Velocity'] = []
        self.results['Energy'] = []
        self.results['Energy loss'] = []
        self.results['Energy slope'] = []
        self.results['Boundary shear stress'] = []
        self.results['Max shear stress'] = []

        self.plot_x = []
        self.plot_y = []

        self.upstream_depths = []
        self.upstream_flow_areas = []
        self.upstream_velocities = []
        self.upstream_energies = []

        self.downstream_depths = []
        self.downstream_flow_areas = []
        self.downstream_velocities = []
        self.downstream_energies = []

        # vena contracta
        self.vena_contracta_lengths = []
        self.vena_contracta_depths = []
        self.vena_contracta_x = []
        self.vena_contracta_y = []

        # Hydraulic Jump
        self.hyd_jump_stations = []
        self.hyd_jump_depths = []
        self.hyd_jump_sequent_depths = []
        self.hyd_jump_lengths = []
        self.hyd_jump_type_b_over_breaks = []
        self.hyd_jump_exists_list = []
        self.hyd_jump_swept_outs = []
        self.froude_jumps = []
        self.hyd_jump_station_absolutes = []
        self.hyd_jump_final_station_absolutes = []
        self.hyd_jump_elevations = []
        self.hyd_jump_sequent_elevations = []
        self.froude_classes = []

        self.flow_profile = ''
        self.flow_profiles = []

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        _, self.null_data = self.get_data('Null data', -9999.0)
        _, self.zero_tol = self.get_data('Zero tolerance', 1e-6)
        self._update_manning_calc()

        flows = self.input_dict['calc_data']['Flows']

        # Establish slope type (use largest flow)
        self.full_flow = sys.float_info.max  # Set to avoid full flow assumption
        index = flows.index(max(flows))
        self.flow = flows[index]
        self._determine_direction_and_starting_depth(index=index)

        # Compute the full flow parameters if applicable
        self.flowing_full = False
        compute_full_flow = self._compute_full_flow_parameters()

        # Compute WSE
        index = 0
        for flow in flows:
            self.flow = flow
            # Determine direction and starting depth
            self._determine_direction_and_starting_depth(index=index)

            # Determine if the backwater curve traveling in the downstream direction is full flow
            if compute_full_flow:
                self.flowing_full = self.determine_culvert_flows_full()

            # TODO: In culverts, if inlet_control_depth > outlet_control_depth and first culvert
            # (not middle or lower culvert in brokenbacks), compute_vena_contracta = True
            self.compute_vena_contracta_data()

            # Compute a backwater curve traveling in the upstream direction
            self._compute_good_backwater_curve(
                flow, 'upstream', self.up_starting_depth, self.up_increase_in_depth)
            self.results_upstream = copy.deepcopy(self.intermediate_results)

            # modify the x,y for the upstream direction, so we know we have the profile going downstream
            self._convert_upstream_x_y_to_match_downstream()

            # Compute a backwater curve traveling in the downstream direction
            self._compute_good_backwater_curve(
                flow, 'downstream', self.down_starting_depth, self.down_increase_in_depth)
            self.results_downstream = copy.deepcopy(self.intermediate_results)

            # Check for Hydraulic Jump
            self._check_for_hydraulic_jump()

            # Prepare curve data
            self._combine_backwater_curve()

            self._convert_depths_and_distances_to_elevations_and_stations()

            self.assign_results()

            index += 1
        return True

    def set_plotting_data(self, index):
        """Sets the plotting data for the cross-section.

        Args:
            index (int): The index of the cross-section to plot.
        """
        if 'Longitudinal profile' not in self.plot_dict:
            return

        # Determine the channel profile
        display_flat = self.input_dict['calc_data']['Display channel slope as flat']

        up_station = self.input_dict['calc_data']['Upstream invert station']
        up_elevation = self.input_dict['calc_data']['Upstream invert elevation']
        down_station = self.input_dict['calc_data']['Downstream invert station']
        down_elevation = self.input_dict['calc_data']['Downstream invert elevation']

        if display_flat:
            up_elevation = 0.0
            down_elevation = 0.0

        closed_shape = self.input_dict['calc_data']['Geometry']['calculator'].determine_if_shape_is_closed()
        rise = self.input_dict['calc_data']['Geometry']['calculator'].rise_to_crown
        rise += self.min_embedment_depth

        channel_x = [up_station, down_station]
        channel_y = [up_elevation, down_elevation]
        x_col = copy.copy(channel_x)
        y_col = copy.copy(channel_y)
        if closed_shape:
            channel_x = [up_station, down_station, down_station, up_station, up_station]
            channel_y = [up_elevation, down_elevation, down_elevation + rise, up_elevation + rise, up_elevation]

        # Embedment
        embed_x = []
        embed_y = []
        if self.min_embedment_depth > self.zero_tol:
            embed_x = copy.deepcopy(x_col)
            embed_y = [y + self.min_embedment_depth for y in y_col]
            for x in reversed(x_col):
                embed_x.append(x)
            for y in reversed(y_col):
                embed_y.append(y)
            embed_x.append(embed_x[0])
            embed_y.append(embed_y[0])

        critical_y = []
        if len(self.critical_depths) > index and len(self.critical_depths) > 0:
            critical_elev = self.critical_depths[index] + self.min_embedment_depth
            if critical_elev > self.min_embedment_depth:
                critical_y = [y + critical_elev for y in y_col]

        normal_y = []
        if len(self.normal_depths) > index and len(self.normal_depths) > 0:
            normal_elev = self.normal_depths[index] + self.min_embedment_depth
            if normal_elev > self.min_embedment_depth:
                normal_y = [y + normal_elev for y in y_col]

        # Water Surface Profile
        wsp_x = []
        wsp_y = []
        if len(self.plot_x) > 0 and len(self.plot_x[index]) > 0 and len(self.plot_x[index]) > 0:
            wsp_x = list(self.plot_x[index]) + list(reversed(x_col)) + [self.plot_x[index][0]]
            wsp_y = list(self.plot_y[index]) + list(reversed(y_col)) + [self.plot_y[index][0]]

        vena_x = []
        vena_y = []
        if self.compute_vena_contracta_data and len(self.vena_contracta_xs) > 0 and len(
                self.vena_contracta_xs[index]) > 0:
            vena_x = self.vena_contracta_xs[index]
            vena_y = self.vena_contracta_ys[index]

        jump_x = []
        jump_y = []
        if len(self.hyd_jump_exists_list) > 0 and self.hyd_jump_exists_list[index]:
            jump_x = [self.hyd_jump_station_absolutes[index], self.hyd_jump_lengths[index]
                      + self.hyd_jump_station_absolutes[index]]
            if display_flat:
                jump_y = [self.hyd_jump_depths[index] + self.min_embedment_depth,
                          self.hyd_jump_sequent_depths[index] + self.min_embedment_depth]
            else:
                jump_y = [self.hyd_jump_elevations[index] + self.min_embedment_depth,
                          self.hyd_jump_sequent_elevations[index] + self.min_embedment_depth]

        # Channel
        self.plot_dict['Longitudinal profile']['series'][0]['x var'].set_val(channel_x)
        self.plot_dict['Longitudinal profile']['series'][0]['y var'].set_val(channel_y)

        # Embedment
        self.plot_dict['Longitudinal profile']['series'][1]['x var'].set_val(embed_x)
        self.plot_dict['Longitudinal profile']['series'][1]['y var'].set_val(embed_y)

        # Water Surface Profile
        self.plot_dict['Longitudinal profile']['series'][2]['x var'].set_val(wsp_x)
        self.plot_dict['Longitudinal profile']['series'][2]['y var'].set_val(wsp_y)

        # Vena contracta
        self.plot_dict['Longitudinal profile']['series'][3]['x var'].set_val(vena_x)
        self.plot_dict['Longitudinal profile']['series'][3]['y var'].set_val(vena_y)

        # Hydraulic Jumps
        self.plot_dict['Longitudinal profile']['series'][4]['x var'].set_val(jump_x)
        self.plot_dict['Longitudinal profile']['series'][4]['y var'].set_val(jump_y)

        # Critical Depth
        self.plot_dict['Longitudinal profile']['series'][5]['x var'].set_val(x_col)
        self.plot_dict['Longitudinal profile']['series'][5]['y var'].set_val(critical_y)

        # Normal Depth
        self.plot_dict['Longitudinal profile']['series'][6]['x var'].set_val(x_col)
        self.plot_dict['Longitudinal profile']['series'][6]['y var'].set_val(normal_y)

    def assign_results(self):
        """Assigns the results from a computational run to the appropriate locations."""
        # Assign the results
        first_station = self.input_dict['calc_data']['Upstream invert station']
        _, gamma = self.get_data('Unit weight of water')

        self.normal_depths.append(self.normal_depth)
        self.results['Normal depth'].append(self.normal_depth)

        self.critical_depths.append(self.critical_depth)
        self.results['Critical depth'].append(self.critical_depth)

        self.results['Downstream depth'].append(self.y[-1])
        self.results['Downstream velocity'].append(self.velocity[-1])
        self.results['Length flowing full'].append(self.length_flowing_full)
        self.results['Length flowing free'].append(self.total_length - self.length_flowing_full)

        self.boundary_shear_stress = []
        self.max_shear_stress = []

        for energy_slope, hyd_rad, depth in zip(self.energy_slope, self.hydraulic_radius, self.y):
            self.boundary_shear_stress.append(gamma * hyd_rad * energy_slope)
            self.max_shear_stress.append(gamma * depth * energy_slope)

        self.results['Station'].append(self.wse_stations)
        self.results['WSE'].append(self.wse_elevations)
        self.results['Distance from inlet'].append(self.x)
        self.results['Depth'].append(self.y)
        self.results['Flow area'].append(self.flow_area)
        self.results['Wetted perimeter'].append(self.wetted_perimeter)
        self.results['Top width'].append(self.top_width)
        self.results['Hydraulic radius'].append(self.hydraulic_radius)
        self.results['Manning n'].append(self.manning_n)
        self.results['Velocity'].append(self.velocity)
        self.results['Energy'].append(self.energy)
        self.results['Energy loss'].append(self.energy_loss)
        self.results['Energy slope'].append(self.energy_slope)
        self.results['Boundary shear stress'].append(self.boundary_shear_stress)
        self.results['Max shear stress'].append(self.max_shear_stress)

        if self.input_dict['calc_data']['Display channel slope as flat']:
            self.plot_x.append(self.x)
            self.plot_y.append(self.y)
        else:
            self.plot_x.append(self.wse_stations)
            self.plot_y.append(self.wse_elevations)

        if self.hyd_jump_swept_out:
            self.warnings['hydraulic_jump'] = (
                f'For flow profile {self.flow} an hydraulic jump forms near the culvert outlet but may be swept into '
                'the tailwater channel'
            )

        self.upstream_depths.append(self.wse_elevations[0] - self.input_dict['calc_data']['Upstream invert elevation'])
        self.upstream_flow_area = 0.0
        if len(self.flow_area) > 0:
            self.upstream_flow_area = self.flow_area[0]
        self.upstream_flow_areas.append(self.upstream_flow_area)
        self.upstream_velocity = 0.0
        if self.upstream_flow_area > 0.0:
            self.upstream_velocity = self.flows[0] / self.upstream_flow_area
        self.upstream_velocities.append(self.upstream_velocity)
        self.upstream_energies.append(self.results['Energy'][0])

        self.downstream_depths.append(self.wse_elevations[-1] - self.input_dict['calc_data'][
            'Downstream invert elevation'])
        self.downstream_flow_area = 0.0
        if len(self.flow_area) > 0:
            self.downstream_flow_area = self.flow_area[-1]
        self.downstream_flow_areas.append(self.downstream_flow_area)
        self.downstream_velocity = 0.0
        if self.downstream_flow_area > 0.0:
            self.downstream_velocity = self.flows[0] / self.downstream_flow_area
        self.downstream_velocities.append(self.downstream_velocity)
        self.downstream_energies.append(self.results['Energy'][-1])

        # vena contracta
        self.vena_contracta_lengths.append(self.vena_contracta_length)
        self.vena_contracta_depths.append(self.vena_contracta_depth)
        self.vena_contracta_xs.append([x + first_station for x in self.vena_contracta_x])
        self.vena_contracta_ys.append(self.vena_contracta_y)

        # Hydraulic Jump
        self.hyd_jump_stations.append(self.hyd_jump_station + first_station)
        self.hyd_jump_depths.append(self.hyd_jump_depth)
        self.hyd_jump_sequent_depths.append(self.hyd_jump_sequent_depth)
        self.hyd_jump_lengths.append(self.hyd_jump_length)
        self.hyd_jump_type_b_over_breaks.append(self.hyd_jump_type_b_over_break)
        self.hyd_jump_exists_list.append(self.hyd_jump_exists)
        self.hyd_jump_swept_outs.append(self.hyd_jump_swept_out)
        self.froude_jumps.append(self.froude_jump)
        self.hyd_jump_station_absolutes.append(self.hyd_jump_station_absolute)
        self.hyd_jump_final_station_absolutes.append(self.hyd_jump_final_station_absolute)
        self.hyd_jump_elevations.append(self.hyd_jump_elevation)
        self.hyd_jump_sequent_elevations.append(self.hyd_jump_sequent_elevation)
        self.froude_classes.append(self.froude_class)

        # Outlet depth
        self.depth_outlet = self.wse_elevations[-1] - self.input_dict['calc_data']['Downstream invert elevation']
        self.vel_outlet = self.velocity[-1]

        self.determine_flow_profile()

    def _compute_full_flow_parameters(self):
        """Computes the full flow parameters.

        Returns:
            Compute_full_flow (bool): True if we should compute the full flow case.
                (False for H, A slopes and open shapes)
        """
        if self.slope_type in ['horizontal', 'adverse']:
            return False

        self._update_manning_calc()
        manning_n_calc = copy.deepcopy(self.manning_n_calc)

        # Determine if we have a closed shape; exit if we do not
        geom_var = manning_n_calc.input_dict['calc_data']['Geometry']['calculator']
        self.closed_shape = geom_var.determine_if_shape_is_closed()
        if not self.closed_shape:
            return False

        manning_n_calc.input_dict['calc_data']['Geometry'] = copy.deepcopy(self.input_dict['calc_data']['Geometry'])
        manning_n_calc.input_dict['calc_data']['Head'] = 'Depth'
        manning_n_calc.input_dict['calc_data']['Calculate'] = 'Flows'

        manning_n_calc.input_dict['calc_data']['Depths'] = [sys.float_info.max]
        manning_n_calc.compute_data()
        stations = manning_n_calc.input_dict['calc_data']['Geometry']['calculator'].stations

        self.full_flow = manning_n_calc.results['Flows'][-1]
        self.full_flow_area = manning_n_calc.results['Flow area'][-1]
        self.full_flow_hydraulic_radius = manning_n_calc.results['Hydraulic radius'][-1]
        self.full_flow_velocity = manning_n_calc.results['Average velocity'][-1]
        self.full_flow_n = manning_n_calc.results['Composite n value'][-1]
        self.span = max(stations) - min(stations)
        self.rise = manning_n_calc.high_elev - manning_n_calc.low_elev
        self.length_flowing_full = 0.0
        return True

    def _compute_full_flow(self):
        """Computes the full flow parameters."""
        # Compute the length of the culvert barrel
        # self.length_flowing_full = self.input_dict['Site data'].get_length()
        down_station = self.input_dict['calc_data']['Downstream invert station']
        up_station = self.input_dict['calc_data']['Upstream invert station']
        self.length_flowing_full = abs(down_station - up_station)

        # Compute the full flow parameters
        # Flow Area
        # Hydraulic Radius
        # n value
        # Velocity at full flow, velocity for critical depth

        v_diff = self.full_flow_velocity - self.downstream_velocity
        if v_diff < 0.0:
            v_diff = 0.0
        _, two_g = self.get_data('Gravity')
        two_g *= 2.0
        l_f = self.length_flowing_full
        r = self.full_flow_hydraulic_radius
        v_f = self.full_flow_velocity
        n = self.full_flow_n
        self.headloss = (self.ke + (29.0 * (n ** 2) * l_f / (r ** (4.0 / 3.0)))) * ((v_f ** 2) / two_g) + (
            1.0 * ((v_diff ** 2) / two_g))

        culvert_rise_over_length = self.inlet_invert_elevation - self.outlet_invert_elevation

        total_energy = 0.0
        self.vel_outlet = 0.0
        self.downstream_depth = self.up_starting_depth
        if self.downstream_depth >= self.rise:  # is outlet_submerged
            total_energy = self.downstream_depth + self.headloss
            self.outlet_control_depth = total_energy - culvert_rise_over_length
            self.depth_outlet = self.rise
            self.up_starting_depth = self.rise
            self.vel_outlet = self.full_flow_velocity
            return
        elif self.downstream_depth > self.critical_depth:
            total_energy = self.downstream_depth + self.headloss
            self.depth_outlet = self.downstream_depth
            self.up_starting_depth = self.downstream_depth
        elif self.downstream_depth <= self.critical_depth:
            if self.critical_depth > 0.75 * self.rise:
                tw_full = (self.critical_depth + self.rise) / 2.0
            else:
                tw_full = self.critical_depth
                _, use_brink_depth = self.get_data('Use brink depth', True)
                if use_brink_depth:
                    tw_full = self.brink_depth
            total_energy = tw_full + self.headloss
            self.depth_outlet = self.critical_depth
            self.up_starting_depth = self.critical_depth
        self.outlet_control_depth = total_energy - self.inlet_invert_elevation - self.outlet_invert_elevation
        if self.depth_outlet <= 0.0:
            self.vel_outlet = 0.0
        else:
            if self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Shape'] in ['box']:
                self.vel_outlet += self.flow / (self.span * self.depth_outlet)
            else:
                # Compute the flow area for a given depth: depth_outlet
                self._update_manning_calc()
                manning_n_calc = copy.deepcopy(self.manning_n_calc)
                manning_n_calc.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Depths'] = [
                    self.depth_outlet]
                manning_n_calc.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Head'] = \
                    'Depth'
                manning_n_calc.input_dict['calc_data']['Geometry']['calculator']._compute_data()
                flow_area = manning_n_calc.input_dict['calc_data']['Geometry']['calculator'].results['Flow area'][-1]
                self.vel_outlet = self.flow / flow_area

    def determine_culvert_flows_full(self):
        """Determines if the culvert is flowing full.

        Returns:
            bool: True if the culvert is flowing full
        """
        self.manning_n_calc.input_dict['calc_data']

        # self.normal_depth = self.normal_depths[index]
        # self.critical_depth = self.critical_depths[index]
        self.rise = self.input_dict['calc_data']['Geometry']['calculator'].rise_to_crown
        self.inlet_invert_elevation = self.input_dict['calc_data']['Upstream invert elevation']
        self.outlet_invert_elevation = self.input_dict['calc_data']['Downstream invert elevation']
        inlet_crown_depth_above_outlet_invert = self.inlet_invert_elevation + self.rise - self.outlet_invert_elevation

        self.full_flow_velocity = self.flow / self.full_flow_area

        tw_depth = self.up_starting_depth

        if self.flow >= self.full_flow:
            self.normal_depth = self.rise
            self._compute_full_flow()
            return True
        elif self.critical_depth < self.normal_depth:
            # Mild slope
            if tw_depth >= inlet_crown_depth_above_outlet_invert:
                self._compute_full_flow()
                return True
        else:
            # Steep slope
            if tw_depth >= inlet_crown_depth_above_outlet_invert:
                self._compute_full_flow()
                if self.inlet_control_depth + self.inlet_invert_elevation > \
                        self.outlet_control_depth + self.inlet_invert_elevation:
                    if self.flow > 0.0:
                        self.show_oscillitary_flow_note = True
                # if (self.outlet_control_depth >= rise or self.inlet_control_depth >= rise) and
                #     self.inlet_control_depth > self.outlet_control_depth and flow_type == 1:
                #     flow_type = 5
                return True

    def _convert_depths_and_distances_to_elevations_and_stations(self):
        """Converts the depths and distances to elevations and stations."""
        # We now have a good x, y water surface profile (wsp)
        # Convert distance down barrel to stations and depths to elevations
        first_station = self.input_dict['calc_data']['Upstream invert station']
        first_elevation = self.input_dict['calc_data']['Upstream invert elevation']
        change_in_station = first_station - self.input_dict['calc_data']['Downstream invert station']
        change_in_elevation = first_elevation - self.input_dict['calc_data']['Downstream invert elevation']
        self.wse_stations = []
        self.wse_elevations = []
        closed_shape = self.input_dict['calc_data']['Geometry']['calculator'].determine_if_shape_is_closed()
        rise = self.input_dict['calc_data']['Geometry']['calculator'].rise_to_crown

        prev_ff = False
        prev_x_ff = False
        first_x_ff = None
        embed_depth = self.min_embedment_depth
        for x, y in zip(self.x, self.y):
            # If we have a closed shape, don't let the depth grow greater than the culvert rise
            if closed_shape and y >= rise:
                y = rise
                prev_x_ff = x
                if prev_ff is False:
                    first_x_ff = x
                    prev_ff = True
                if x <= self.input_dict['calc_data']['Upstream invert station']:
                    self.flowing_full = True
            elif closed_shape and prev_ff:
                self.length_flowing_full = abs(prev_x_ff - first_x_ff)
                prev_ff = False
            station = first_station - change_in_station * (x / self.total_length)
            self.wse_stations.append(station)
            elevation = first_elevation + embed_depth - change_in_elevation * (
                x / self.total_length) + y + self.min_embedment_depth
            self.wse_elevations.append(elevation)
        if self.hyd_jump_exists:
            x = self.hyd_jump_station
            y = self.hyd_jump_depth + self.min_embedment_depth
            self.hyd_jump_station_absolute = first_station - change_in_station * (
                x / self.total_length)
            self.hyd_jump_elevation = first_elevation + embed_depth - change_in_elevation * (
                x / self.total_length) + y
            x += self.hyd_jump_length
            y = self.hyd_jump_sequent_depth + self.min_embedment_depth
            self.hyd_jump_final_station_absolute = first_station - change_in_station * (
                x / self.total_length)
            self.hyd_jump_sequent_elevation = first_elevation + embed_depth - change_in_elevation * (
                x / self.total_length) + y

        if self.length_flowing_full <= 0.0 or first_x_ff is None:
            self.length_flowing_full = 0.0  # No full flow actually occurred
            self.flowing_full = False

        if self.flowing_full:
            # Determine pressure head at inlet
            self.upstream_pressure_head = self.up_starting_depth + self.headloss + self.outlet_invert_elevation - \
                self.inlet_invert_elevation

    def _convert_upstream_x_y_to_match_downstream(self):
        """Convert upstream x, y to match the downstream computations."""
        new_x = []
        self.x = copy.deepcopy(self.results_upstream['x'])
        self.y = copy.deepcopy(self.results_upstream['y'])
        for x in self.x:
            new_x.append(self.total_length - x)  # Convert x to the upstream end
        self.x = new_x
        # Reverse all computed values
        self.x.reverse()
        self.y.reverse()
        for item in self.results_upstream:
            self.results_upstream[item].reverse()
        self.results_upstream['x'] = self.x
        self.results_upstream['y'] = self.y
        self.results_upstream['Depth'] = self.y

    def _update_manning_calc(self):
        """Computes the slope and updates the Manning's n calculator.

        Returns:
            bool: True if successful
        """
        # Compute and set the slope
        rise = self.input_dict['calc_data']['Upstream invert elevation'] - self.input_dict['calc_data'][
            'Downstream invert elevation']
        down_station = self.input_dict['calc_data']['Downstream invert station']
        up_station = self.input_dict['calc_data']['Upstream invert station']
        run = abs(down_station - up_station)
        if run == 0.0:
            run = 10**-10
        self.slope = rise / run

        self.manning_n_calc.input_dict['calc_data']['Slope'] = self.slope

        # Set Flow
        self.flows = self.input_dict['calc_data']['Flows']
        self.manning_n_calc.input_dict['calc_data']['Flows'] = self.input_dict['calc_data']['Flows']

        # Set Geometry
        self.manning_n_calc.input_dict['calc_data']['Geometry'] = self.input_dict['calc_data']['Geometry']
        self.manning_n_calc.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Head'] = \
            'Depth'

        # Set Composite n
        # self.manning_n_calc.input_dict['Composite n'] = self.input_dict['Composite n']
        self.manning_n_calc.input_dict['calc_data']['Composite n'] = self.input_dict['calc_data']['Composite n']
        self.manning_n_calc.input_dict['calc_data']['Geometry']['calculator'].input_dict['Composite n'] = \
            self.input_dict['calc_data']['Composite n']

        # Set unknown
        self.manning_n_calc.unknown = 'Head'
        self.manning_n_calc.input_dict['calc_data']['Calculate'] = self.manning_n_calc.unknown

        # Set length
        self.total_length = run

        self.min_embedment_depth = self.input_dict['calc_data']['Geometry']['calculator'].min_embedment_depth
        self.channel_is_embedded = self.input_dict['calc_data']['Geometry']['calculator'].channel_is_embedded

        if not hasattr(self, 'zero_tol'):
            _, self.null_data = self.get_data('Null data', -9999.0)
            _, self.zero_tol = self.get_data('Zero tolerance', 1e-6)
        self.manning_n_calc.zero_tol = self.zero_tol
        self.manning_n_calc.update_mannings_n_data()

        self.input_dict['calc_data']['Geometry']['calculator'].initialize_geometry()

        self.manning_n_calc.compute_data()

    def determine_critical_and_normal_depths_and_slope(self, index):
        """Determine the critical and normal depths and the slope.

        Args:
            index (int): The index of the flow to compute.

        Returns:
            bool: True if successful
        """
        self._update_manning_calc()
        return self._determine_direction_and_starting_depth(index)

    def _determine_direction_and_starting_depth(self, index):
        """Determine the direction and starting depths of the GVF computations.

        Returns:
            bool: True if successful
        """
        self.froude = 0.0
        self.increase_in_depth = True
        self.up_increase_in_depth = True
        self.down_increase_in_depth = True

        lowest_meaningful_tw = 0.0

        # closed_shape, rise_to_crown = self.input_dict['calc_data']['Geometry'][
        #     'calculator'].determine_if_shape_is_closed()
        closed_shape = self.input_dict['calc_data']['Geometry']['calculator'].shape_is_closed
        rise_to_crown = self.input_dict['calc_data']['Geometry']['calculator'].rise_to_crown
        self.rise = rise_to_crown

        if self.manning_n_calc.can_compute_critical:
            # If horizontal slope, no critical depth
            if 'Critical depth' in self.manning_n_calc.results and \
                    len(self.manning_n_calc.results['Critical depth']) > 0:
                self.critical_depth = self.manning_n_calc.results['Critical depth'][index]
                self.critical_flowarea = self.manning_n_calc.results['Critical flow area'][index]
                self.critical_velocity = self.manning_n_calc.results['Critical velocity'][index]
            if closed_shape and self.critical_depth > rise_to_crown:
                self.critical_depth = rise_to_crown
                if self.full_flow_area > 0.0:
                    self.critical_velocity = self.flow / self.full_flow_area

            lowest_meaningful_tw = self.critical_depth
            _, use_brink_depth = self.get_data('Use brink depth', True)
            if use_brink_depth:
                self.brink_depth_edr = self._get_brink_depth_end_depth_ratio()
                self.brink_depth = self.brink_depth_edr * self.critical_depth
                lowest_meaningful_tw = self.brink_depth

        _, horizontal_slope = self.get_data('Horizontal tolerance')
        if self.slope > horizontal_slope and self.manning_n_calc.can_compute_normal:
            self.froude = self.manning_n_calc.results['Froude number'][index]
            self.normal_depth = self.manning_n_calc.results['Depths'][index]
            self.normal_velocity = self.manning_n_calc.results['Average velocity'][index]
            if self.flow >= self.full_flow and self.flow > self.zero_tol:
                self.normal_depth = rise_to_crown
                if self.full_flow_area > 0.0:
                    self.normal_velocity = self.flow / self.full_flow_area

            if lowest_meaningful_tw > self.normal_depth:
                lowest_meaningful_tw = self.normal_depth

            if math.isclose(self.froude, 1.0, abs_tol=self.zero_tol):
                # Critical flow - Use mild slope assumptions
                self.slope_type = 'critical'
                self.direction = 'upstream'
                self.up_starting_depth, self.up_starting_velocity = \
                    self.input_dict['calc_data']['Downstream water depth']['calculator']. \
                    get_starting_water_depth_and_velocity(self.normal_depth, self.normal_velocity, self.critical_depth,
                                                          self.critical_velocity, self.flows, index=index)
                self.down_starting_depth, self.down_starting_velocity = \
                    self.input_dict['calc_data']['Upstream water depth']['calculator']. \
                    get_starting_water_depth_and_velocity(self.normal_depth, self.normal_velocity, self.critical_depth,
                                                          self.critical_velocity, self.flows, index=index)
                self.starting_depth = self.up_starting_depth

                if self.down_starting_depth < self.zero_tol:
                    self.warnings['upstream_boundary_condition'] = (
                        'Please specify a depth greater than zero for the upstream boundary condition!'
                    )
                    return False

                if self.up_starting_depth < lowest_meaningful_tw:
                    self.up_starting_depth = lowest_meaningful_tw

                # determine increase in depth for going up and downstream
                if self.up_starting_depth > self.normal_depth:
                    self.up_increase_in_depth = False
                elif self.up_starting_depth < self.normal_depth:
                    self.up_increase_in_depth = True

                if self.down_starting_depth > self.normal_depth:
                    self.down_increase_in_depth = False
                elif self.down_starting_depth < self.normal_depth:
                    self.down_increase_in_depth = True

            elif self.froude > 1.0:
                # Supercritical flow
                self.slope_type = 'steep'
                self.direction_list = 'downstream'
                self.up_starting_depth, self.up_starting_velocity = \
                    self.input_dict['calc_data']['Downstream water depth']['calculator']. \
                    get_starting_water_depth_and_velocity(self.normal_depth, self.normal_velocity, self.critical_depth,
                                                          self.critical_velocity, self.flows, index=index)
                self.down_starting_depth, self.down_starting_velocity = \
                    self.input_dict['calc_data']['Upstream water depth']['calculator']. \
                    get_starting_water_depth_and_velocity(self.normal_depth, self.normal_velocity, self.critical_depth,
                                                          self.critical_velocity, self.flows, index=index)
                self.starting_depth = self.down_starting_depth

                if self.down_starting_depth < self.zero_tol:
                    self.warnings['upstream_boundary_condition'] = (
                        'Please specify a depth greater than zero for the upstream boundary condition!'
                    )
                    return False

                if self.up_starting_depth < lowest_meaningful_tw:
                    self.up_starting_depth = lowest_meaningful_tw

                # determine increase in depth for going up and downstream
                # Prior there was a check for normal depth here, but it cannot be lower than normal depth
                self.up_increase_in_depth = False

                if self.down_starting_depth < self.normal_depth:
                    self.down_increase_in_depth = True
                else:
                    self.down_increase_in_depth = False

            else:
                # Subcritical flow
                self.slope_type = 'mild'
                self.direction = 'upstream'
                self.up_starting_depth, self.up_starting_velocity = \
                    self.input_dict['calc_data']['Downstream water depth']['calculator']. \
                    get_starting_water_depth_and_velocity(self.normal_depth, self.normal_velocity, self.critical_depth,
                                                          self.critical_velocity, self.flows, index=index)
                self.down_starting_depth, self.down_starting_velocity = \
                    self.input_dict['calc_data']['Upstream water depth']['calculator']. \
                    get_starting_water_depth_and_velocity(self.normal_depth, self.normal_velocity, self.critical_depth,
                                                          self.critical_velocity, self.flows, index=index)
                self.starting_depth = self.up_starting_depth

                if self.down_starting_depth < self.zero_tol:
                    self.warnings['upstream_boundary_condition'] = (
                        'Please specify a depth greater than zero for the upstream boundary condition!'
                    )
                    return False

                if self.up_starting_depth < lowest_meaningful_tw:
                    self.up_starting_depth = lowest_meaningful_tw

                # determine increase in depth for going up and downstream
                if self.up_starting_depth > self.normal_depth:
                    self.up_increase_in_depth = False
                elif self.up_starting_depth < self.normal_depth:
                    self.up_increase_in_depth = True

                if self.down_starting_depth > self.normal_depth:
                    self.down_increase_in_depth = False
                elif self.down_starting_depth < self.normal_depth:
                    self.down_increase_in_depth = True

            return True
        else:
            # Horizontal or adversely sloped culvert
            self.slope_type = 'adverse'
            if abs(self.slope) < horizontal_slope:
                self.slope_type = 'horizontal'

            if self.input_dict['calc_data']['Downstream water depth']['calculator'].input_dict['calc_data'][
                    'Select'] == 'normal depth':
                self.warnings['downstream_boundary_condition'] = (
                    'The downstream boundary condition is set to normal depth, but the slope is '
                    f'{self.slope_type}')
                return False

            self.direction = 'upstream'
            self.up_starting_depth, self.up_starting_velocity = \
                self.input_dict['calc_data']['Downstream water depth']['calculator']. \
                get_starting_water_depth_and_velocity(self.normal_depth, self.normal_velocity, self.critical_depth,
                                                      self.critical_velocity, self.flows, index=index)
            self.down_starting_depth, self.down_starting_velocity = \
                self.input_dict['calc_data']['Upstream water depth']['calculator']. \
                get_starting_water_depth_and_velocity(self.normal_depth, self.normal_velocity, self.critical_depth,
                                                      self.critical_velocity, self.flows, index=index)

            if self.down_starting_depth < self.zero_tol:
                self.warnings['upstream_boundary_condition'] = (
                    'Please specify a depth greater than zero for the upstream boundary condition!'
                )
                return False

            if self.up_starting_depth < lowest_meaningful_tw:
                self.up_starting_depth = lowest_meaningful_tw

            # determine increase in depth for going up and downstream
            self.up_increase_in_depth = True
            self.down_increase_in_depth = True
            return True

    def _get_brink_depth_end_depth_ratio(self):
        """Gets the brink depth end depth ratio from the input dictionary."""
        if self.input_dict['calc_data']['Geometry']['Shape'] in ['circle', 'ellipse', 'pipe arch', 'horseshoe',
                                                                 'parabola']:
            edr = 0.75
        elif self.input_dict['calc_data']['Geometry']['Shape'] in ['box', 'rectangle', 'round-cornered rectangle']:
            edr = 0.715
        else:
            _, edr = self.get_data('Brink depth end depth ratio', 0.8)
        return edr

    def _compute_good_backwater_curve(self, flow, direction, starting_depth,
                                      increase_in_depth):
        """Computes and adjusts the backwater curve until it fits several criteria.

        Args:
            flow (float): The flow value to compute for a backwater curve
            direction (string): The direction to compute: up or down.
            starting_depth (float): The depth that will be used to start the GVF curve.
            increase_in_depth (bool): Whether the depth of the GVF curve will increase or decrease along the
                calculations.

        Returns:
            True if successful, otherwise, False.
        """
        result = False
        self.increment = 0.05
        num_points_target = 20
        _, min_points_curve = self.get_data('Minimum number of points for direct step curve')
        _, max_points_curve = self.get_data('Maximum number of points for direct step curve')
        _, num_iterations = self.get_data('Max number of iterations')
        counter = 0
        while not result and counter < num_iterations:
            result = self._perform_backwater_curve(flow, direction, starting_depth, increase_in_depth)
            # Check that we don't have too few or too many points
            if result:
                if len(self.intermediate_results['x']) < min_points_curve and not self.accept_low_results:
                    self.increment = self.increment / num_points_target
                    result = False
                elif len(self.intermediate_results['x']) > max_points_curve:
                    if len(self.y) > 0:
                        change = abs(self.y[0] - self.y[-1])
                        self.increment = change / num_points_target
                        result = False
                    else:
                        # It may be that we need a smaller increment
                        result = False
                        self.increment = self.increment / num_points_target
                        if counter > num_iterations or self.increment < self.zero_tol:
                            self.warnings['backwater_curve'] = (
                                'GVF CalcData was unable to find a result')
                            return False
            else:
                # It may be that we need a smaller increment
                self.increment = self.increment / num_points_target
                if counter > num_iterations or self.increment < self.zero_tol:
                    self.warnings['backwater_curve'] = (
                        'GVF CalcData was unable to find a result')
                    return False
            counter += 1
        return True

    def _perform_backwater_curve(self, flow, direction, starting_depth,
                                 increase_in_depth):
        """Computes a single backwater curve for a given flow, direction, starting depth, and increase.

        Args:
            flow (float): The flow value to compute for a backwater curve
            direction (string): The direction to compute: up or down.
            starting_depth (float): The depth that will be used to start the GVF curve.
            increase_in_depth (bool): Whether the depth of the GVF curve will increase or decrease along the
                calculations.

        Returns:
            bool: True if successful
        """
        # Initialize and gather data
        _, num_loops = self.get_data('Max number of iterations')
        _, max_points = self.get_data('Maximum number of points for direct step curve')
        num_iterations = max(num_loops, max_points)
        # 'Composite n' change
        n = self.input_dict['calc_data']['Composite n']['calculator'].get_intermediate_composite_n()
        _, k = self.get_data('Manning constant')
        _, two_g = self.get_data('Gravity')
        two_g *= 2.0
        self.accept_low_results = False

        self.direction = direction
        self.increase_in_depth = increase_in_depth
        current_depth = starting_depth
        current_length = 0.0
        increment = self.increment
        if not self.increase_in_depth:
            increment *= -1

        # Set depth_2 to initialize depth_1 in loop, initialize depth_1 for scope
        depth_1 = current_depth
        depth_2 = current_depth

        # Clear variables
        self._clear_intermediate_results()

        # check for vena contracta
        if self.compute_vena_contracta and direction == 'downstream' and self.vena_contracta_length > 0.0:
            # flow_area_1, wetted_perimeter_1, top_width_1, hydraulic_radius_1, manning_n_1, velocity_1, energy_1 \
            #     = self._compute_direct_step_variables(flow, n, depth_1, two_g)
            # friction_slope_1 = self._compute_friction_slope(flow, manning_n_1, k, flow_area_1, hydraulic_radius_1)
            _ = self._compute_and_assign_delta_x_for_single_direct_step(
                flow, n, k, starting_depth, self.vena_contracta_depth, self.vena_contracta_length,
                two_g, True, 0)
            self.vena_contracta_x = [current_length, current_length + self.vena_contracta_length]
            self.intermediate_results['x'] = [current_length]
            if self.input_dict['calc_data']['Display channel slope as flat']:
                self.vena_contracta_y = [starting_depth, self.vena_contracta_depth]
            else:
                self.inlet_invert_elevation = self.input_dict['calc_data']['Upstream invert elevation']
                self.vena_contracta_y = [
                    starting_depth + self.inlet_invert_elevation + self.min_embedment_depth,
                    self.vena_contracta_depth + self.inlet_invert_elevation + self.min_embedment_depth
                    - self.slope * self.vena_contracta_length]
            depth_2 = self.vena_contracta_depth
            current_length = self.vena_contracta_length
        else:
            self.vena_contracta_x = []
            self.vena_contracta_y = []

        # Check if this is a full flow profile:
        if self.flowing_full:
            if self.direction == 'downstream' or self.direction == 'upstream' and current_depth >= self.rise:
                _ = self._compute_and_assign_delta_x_for_single_direct_step(
                    flow, n, k, self.rise, self.rise, current_length, two_g,
                    True, 1)
                self.intermediate_results['x'][-1] = self.total_length
                self.accept_low_results = True
                return True

        # Check if we are starting with a normal depth - if so, that makes it easy
        if direction == 'downstream' and math.isclose(current_depth, self.normal_depth, abs_tol=self.zero_tol):
            _ = self._compute_and_assign_delta_x_for_single_direct_step(
                flow, n, k, current_depth, current_depth, current_length,
                two_g, True, 1)
            self.intermediate_results['x'][-1] = self.total_length
            self.accept_low_results = True
            return True

        # Check if we are starting with zero depth
        if math.isclose(current_depth, 0.0, abs_tol=self.zero_tol):
            _ = self._compute_and_assign_delta_x_for_single_direct_step(
                flow, n, k, current_depth, 0.0, current_length, two_g, True, 1)
            self.intermediate_results['x'][-1] = self.total_length
            self.accept_low_results = True

        # Check if we start with normal Depth
        if abs(depth_1 - self.normal_depth) < self.increment:
            delta_x = self._compute_and_assign_delta_x_for_single_direct_step(
                flow, n, k, current_depth, self.normal_depth, current_length, two_g, True, 1)
            # self.intermediate_results['x'] = [current_length, self.total_length]
            # self.intermediate_results['y'] = [depth_1, depth_2]
            if self.intermediate_results['x'][-1] >= self.total_length or delta_x <= self.zero_tol:
                self.intermediate_results['x'][-1] = self.total_length
            else:
                _ = self._compute_and_assign_delta_x_for_single_direct_step(
                    flow, n, k, self.normal_depth, self.normal_depth, current_length, two_g, True, 0)
                self.intermediate_results['x'][-1] = self.total_length
            self.accept_low_results = True
            return True

        delta_x = 0.0
        count = 0
        while current_length < self.total_length and num_iterations > count:
            depth_1 = depth_2
            depth_2 = depth_1 + increment

            # Check if we have reached zero depth
            if depth_2 <= 0.0:
                delta_x = self._compute_and_assign_delta_x_for_single_direct_step(
                    flow, n, k, depth_1, 0.0, current_length, two_g, True, 1)
                # Replace the last length with the total channel length
                if len(self.intermediate_results['x']) > 0:
                    current_length = self.total_length
                    self.intermediate_results['x'][-1] = current_length
                    break

            delta_x = self._compute_and_assign_delta_x_for_single_direct_step(
                flow, n, k, depth_1, depth_2, current_length, two_g, True, 0)

            current_length += abs(delta_x)

            # Check if we have reached normal Depth
            if abs(depth_2 - self.normal_depth) < self.increment and current_length < self.total_length:
                # Check if we continue the curve to the normal depth or just jump to it (as next increment would pass
                # over it)
                add_first_depth = True
                first_depth = depth_2
                second_depth = self.normal_depth
                delta_x = self._compute_and_assign_delta_x_for_single_direct_step(
                    flow, n, k, first_depth, second_depth, current_length, two_g, add_first_depth, 2)
                # Replace the last length with the total channel length
                if len(self.intermediate_results['x']) > 0:
                    current_length = self.total_length
                    self.intermediate_results['x'][-1] = current_length
                    # Check if we extended past the channel length; if so, adjust, depending on which index passed it
                    # Note, we correct too much length later, but if the y values plateau, it can throw it off
                    if self.intermediate_results['x'][-2] > self.total_length:
                        self._remove_last_assignment()

            count += 1

        # Check and correct the assigned data
        if self.intermediate_results['x'][-1] > self.total_length:
            x1 = self.intermediate_results['x'][-2]
            x2 = self.intermediate_results['x'][-1]
            y1 = self.intermediate_results['y'][-2]
            y2 = self.intermediate_results['y'][-1]
            depth_2 = y1 + (self.total_length - x1) / (x2 - x1) * (y2 - y1)
            self.intermediate_results['x'][-1] = self.total_length
            self.intermediate_results['y'][-1] = depth_2

        # If we have overshot the end, Determine the depth at the end
        elif current_length > self.total_length:
            x1 = self.intermediate_results['x'][-1]
            x2 = current_length
            y1 = self.intermediate_results['y'][-1]
            y2 = depth_2
            depth_2 = y1 + (self.total_length - x1) / (x2 - x1) * (y2 - y1)
            _ = self._compute_and_assign_delta_x_for_single_direct_step(
                flow, n, k, depth_1, depth_2, current_length, two_g, False, 1)
            self.intermediate_results['x'][-1] = self.total_length

        return True

    def compute_vena_contracta_data(self):
        """Compute the vena contracta if it exists."""
        self.closed_shape = self.input_dict['calc_data']['Geometry']['calculator'].determine_if_shape_is_closed()
        self.rise = self.input_dict['calc_data']['Geometry']['calculator'].rise_to_crown
        if not self.compute_vena_contracta or not self.closed_shape:
            return
        self.vena_contracta_coefficient = 0.7
        slope_correction_term = 0.5
        culv_shape = self.input_dict['calc_data']['Geometry']['Shape']

        if culv_shape in ['box', 'low_profile_box', 'conspan', 'south_dakota_box']:
            self.vena_contracta_coefficient = 0.8

        inlet_water_depth_vc = copy.copy(self.down_starting_depth)
        if self.inlet_control_depth > 0.0 and self.inlet_control_depth < self.rise:
            # TODO: Ensure that this is properly set by culvert calc
            inlet_water_depth_vc = self.inlet_control_depth

        self.vena_contracta_length = slope_correction_term * self.rise
        self.vena_contracta_depth = self.vena_contracta_coefficient * inlet_water_depth_vc

        # if self.vena_contracta_depth < self.criticalDepth:
        #     if self.down_increase_in_depth:
        #         # All is good
        #         pass
        #     else:
        #         if self.hydJumpCheckHasSameIncreaseDepth:
        #             self.hydJumpCheckHasSameIncreaseDepth = False
        #         else:
        #             # All is good
        #             pass
        #     self.increasingDepth = True
        # else:
        #     if self.hydJumpCheckHasSameIncreaseDepth:
        #         # All is good
        #         pass
        #     else:
        #         self.hydJumpCheckHasSameIncreaseDepth = True
        #     self.increasingDepth = False

    def determine_flow_profile(self, inlet_control_depth=None, outlet_control_depth=None):
        """Determine the flow profile."""
        # HW < D: 1, 2, 3
        # HW > D: 5, 4, 6, 7
        # Flow Profile
        # Inlet depth: n: normal, c: critical, f:Full
        # Outlet depth: n: normal, c: critical/brink, tw: TailWater, f:Full

        # TODO: Verify that we set the self.start_station - rename to first_station? - and that we add this to the x
        # in results (though, may just add a new row of results)
        # TODO: Verify that we handle inlet_control_depth from culvert code and outlet_control_depth (from this code)

        if self.hyd_jump_exists:
            up_regime, tw = self._determine_flow_profile_from_intermediate_results(
                self.results_upstream, self.start_station, self.hyd_jump_station)
            down_regime, _ = self._determine_flow_profile_from_intermediate_results(
                self.results_downstream, self.hyd_jump_station + self.hyd_jump_length,
                self.start_station + self.total_length)
            self.flow_profile = f'{down_regime}-J-{up_regime}{tw}'
        elif self.solution == 'upstream direction':
            up_regime, tw = self._determine_flow_profile_from_intermediate_results(
                self.results_upstream, self.start_station, self.start_station + self.total_length)
            self.flow_profile = f'{up_regime}{tw}'
        elif self.solution == 'downstream direction':
            down_regime, tw = self._determine_flow_profile_from_intermediate_results(
                self.results_downstream, self.start_station, self.start_station + self.total_length)
            self.flow_profile = f'{down_regime}{tw}'
        # I believe this is a faulty HY-8 artifact that does not apply to this updated code
        # else:
        #     # Need to handle this case where 2 curves joined without a jump
        #     pass

        self.usgs_type = self._determine_usgs_flow_type(inlet_control_depth, outlet_control_depth, self.flow_profile,
                                                        tw)

        if self.usgs_type is not None:
            self.flow_profile = f'{self.usgs_type}-{self.flow_profile}'

        self.flow_profiles.append(self.flow_profile)
        self.usgs_types.append(self.usgs_type)

    def _determine_usgs_flow_type(self, inlet_control_depth: float, outlet_control_depth: float, flow_profile: str,
                                  tw: str) -> int:
        """Determine the USGS flow type."""
        usgs_type = None
        if inlet_control_depth is None or outlet_control_depth is None:
            return usgs_type
        if inlet_control_depth > outlet_control_depth:
            # Inlet control governs
            if inlet_control_depth >= self.rise - self.zero_tol:
                # Inlet submerged
                usgs_type = 5
            else:
                # Inlet NOT submerged
                usgs_type = 1
        else:  # Outlet control governs
            if outlet_control_depth >= self.rise - self.zero_tol:
                # Inlet submerged
                if 'S1' in flow_profile:
                    usgs_type = 5
                elif 'FF' in flow_profile and tw == 'f':
                    usgs_type = 4
                elif 'FF' in flow_profile:
                    usgs_type = 6
                else:
                    usgs_type = 7
            else:
                # Inlet NOT submerged
                if 'S1' in flow_profile:
                    usgs_type = 1
                elif tw == 'c':
                    usgs_type = 2
                else:
                    usgs_type = 3
        return usgs_type

    def _determine_flow_profile_from_intermediate_results(self, results_dict: dict, min_station: float,
                                                          max_station: float) -> tuple:
        """Determine the flow profile from the intermediate results."""
        ave_depth = 0.0  # Determine the weighted average depth
        for index in range(1, len(results_dict['x'])):
            if min_station <= results_dict['x'][index - 1] <= max_station or \
                    min_station <= results_dict['x'][index] <= max_station:
                delta_x = results_dict['x'][index] - results_dict['x'][index - 1]
                depth = results_dict['Depth'][index]
                ave_depth += depth * delta_x
        ave_depth /= (max_station - min_station)

        regime_str = ''
        if self.closed_shape and self.length_flowing_full > 0.0:
            regime_str = 'FF'
        else:
            if self.slope_type == 'mild':
                regime_str = 'M'
                if self.normal_depth < ave_depth:
                    regime_str += '1'
                elif self.critical_depth <= ave_depth <= self.normal_depth:
                    regime_str += '2'
                elif ave_depth < self.critical_depth:
                    regime_str += '3'
            elif self.slope_type == 'steep':
                regime_str = 'S'
                if self.critical_depth < ave_depth:
                    regime_str += '1'
                elif self.normal_depth <= ave_depth <= self.critical_depth:
                    regime_str += '2'
                elif ave_depth < self.normal_depth:
                    regime_str += '3'
            elif self.slope_type == 'critical':
                regime_str = 'C'
                if self.critical_depth < ave_depth:
                    regime_str += '1'
                else:
                    regime_str += '3'
            elif self.slope_type == 'adverse':
                regime_str = 'A'
                if self.critical_depth < ave_depth:
                    regime_str += '2'
                elif ave_depth < self.critical_depth:
                    regime_str += '3'
            elif self.slope_type == 'horizontal':
                regime_str = 'H'
                if self.critical_depth < ave_depth:
                    regime_str += '2'
                elif ave_depth < self.critical_depth:
                    regime_str += '3'

        depth_tol = 0.01  # Use a tolerance that the UI would not show a difference
        final_depth = results_dict['Depth'][-1]
        if self.closed_shape and self.rise - self.zero_tol <= final_depth:
            tw_str = 'f'
        # Critical depth more important than normal depth, so check it first (particularly for critical slope)
        elif math.isclose(final_depth, self.critical_depth, abs_tol=depth_tol):
            tw_str = 'c'
        elif math.isclose(final_depth, self.brink_depth, abs_tol=depth_tol):
            tw_str = 'c'  # Call brink depth critical for flow profile purposes
        elif math.isclose(final_depth, self.normal_depth, abs_tol=depth_tol):
            tw_str = 'n'
        else:
            tw_str = 'tw'

        return regime_str, tw_str

    def _clear_intermediate_results(self):
        """Remove the last assignment from the intermediate results."""
        for key in self.intermediate_results:
            self.intermediate_results[key] = []

    def _remove_last_assignment(self):
        """Remove the last assignment from the intermediate results."""
        for key in self.intermediate_results:
            if len(self.intermediate_results[key]) > 0:
                self.intermediate_results[key] = self.intermediate_results[key][:-1]

    def _compute_and_assign_delta_x_for_single_direct_step(self, flow, n, k, depth_1, depth_2, current_length, two_g,
                                                           add_1_to_list, times_to_add_2_to_list):
        """Compute the change in x for a single direct step.

        Args:
            flow (float): amount of flow in channel
            n (float): Manning's n variable to describe the roughness
            k (float): Unit conversion for Manning's n equation.
            depth_1 (float): depth at location 1
            depth_2 (float): depth at location 2
            current_length (float): current length along GVF calculations.
            two_g (float): 2 * g; used frequently in the direct step calculations.
            add_1_to_list (bool): add the results of location 1 to the results.
            times_to_add_2_to_list (int): number of times to add location 2 to the list.

        Returns:
            delta_x (float): incremental change in x for a single direct step.
        """
        delta_x = 0.0
        warning = False

        flow_area_1, wetted_perimeter_1, top_width_1, hydraulic_radius_1, manning_n_1, velocity_1, energy_1 \
            = self._compute_direct_step_variables(flow, n, depth_1, two_g)
        friction_slope_1 = self._compute_friction_slope(flow, manning_n_1, k, flow_area_1, hydraulic_radius_1)

        flow_area_2, wetted_perimeter_2, top_width_2, hydraulic_radius_2, manning_n_2, velocity_2, energy_2 \
            = self._compute_direct_step_variables(flow, n, depth_2, two_g)
        friction_slope_2 = self._compute_friction_slope(flow, manning_n_2, k, flow_area_2, hydraulic_radius_2)

        energy_loss = energy_2 - energy_1
        if energy_loss < 0.0:
            if self.direction == 'downstream':
                warning = True  # Energy loss is negative!
        else:
            if self.direction == 'upstream':
                warning = True  # Energy loss is NOT negative!
        friction_slope_ave = (friction_slope_1 + friction_slope_2) / 2.0
        denominator = self.slope - friction_slope_ave
        if denominator != 0.0:
            delta_x = energy_loss / (denominator)
        else:
            delta_x = 0.0
        if delta_x < 0.0:
            if self.direction == 'downstream':
                warning = True  # delta_x loss is negative!
        else:
            if self.direction == 'upstream':
                warning = True  # delta_x loss is NOT negative!

        if warning:
            pass
        # If we are not adding the first element but we are adding the second element, we just want the hydraulic
        # parameters to finish the curve
        if not add_1_to_list and times_to_add_2_to_list and len(
                self.intermediate_results['x']) > 0:
            delta_x = current_length - self.intermediate_results['x'][-1]

        if add_1_to_list:
            self.intermediate_results['flow_area'].append(flow_area_1)
            self.intermediate_results['wetted_perimeter'].append(wetted_perimeter_1)
            self.intermediate_results['top_width'].append(top_width_1)
            self.intermediate_results['hydraulic_radius'].append(hydraulic_radius_1)
            self.intermediate_results['manning_n'].append(manning_n_1)
            self.intermediate_results['Velocity'].append(velocity_1)
            self.intermediate_results['Energy'].append(energy_1)
            self.intermediate_results['energy_loss'].append(energy_loss)
            self.intermediate_results['energy_slope'].append(friction_slope_1)
            self.intermediate_results['friction_slope_ave'].append(friction_slope_ave)
            self.intermediate_results['delta_x'].append(delta_x)
            self.intermediate_results['x'].append(current_length)
            self.intermediate_results['y'].append(depth_1)
            self.intermediate_results['Depth'].append(depth_1)
            sequent_depth_1, froude_1 = self.compute_sequent_depth_and_froude(depth_1, flow)
            self.intermediate_results['sequent_depth'].append(sequent_depth_1)
            self.intermediate_results['froude'].append(froude_1)

        for _ in range(times_to_add_2_to_list):
            self.intermediate_results['flow_area'].append(flow_area_2)
            self.intermediate_results['wetted_perimeter'].append(wetted_perimeter_2)
            self.intermediate_results['top_width'].append(top_width_2)
            self.intermediate_results['hydraulic_radius'].append(hydraulic_radius_2)
            self.intermediate_results['manning_n'].append(manning_n_2)
            self.intermediate_results['Velocity'].append(velocity_2)
            self.intermediate_results['Energy'].append(energy_2)
            self.intermediate_results['energy_loss'].append(energy_loss)
            self.intermediate_results['energy_slope'].append(friction_slope_2)
            self.intermediate_results['friction_slope_ave'].append(friction_slope_ave)
            self.intermediate_results['delta_x'].append(delta_x)
            self.intermediate_results['x'].append(current_length + abs(delta_x))
            self.intermediate_results['y'].append(depth_2)
            self.intermediate_results['Depth'].append(depth_2)
            sequent_depth_2, froude_2 = self.compute_sequent_depth_and_froude(depth_2, flow)
            self.intermediate_results['sequent_depth'].append(sequent_depth_2)
            self.intermediate_results['froude'].append(froude_2)

        return delta_x

    def _compute_direct_step_variables(self, flow, n, depth, two_g):
        """Compute the variables needed at one point for a direct step computation.

        Args:
            flow (float): amount of flow in channel
            n (float): Manning's n variable to describe the roughness
            depth (float): depth at location
            two_g (float): 2 * g; used frequently in the direct step calculations.

        Returns:
            flow_area (float): The flow area of the channel with the given depth
            wetted_perimeter (float): The wetted perimeter of the channel with the given depth
            top_width (float): The top width of the channel with the given depth
            hydraulic_radius (float): The hydraulic radius of the channel with the given depth
            n (float): The manning's n of the channel with the given depth
            velocity (float): The velocity of the channel with the given depth
            energy (float): The energy of the channel with the given depth

        """
        if depth < self.zero_tol:
            return 0.0, 0.0, 0.0, 0.0, n, 0.0, 0.0

        flow_area, wetted_perimeter, top_width = self.manning_n_calc.input_dict['calc_data']['Geometry'][
            'calculator'].compute_area_perimeter_top_width(depth)

        hydraulic_radius = flow_area / wetted_perimeter
        # May need to add Manning n computation here
        velocity = flow / flow_area
        energy = depth + velocity**2.0 / two_g

        return flow_area, wetted_perimeter, top_width, hydraulic_radius, n, velocity, energy

    @staticmethod
    def _compute_friction_slope(flow, n, k, flow_area, hydraulic_radius):
        """Compute the Friction Slope for a given flow, n, k, flow area, and hydraulic Radius.

        Args:
            flow (float): The flow of the channel
            n (float): The manning's n of the channel with the given depth
            k (float): Unit conversion for Manning's n equation.
            flow_area (float): The flow area of the channel with the given depth
            hydraulic_radius (float): The hydraulic radius of the channel with the given depth

        Returns:
            friction_slope (float): the slope of the friction
        """
        if flow_area <= 0.0:
            return 0.0
        # Sf = (Qn/kAR^(2/3))^2
        friction_slope = (flow * n / (k * flow_area * hydraulic_radius**(2 / 3)))**2
        return friction_slope

    def compute_sequent_depth_and_froude(self, depth, flow):
        """Computes the sequent depth for a given depth.

        Args:
            depth (float): the original depth computed by the direct step calculator.
            flow (float): the flow for the channel.

        Returns:
            sequent_depth_1 (float): The computed sequent depth from the original depth
            froude_1 (float): The froude number of the original depth
        """
        geometry = self.manning_n_calc.input_dict['calc_data']['Geometry']['calculator']
        sequent_depth_1, froude_1 = geometry.compute_sequent_depth_and_froude(depth, flow)
        return sequent_depth_1, froude_1

    def _check_for_hydraulic_jump(self):
        """Check for a hydraulic jump by comparing the upstream's sequent depth and the downstream's depth."""
        # Next items are using the convention of the arc that will be used up or downstream of hydraulic jump
        # as opposed to the direction of computation as used elsewhere in this calculator
        if self.slope_type == 'critical':
            return  # No hydraulic jump possible on critical slope
        upstream_x = self.results_downstream['x']
        upstream_depth = self.results_downstream['Depth']
        upstream_froudes = self.results_downstream['froude']
        downstream_x = self.results_upstream['x']
        downstream_depth = self.results_upstream['Depth']

        sequent_x = self.results_downstream['x']
        sequent_y = self.results_downstream['y']
        sequent_depth = self.results_downstream['sequent_depth']
        depth_x = self.results_upstream['x']
        depth_y = self.results_upstream['Depth']

        self.join_curves = False
        self.hyd_jump_exists = False
        self.hyd_jump_swept_out = False
        self.froude_class = ''

        valid_found = any(
            (not math.isclose(val, self.null_data, abs_tol=self.zero_tol)) and val > 0.0
            for val in sequent_depth
        )
        if not valid_found:
            # All values of sequent depth are null, zero, or negative
            return

        if self.flowing_full:
            if self.up_starting_depth >= self.rise:
                # The culvert is flowing full and the outlet is submerged,
                # so the culvert is flowing full along full length
                return
            # There is a chance that we should join the curves and not have the full length of barrel flowing full

        depth_interp = Interpolation(depth_x, depth_y, null_data=self.null_data, zero_tol=self.zero_tol)

        crossings = []
        all_delta_s = []
        delta_s_ave = 0.0
        num_delta_s = 0
        prev_diff = None
        prev_d_index = 0
        delta_s = 0.0

        for s_index in range(len(sequent_x)):
            if self.compute_vena_contracta and sequent_x[s_index] < self.vena_contracta_length:
                continue
            depth_at_x, d_index = depth_interp.interpolate_y(sequent_x[s_index])
            diff = sequent_depth[s_index] - depth_at_x
            if s_index > 0 and not math.isclose(sequent_depth[s_index], self.null_data, abs_tol=self.zero_tol) and \
                    not math.isclose(sequent_depth[s_index - 1], self.null_data, abs_tol=self.zero_tol):
                delta_s = abs(sequent_depth[s_index] - sequent_depth[s_index - 1])
                all_delta_s.append(delta_s)
                delta_s_ave += delta_s
                num_delta_s += 1

            if prev_diff is not None:
                # Check if sign changed (crossing occurred)
                if (prev_diff > 0 and diff < 0) or (prev_diff < 0 and diff > 0):
                    # Store indexes before and after crossing
                    last_crossing = {
                        's_index_before': s_index - 1,
                        'd_index_before': prev_d_index,
                        's_index_after': s_index,
                        'd_index_after': d_index,
                        'delta_s': delta_s
                    }
                    crossings.append(last_crossing)
            prev_diff = diff
            prev_d_index = d_index

        if len(crossings) == 0:
            # The sequent curve never crosses the depth curve
            return

        # Assess the crossing and take only good crossings
        delta_s_std = delta_s_ave
        if num_delta_s > 1:
            delta_s_ave /= num_delta_s if num_delta_s > 0 else 1
            delta_s_std = statistics.stdev(all_delta_s)
        num_std_dev = 3
        accepted_crossings = []
        for crossing in crossings:
            if abs(crossing['delta_s'] - delta_s_ave) <= num_std_dev * delta_s_std:
                accepted_crossings.append(crossing)

        if len(accepted_crossings) == 0:
            # The sequent curve never crosses the depth curve
            return

        # take the last good crossing
        s_index_before = accepted_crossings[-1]['s_index_before']
        s_index_after = accepted_crossings[-1]['s_index_after']
        d_index_before = accepted_crossings[-1]['d_index_before']
        d_index_after = accepted_crossings[-1]['d_index_after']

        s_index_before = 0 if s_index_before < 2 else s_index_before - 2
        d_index_before = 0 if d_index_before < 2 else d_index_before - 2
        s_index_after = len(sequent_x) - 1 if s_index_after > len(sequent_x) - 2 else s_index_after + 1
        d_index_after = len(depth_x) - 1 if d_index_after > len(depth_x) - 2 else d_index_after + 1

        short_sequent_x = sequent_x[s_index_before:s_index_after + 1]
        short_sequent_y = sequent_y[s_index_before:s_index_after + 1]
        short_sequent_depth = sequent_depth[s_index_before:s_index_after + 1]
        short_depth_x = depth_x[d_index_before:d_index_after + 1]
        short_depth_y = depth_y[d_index_before:d_index_after + 1]

        # Replace null values in short_sequent_depth with corresponding values from sequent_y
        for i in range(len(short_sequent_depth)):
            if math.isclose(short_sequent_depth[i], self.null_data, abs_tol=self.zero_tol):
                short_sequent_depth[i] = short_sequent_y[i]

        x_at_froude17 = None

        for s_index in range(len(short_sequent_x) - 1):
            for d_index in range(len(short_depth_x) - 1):
                # If sequent depth and downstream depth curves cross, we have an hydraulic jump
                if do_intersect_xy(
                        short_sequent_x[s_index], short_sequent_depth[s_index], short_sequent_x[s_index + 1],
                        short_sequent_depth[s_index + 1], short_depth_x[d_index], short_depth_y[d_index],
                        short_depth_x[d_index + 1], short_depth_y[d_index + 1]):
                    # Hydraulic Jump Occurs!
                    self.hyd_jump_exists = True
                    self.hyd_jump_station, self.hyd_jump_sequent_depth, intersect = find_intersection_xy(
                        short_sequent_x[s_index], short_sequent_depth[s_index], short_sequent_x[s_index + 1],
                        short_sequent_depth[s_index + 1], short_depth_x[d_index], short_depth_y[d_index],
                        short_depth_x[d_index + 1], short_depth_y[d_index + 1])
                    if min(upstream_froudes) <= 1.7 <= max(upstream_froudes):
                        rev_froude_interp = Interpolation(upstream_froudes, sequent_x, null_data=self.null_data,
                                                          zero_tol=self.zero_tol)
                        x_at_froude17, _ = rev_froude_interp.interpolate_y(1.7)
                        self.froude_jump = True
                        # The next code is from HY-8, but we cannot reach it, because hyd_jump_exists is set True above
                        # and is only changed by compute_hyd_jump_length below.
                        # if not self.hyd_jump_exists:
                        #     self.hyd_jump_exists = True
                        #     self.hyd_jump_station = x_at_froude17
                    # This is logic from HY-8; I do not think it applies, because HY-8 would put non-sequent depths as
                    # the sequent depth when the flow was subcritical and this code uses null-data in that circumstance
                    # so this code should never have sequent depth and sequent y match
                    # # Check if this curves meeting or hydraulic jump
                    # if short_sequent_depth[s_index] == short_sequent_y[s_index]:
                    #     # No jump, but curves do meet
                    #     self.join_curves = True
                    #     self.hyd_jump_exists = False
                    #     continue
                    # Determine the upstream depth and Froude number for jump length computations
                    y_interp = Interpolation(
                        upstream_x, upstream_depth, null_data=self.null_data, zero_tol=self.zero_tol)
                    froude_interp = Interpolation(
                        upstream_x, upstream_froudes, null_data=self.null_data, zero_tol=self.zero_tol)
                    self.hyd_jump_depth, _ = y_interp.interpolate_y(self.hyd_jump_station)
                    upstream_froude, _ = froude_interp.interpolate_y(self.hyd_jump_station)
                    self.froude_class, self.froude_class_note = self.determine_jump_class(upstream_froude,
                                                                                          classification='Classical')

                    # Determine jump length
                    self.hyd_jump_length = self.compute_hyd_jump_length(upstream_froude)

                    # Determine the WSE at the end of the hydraulic jump
                    if self.hyd_jump_exists:
                        down_interp = Interpolation(
                            downstream_x, downstream_depth, null_data=self.null_data, zero_tol=self.zero_tol)
                        self.hyd_jump_sequent_depth, _ = down_interp.interpolate_y(
                            self.hyd_jump_station + self.hyd_jump_length)

                    closed_shape = self.input_dict['calc_data']['Geometry']['calculator'].determine_if_shape_is_closed()
                    rise = self.input_dict['calc_data']['Geometry']['calculator'].rise_to_crown
                    if closed_shape and self.hyd_jump_sequent_depth > rise:
                        r_x, r_y, intersect = find_intersection_xy(
                            self.hyd_jump_station, self.hyd_jump_depth, self.hyd_jump_station + self.hyd_jump_length,
                            self.hyd_jump_sequent_depth, self.hyd_jump_station, rise,
                            self.hyd_jump_station + self.hyd_jump_length, rise)
                        self.hyd_jump_length = r_x - self.hyd_jump_station
                        self.hyd_jump_sequent_depth = r_y
                    return  # Once we have found a jump, we need to end the search
        return  # Function return (so we can set a break point here, in case there is no jump found)

    def determine_jump_class(self, froude, classification='Classical'):
        """Determine the Froude class of a given Froude number.

        Args:
            froude (float): The Froude number to classify
            classification (str): The classification system to use.

        Returns:
            str: The Froude class as a string
        """
        # Table A-1 from Nathan Lowe's Thesis, pdf page 91
        if classification == 'Classical':
            if 1.0 < froude <= 1.7:
                return 'Undular Jump', 'Slight difference between y1 and y2, separated by a long transition of ' \
                       'standing waves <5% energy dissipation'
            elif 1.7 < froude <= 2.5:
                return 'A: “Weak Jump” or “Prejump Stage”', 'Smooth surface, fairly uniform velocities throughout, ' \
                    'with a series of small surface rollers 5 to 15% energy dissipation'
            elif 2.5 < froude <= 4.5:
                return 'B: “Oscillating Jump” or “Transition Stage”', 'Entering jet oscillates from bottom to ' \
                    'surface with no regular period, causing undesirable and erosive surface waves that can ' \
                    'travel far downstream 15 to 45% energy dissipation'
            elif 4.5 < froude <= 9.0:
                return 'C: “Steady” or “Well-Balanced” Jump', 'Stable jump, in which the jet consistently leaves the ' \
                       'bottom near end of the surface roller, and the downstream water surface is relatively smooth ' \
                       '45 to 70% energy dissipation'
            elif froude > 9.0:
                return 'D: “Strong” or “Rough” Jump', 'Sensitive and unpredictable jump, in which "slugs of water ' \
                       'rolling down the front face of the jump intermittently fall into the high-velocity jet ' \
                       'generating additional waves downstream, and a rough surface can prevail" 70 to 85% energy ' \
                       'dissipation'
        # Table A-2 from Nathan Lowe's Thesis, pdf page 91
        elif classification == 'Sloped':
            tol = 0.01 * self.total_length  # Arbitrary decision to consider anything 1% of total length to be at end
            if self.slope_type == 'horizontal' and self.hyd_jump_station < self.total_length and \
                    self.hyd_jump_station + self.hyd_jump_length < self.total_length:
                return 'Type A: Jump over horizontal channel', 'Forms entirely on horizontal section Common form for ' \
                       'horizontal channels Tends to move downstream'
            # TODO: should check that next section is horizontal...
            elif self.slope_type in ['steep', 'mild', 'critical'] and self.hyd_jump_station < self.total_length and \
                    self.hyd_jump_station + self.hyd_jump_length > self.total_length:
                return 'Type B: Jump over a break', 'Begins on steep positive slope and ends on horizontal section'
            elif self.slope_type in ['steep', 'mild', 'critical'] and self.hyd_jump_station < self.total_length and \
                    self.total_length - tol <= self.hyd_jump_station + self.hyd_jump_length <= self.total_length + tol:
                return 'Type C: Jump at transition', 'Begins on steep positive slope and ends at the transition ' \
                    'between steep and horizontal sections'
            elif self.slope_type in ['steep'] and self.hyd_jump_station < self.total_length and \
                    self.hyd_jump_station + self.hyd_jump_length < self.total_length + tol:
                return 'Type D: Jump on steep slope', 'Located entirely on steep positive slope Most common form ' \
                       'used for jumps in sloping channels'
            elif self.slope_type == 'mild' and self.hyd_jump_station < self.total_length and \
                    self.hyd_jump_station + self.hyd_jump_length < self.total_length + tol:
                return 'Type E: Jump on mild slope', 'Located entirely on mild positive slope Uncommon, since ' \
                    'supercritical flows do not naturally form on mild slopes'
            elif self.slope_type == 'adverse' and self.hyd_jump_station < self.total_length and \
                    self.hyd_jump_station + self.hyd_jump_length < self.total_length + tol:
                return 'Type F: Jump on adverse slope', 'Forms on adverse (negative) slopes Uncommon Unstable and ' \
                    'almost impossible to control Tends to move downstream'
        # Table A-3 from Nathan Lowe's Thesis
        elif 'Positive step' in classification:
            pass  # Not implemented yet
            # If the entire jump is before the step:
            #     return 'Type A', 'Begins upstream of the step, and ends at the step Highest tailwater before jumps
            #       moves upstream'
            # If the jump starts before the step and ends after the step:
            #     return 'Type B', 'Begins upstream of the step and ends downstream Lower tailwater than A-jumps, but
            #       higher than W-jumps'
            # If the entire jump is after the step:
            #     return 'Type C', 'Forms a standing wave that passes over the step May or may not be aerated Entirely
            #       supercritical flow, and poor energy dissipation Lower tailwater than B-jumps'

        # Table A-4 from Nathan Lowe's Thesis
        elif 'Negative step' in classification:
            pass
            # Not implemented yet
            # If the entire jump is before the step:
            #     return 'Type A', 'Begins upstream of the step, and ends at the step Highest tailwater before
            #       jump moves upstream'
            # If the ?:
            #     return 'Type W', 'Forms a standing wave that begins at the step Lower tailwater than A-jumps,
            #       but higher than B-jumps'
            # If the jump starts at the step:
            #     return 'Type B', 'Begins at the step and ends downstream Lower tailwater than W-jumps'
            # If the entire jump is after the step:
            #     return 'Type C', 'Begins downstream of step Lowest tailwater before jump moves downstream'

        # Table A-5 from Nathan Lowe's Thesis
        elif 'Submerged sill' in classification:
            pass
            # Not implemented yet
            # A or I: Begins upstream of the sill, and ends at the sill Highest tailwater before jump moves upstream
            # B or I/III: Lower tailwater than A-jumps, but higher than C-jumps Shorter and less stable than A-jumps
            # Surface boil appears at sill
            # Min B or II*: Lowest tailwater before main flow begins striking channel bottom Plunging after the sill,
            # followed by a second surface roller
            # C or IV: Lower tailwater than B-jumps, but higher than W-jumps Heavy plunging and striking of main flow
            # on channel bottom, causing erosion in unprotected channels
            # W or VI: Forms a standing wave that passes over the sill Lower tailwater than C-jumps Entirely
            # supercritical flow, and poor energy dissipation

        # Table A-6 from Nathan Lowe's Thesis
        elif 'Weir Jump' in classification:
            pass
            # Not implemented yet
            # A: Swept-out jump Lower tailwater than B-jumps Similar to classical jumps
            # B: Optimum jump Higher tailwater than A-jumps, but lower than C-jumps Jump toe is ideally located
            # immediately downstream of the point of nappe impact, to minimize channel protection
            # C: Plunging nappe Higher tailwater than B-jumps, but lower than D-jumps Jump is pushed against the weir
            # and the nappe plunges, submerging the jump and creating an aerated “hydraulic”
            # D: Surface nappe Higher tailwater than C-jumps Weir is immersed completely, the nappe stays at the
            # surface, and no jump occurs

        # Table A-7 from Nathan Lowe's Thesis
        elif 'Abruptly Expanded Channel' in classification:
            pass
            # Not implemented yet
            # R: “Repelled” Jump Supercritical flow is swept out into expanded channel, where toe forms at point „P‟
            # Limiting condition, where jump cannot move further upstream without collapsing into an S-Jump
            # S: “Spatial” Jump Oscillatory asymmetric extension of supercritical flow into expanded channel, where
            # long fronts between forward and backward flow replace the toe Undesirable because of unpredictability,
            # length, and efficiency, since it is closer to a jet than a jump May shift to an R-Jump periodically
            # T: “Transitional” Jump A well-developed toe forms perpendicular to the flow upstream of the expansion,
            # while the jump extends past the expansion May be symmetric or asymmetric, depending on the jump position
            # relative to the expansion

        else:
            raise ValueError(f'Invalid classification system: {classification}')
        return '', ''

    def compute_hyd_jump_length(self, upstream_froude):
        """Computes the length of the Hydraulic Jump.

        Args:
            upstream_froude (float): the froude number of the flow upstream of the jump

        Returns:
            self.hyd_jump_length (float): The length of the hydraulic jump.
        """
        flat_jump_length = 0.0
        slope_jump_length = 0.0
        self.hyd_jump_type_b_over_break = False
        # From Nathan Lowe's Thesis: pdf page 86

        # Bradley and Peterka(1957)
        # Lj / y2 = 220 * tanh((fr - 1) / 22)

        # Get the froude and depth from downstream direction at the location of jump
        # dnFroudeXY.InterpolateY(m_hydraulicJumpStation, m_upstreamFroude, false);

        flat_jump_length_box = self.hyd_jump_depth * 220 * math.tanh((upstream_froude - 1.0) / 22.0)
        flat_jump_length_cir = 6.0 * self.hyd_jump_sequent_depth

        if self.input_dict['calc_data']['Geometry']['Shape'] in ['box', 'rectangle']:
            flat_jump_length = flat_jump_length_box
        elif not self.channel_is_embedded and self.input_dict['calc_data']['Geometry']['Shape'] == 'circle':
            flat_jump_length = flat_jump_length_cir
        else:
            flat_jump_length = flat_jump_length_cir
            if flat_jump_length < flat_jump_length_box:
                flat_jump_length = flat_jump_length_box
        hydraulic_jump_length = flat_jump_length

        _, horizontal_slope = self.get_data('Horizontal tolerance')
        # Now that have the flat jump length, check on slope
        if self.slope > horizontal_slope:
            # Adjust the jump length for sloped culverts
            # Equation from Energy Dissipators and Hydraulic Jump by Willi H.Hager, volume 8
            # Equation 3.5 in Hager HydJumps, page 44.
            # lj / l * j = exp (- 4 / 3 theta)
            # valid for bottoms slopes of theta < 17 degrees
            theta = math.atan(self.slope)
            jump_factor = math.exp(-1.3333333333 * theta)
            slope_jump_length = flat_jump_length * jump_factor
            hydraulic_jump_length = slope_jump_length

            # Now check if it is a type B jump (over a break) and not the lower(runout) section
            if self.hyd_jump_station + hydraulic_jump_length >= self.total_length and \
                    self.ignore_energy_in_full_flow_calcs:
                self.hyd_jump_type_b_over_break = True

        # Assume Position 1 (jump ends at vertical line)

        # Assume Position 2 (jump is at middle of vertical line
        # Assume Position 3 (jump begins at vertical line)
        # startJumpStation = self.hyd_jump_station
        final_jump_station = self.hyd_jump_station + hydraulic_jump_length
        self.hyd_jump_length = hydraulic_jump_length
        # If the jump runs out of the culvert, assume it does not occur(conservative)
        if (final_jump_station >= self.total_length) and not self.froude_jump and \
                not self.ignore_energy_in_full_flow_calcs:
            self.hyd_jump_exists = False
            self.hyd_jump_swept_out = True
            # If the jump is swept out, we need the profile from up to down
            # if m_computationDirection == CD_UPSTREAM:
            #    m_computationDirection = CD_DOWNSTREAM

        return self.hyd_jump_length

    def _combine_backwater_curve(self):
        """Complete the backwater curve by stitching in two curves with an hydraulic jump or choose one."""
        if self.hyd_jump_exists or self.join_curves:
            self.solution = 'curves joined'
            self.x, self.y, _, _ = self.combine_wses_with_hydraulic_jump(
                self.hyd_jump_exists, self.hyd_jump_station,
                self.hyd_jump_depth, self.hyd_jump_length,
                self.hyd_jump_sequent_depth, self.results_upstream,
                self.intermediate_results, self.results_downstream)
        elif self.hyd_jump_swept_out:
            # This case is an hydraulic jump forms near tailwater but is swept out, meaning, that the downstream
            # calculations will be used.
            self.solution = 'downstream direction'
            self.intermediate_results = self.results_downstream
            self.x = self.results_downstream['x']
            self.y = self.results_downstream['y']
        else:
            if self.enforce_upstream_depth:
                self.solution = 'downstream direction'
                self.intermediate_results = self.results_downstream
                self.x = self.results_downstream['x']
                self.y = self.results_downstream['y']
            # if they end the same place (normal depth), take the downstream approach
            elif math.isclose(self.results_upstream['y'][-1], self.results_downstream['y'][-1],
                              abs_tol=self.zero_tol) and \
                    math.isclose(self.results_upstream['y'][-1], self.normal_depth, abs_tol=self.zero_tol):
                self.solution = 'downstream direction'
                self.intermediate_results = self.results_downstream
                self.x = self.results_downstream['x']
                self.y = self.results_downstream['y']
            elif math.isclose(
                    self.results_upstream['y'][0], self.results_downstream['y'][0], abs_tol=self.zero_tol):
                # If they are the same depth (happens as we set the upstream water depth based on a water profile)
                # Choose the lower tailwater depth (less energy to get downstream)
                if self.results_upstream['y'][-1] < self.results_downstream['y'][-1]:
                    self.solution = 'upstream direction'
                    self.intermediate_results = self.results_upstream
                    self.x = self.results_upstream['x']
                    self.y = self.results_upstream['y']
                else:
                    self.solution = 'downstream direction'
                    self.intermediate_results = self.results_downstream
                    self.x = self.results_downstream['x']
                    self.y = self.results_downstream['y']
            # If the slope is mild, horizontal, or adverse, we always take the upstream direction
            if self.slope_type in ['mild', 'horizontal', 'adverse', ]:
                self.solution = 'upstream direction'
                self.intermediate_results = self.results_upstream
                self.x = self.results_upstream['x']
                self.y = self.results_upstream['y']
            else:  # steep or critical slope
                downstream_dir_inlet_depth = self.results_downstream['y'][0]
                first_valid = next(
                    (val for val in self.results_downstream['sequent_depth']
                        if not math.isclose(val, self.null_data, abs_tol=self.zero_tol)),
                    None  # Returns None if all are close to null_data
                )
                if first_valid is not None and first_valid > downstream_dir_inlet_depth:
                    downstream_dir_inlet_depth = first_valid
                # TODO: Handle vena contracta case!
                # If the sequent depth (or depth if no sequent depth) is greater, it governs
                if downstream_dir_inlet_depth > self.results_upstream['y'][0]:
                    self.solution = 'downstream direction'
                    self.intermediate_results = self.results_downstream
                    self.x = self.results_downstream['x']
                    self.y = self.results_downstream['y']
                else:  # The inlet is surcharged by tailwater
                    self.solution = 'upstream direction'
                    self.intermediate_results = self.results_upstream
                    self.x = self.results_upstream['x']
                    self.y = self.results_upstream['y']

        self.flow_area = self.intermediate_results['flow_area']
        self.wetted_perimeter = self.intermediate_results['wetted_perimeter']
        self.top_width = self.intermediate_results['top_width']
        self.hydraulic_radius = self.intermediate_results['hydraulic_radius']
        self.manning_n = self.intermediate_results['manning_n']
        self.velocity = self.intermediate_results['Velocity']
        self.energy = self.intermediate_results['Energy']
        self.energy_loss = self.intermediate_results['energy_loss']
        self.energy_slope = self.intermediate_results['energy_slope']

    def combine_wses_with_hydraulic_jump(self, hyd_jump_exists: bool, hyd_jump_station: float, hyd_jump_depth: float,
                                         hyd_jump_length: float, hyd_jump_sequent_depth: float, results_upstream: dict,
                                         intermediate_results: dict, results_downstream: dict,
                                         split_at_station: float = sys.float_info.max) -> tuple:
        """Combine the water surface elevations with the hydraulic jump.

        Args:
            hyd_jump_exists (bool): whether an hydraulic jump exists
            hyd_jump_station (float): the station that the hydraulic jump starts.
            hyd_jump_depth (float): the depth at the beginning of the hydraulic jump.
            hyd_jump_length (float): the length of the hydraulic jump.
            hyd_jump_sequent_depth (float): the sequent depth of the upstream depth.
            results_upstream (dict of variables): tracks the hydraulic results along the upstream direction
            intermediate_results (dict of variables): tracks the hydraulic results from the most recent computations.
            results_downstream (dict of variables): tracks the hydraulic results along the downstream direction.
            split_at_station (float): stitches the two GVF curves at the specified station.

        Returns:
            x_up (list): the x values upstream and downstream of the hydraulic jump but before the split_at_station.
            y_up (list): the y values upstream and downstream of the hydraulic jump but before the split_at_station.
            x_down (list): the x values upstream and downstream of the hydraulic jump but after the split_at_station.
            y_down (list): the y values upstream and downstream of the hydraulic jump but after the split_at_station.
        """
        x_up = []
        y_up = []
        x_down = []
        y_down = []
        for item in results_downstream:
            intermediate_results[item] = []
        hyd_jump_last_station = hyd_jump_station + hyd_jump_length
        for index in range(len(results_downstream['x'])):
            if results_downstream['x'][index] < hyd_jump_station:
                if results_downstream['x'][index] <= split_at_station:
                    x_up.append(results_downstream['x'][index])
                    y_up.append(results_downstream['y'][index])
                else:
                    x_down.append(results_downstream['x'][index])
                    y_down.append(results_downstream['y'][index])
                for item in results_downstream:
                    if index < len(results_downstream[item]):
                        intermediate_results[item].append(
                            results_downstream[item][index])
            elif hyd_jump_exists:
                # Add the Hydraulic Jump
                if hyd_jump_station <= split_at_station:
                    x_up.append(hyd_jump_station)
                    y_up.append(hyd_jump_depth)
                else:
                    x_down.append(hyd_jump_station)
                    y_down.append(hyd_jump_depth)
                # Interpolate the results for upstream side of the jump
                for item in results_downstream:
                    if results_downstream[item] is None or len(results_downstream[item]) == 0 or \
                            all(item == self.null_data for item in results_downstream[item]):
                        continue
                    item_interp = Interpolation(results_downstream['x'], results_downstream[item],
                                                null_data=self.null_data, zero_tol=self.zero_tol)
                    item_value, _ = item_interp.interpolate_y(hyd_jump_station)
                    intermediate_results[item].append(item_value)
                if hyd_jump_last_station <= split_at_station:
                    x_up.append(hyd_jump_last_station)
                    y_up.append(hyd_jump_sequent_depth)
                else:
                    x_down.append(hyd_jump_last_station)
                    y_down.append(hyd_jump_sequent_depth)
                # Interpolate the results for downstream side of the jump
                for item in results_upstream:
                    if results_upstream[item] is None or len(results_upstream[item]) == 0 or \
                            all(item == self.null_data for item in results_upstream[item]):
                        continue
                    item_interp = Interpolation(results_upstream['x'], results_upstream[item],
                                                null_data=self.null_data, zero_tol=self.zero_tol)
                    item_value, _ = item_interp.interpolate_y(hyd_jump_last_station)
                    intermediate_results[item].append(item_value)
                break
        for index in range(len(results_upstream['x'])):
            if results_upstream['x'][index] > hyd_jump_last_station:
                if results_upstream['x'][index] <= split_at_station:
                    x_up.append(results_upstream['x'][index])
                    y_up.append(results_upstream['y'][index])
                else:
                    x_down.append(results_upstream['x'][index])
                    y_down.append(results_upstream['y'][index])
                for item in results_downstream:
                    if index < len(results_upstream[item]):
                        intermediate_results[item].append(
                            results_upstream[item][index])
                    else:
                        pass
                        # error = True
        if split_at_station < sys.float_info.max * 0.9:
            return x_up, y_up, x_down, y_down
        else:
            return x_up, y_up, None, None
