"""Bridge Deck Table Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.table_calc import TableCalc
from xms.FhwaVariable.core_data.calculator.table_data import TableData

# 4. Local modules


class BridgeDeckCalc(TableCalc):
    """A class that defines the bridge deck table calculator."""

    def _compute_data(self):
        """Compute the data.

        Returns:
            True, if successful; otherwise, False
        """
        super()._compute_data()

        _, size = self.get_data('Number of items')
        stations = self.input_dict['calc_data']['Data input']['Station']
        low_chords = self.input_dict['calc_data']['Data input']['Low chord']
        high_chords = self.input_dict['calc_data']['Data input']['High chord']
        self.results['Bridge deck'] = high_chords

        min_size = min(len(stations), len(low_chords), len(high_chords))
        if size > min_size:
            size = min_size

        x_data = []
        y_data = []
        for i in range(size):
            x_data.append(stations[i])
            y_data.append(low_chords[i])

        for i in range(size - 1, -1, -1):
            x_data.append(stations[i])
            y_data.append(high_chords[i])

        if len(x_data) > 0:
            x_data.append(stations[0])
            y_data.append(low_chords[0])

        self.results['Station'] = x_data
        self.results['Bridge deck'] = y_data

        x_var = self.plot_dict['Bridge deck geometry']['series'][0]['x var']
        y_var = self.plot_dict['Bridge deck geometry']['series'][0]['y var']
        if x_var:
            self.plot_dict['Bridge deck geometry']['series'][0]['x var'].set_val(self.results['Station'])
        if x_var and not y_var:
            self.plot_dict['Bridge deck geometry']['series'][0]['y var'] = copy.deepcopy(x_var)
            y_var = self.plot_dict['Bridge deck geometry']['series'][0]['y var']
            y_var.name = 'Bridge deck'
        if self.plot_dict['Bridge deck geometry']['series'][0]['y var']:
            self.plot_dict['Bridge deck geometry']['series'][0]['y var'].set_val(self.results['Bridge deck'])

        return True


class BridgeDeckTable(TableData):
    """Override the table class, so we can draw plots based on widths."""
    def __init__(self, theme, name, plot_names, name_of_items='points', stand_alone_calc=True, input=None, min_items=1,
                 max_items=sys.maxsize, show_increment=False, app_data=None, model_name=None, project_uuid=None):
        """Initializes the calculator.

        Args:
            theme (dict): the theme
            name (str): the name of the calculator
            plot_names (list of str): list of plot names
            name_of_items (str): name of the items
            stand_alone_calc (bool): whether the calculator is a stand-alone calculator
            input (dict): dictionary of input variables
            min_items (int): minimum number of items
            max_items (int): maximum number of items
            show_increment (bool): whether to show the increment
            app_data (AppData): the application data
            model_name (str): the name of the model
            project_uuid (str): the project UUID
        """
        super().__init__(theme=theme, name=name, plot_names=plot_names, name_of_items=name_of_items,
                         stand_alone_calc=stand_alone_calc, input=input, min_items=min_items, max_items=max_items,
                         show_increment=show_increment, app_data=app_data, model_name=model_name,
                         project_uuid=project_uuid)

        self.calculator = BridgeDeckCalc()
