"""GradationManagerData Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import math

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import CalcData
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.gradations.bore_hole_data import BoreHoleData
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.gradations.bore_hole_manager_calc import BoreHoleManagerCalc
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.gradations.bore_hole_manager_data import BoreHoleManagerData
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.gradations.gradation_manager_calc import GradationManagerCalc


class GradationManagerData(CalcData):
    """Provides a class that will define a class that manages a list of gradation sets."""

    locations = BoreHoleManagerCalc.locations

    def __init__(self, app_data=None, model_name=None, project_uuid=None):
        """Initializes the Class.

        Args:
            app_data (AppData): The application data.
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'Gradation Manager'
        self.type = 'GradationManagerData'

        self.calculator = GradationManagerCalc()

        self.theme = self.get_theme()

        self.calc_support_plot = False

        # max_value = self.max_value
        # Input
        self.input = {}

        self.contracted_options = ['Single D50', 'Single borehole', 'Multiple boreholes']
        self.approach_options = ['None', 'Single D50', 'Gradations']

        self.input['Contracted gradation input options'] = Variable(
            'Contracted gradation input options', 'list', 0, self.contracted_options)

        # Contracted Gradation
        self.input['Contracted D50'] = Variable(
            'Contracted D50 (D50)',
            'float',
            0.0, [],
            precision=6,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_short_length,
            si_units=self.si_short_length,
            note='Used for clear water scour depth calculations, 0.2 mm is the '
            'lower limit for noncohesive material')
        self.input['Contracted D50'].selected_us_unit = 'mm'
        self.input['Contracted D50'].selected_si_unit = 'mm'
        # self.input['Selected borehole'] = Variable(
        #     'Selected borehole along contracted cross-section',
        #     'uuid_dict',
        #     -1, {},
        #     note='')
        # self.input['Critical shear stress (τc)'] = Variable(
        #     "Critical shear stress (τc)",
        #     "float",
        #     0.0,
        #     [],
        #     precision=4,
        #     unit_type=['stress'],
        #     native_unit='psf',
        #     us_units=[['psf']],
        #     si_units=[['pa']],
        #     note='The level of shear required to erode the soil layer')
        _, gamma_s = self.get_setting('Unit weight of soil (γs)')
        self.input['Unit weight of soil (γs)'] = Variable(
            'Unit weight of soil (γs)',
            'float',
            gamma_s,
            precision=4,
            complexity=1,
            unit_type=["specific_weight"],
            native_unit="lb/ft^3",
            us_units=[["lb/ft^3"]],
            si_units=[["kN/m^3", "N/m^3"]])

        self.input['Angle of repose (Θ)'] = Variable('Angle of repose (Θ)', 'float', math.radians(44), [],
                                                     limits=(math.radians(30), math.radians(44)),
                                                     precision=2, unit_type='angle',
                                                     native_unit='radians',
                                                     us_units=['degrees', 'radians', 'H:1V'],
                                                     si_units=['degrees', 'radians', 'H:1V'])
        self.input['Angle of repose (Θ)'].selected_us_unit = 'degrees'

        self.input['Single borehole'] = Variable(
            'Borehole at contracted section', 'calc', BoreHoleData(app_data=app_data, model_name=model_name,
                                                                   project_uuid=project_uuid))

        self.input['Boreholes'] = Variable(
            'Boreholes at contracted section', 'calc', BoreHoleManagerData(app_data=app_data, model_name=model_name,
                                                                           project_uuid=project_uuid))

        self.input['Approach gradation input options'] = Variable(
            'Approach gradation input options', 'list', 0, self.approach_options)

        # Approach Gradation
        self.input['Approach D50'] = Variable(
            'Approach D50 (D50)',
            'float',
            0.0, [],
            precision=6,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_short_length,
            si_units=self.si_short_length,
            note='')
        self.input['Approach D50'].selected_us_unit = 'mm'
        self.input['Approach D50'].selected_si_unit = 'mm'
        # self.input['Selected approach location'] = Variable(
        #     'Selected gradation location along approach cross-section',
        #     'list', -1, ['left overbank', 'main channel', 'right overbank'], note='')

        self.input['Approach gradations'] = Variable(
            'Gradations at Approach section', 'calc', BoreHoleManagerData(
                app_data=app_data, model_name=model_name, project_uuid=project_uuid))

        self.input['Approach gradations'].get_val().input['Location'].set_val(BoreHoleManagerData.locations[0])
        self.input['Approach gradations'].get_val().input['Approach layer'].type = 'class'
        self.input['Approach gradations'].get_val().input['Approach layer'].get_val().input['Name'].set_val(
            'Main channel gradation')
        self.input['Approach gradations'].get_val().input['Left overbank layer'].type = 'class'
        self.input['Approach gradations'].get_val().input['Left overbank layer'].get_val().input['Name'].set_val(
            'Left overbank gradation')
        self.input['Approach gradations'].get_val().input['Right overbank layer'].type = 'class'
        self.input['Approach gradations'].get_val().input['Right overbank layer'].get_val().input['Name'].set_val(
            'Right overbank gradation')

        # Intermediate
        # self.compute_prep_functions = []
        self.intermediate_to_copy.extend(['x_data', 'y_data', 'min_y', ])

        self.x_data = [0.0]
        self.y_data = [0.0]
        self.min_y = 0.0

        self.warnings = []
        self.results = {}
        # self.results['result'] = Variable('result', 'float_list', 0.0, [], precision=precision,
        #                                   unit_type=unit_type, native_unit=native_unit,
        #                                   us_units=us_units, si_units=si_units)
        self.results['Approach gradations'] = {}
        self.results['Contraction gradations'] = {}
        self.results['Data complete'] = Variable('Data Complete', 'bool', 0.0, [])
        self.results['Number of boreholes'] = Variable('Number of boreholes', 'int', 0, [])

    def set_bridge_cs_and_piers(self, bcs_data, pier_data, bh_cl, abutment_dict):
        """Set the bridge cross-section data.

        Args:
            bcs_data (BridgeCrossSection): The bridge cross-section data.
            pier_data (Pier): The pier data.
            bh_cl (float): The elevation of the bridge cross-section.
            abutment_dict (dict): The abutment data dictionary.
        """
        for bh in self.input['Boreholes'].value.input['Boreholes'].value.item_list:
            bh.bcs_data = bcs_data
            bh.pier_data = pier_data
            bh.bh_cl = bh_cl
            bh.abutment_data = abutment_dict
        self.input['Single borehole'].value.bcs_data = bcs_data
        self.input['Single borehole'].value.pier_data = pier_data
        self.input['Single borehole'].value.bh_cl = bh_cl
        self.input['Single borehole'].value.abutment_data = abutment_dict

    def get_input_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and left out of the input dictionary)

        Returns:
              input_vars (dictionary of variables): the input variables
        """
        input_vars = {}

        approach_options = copy.deepcopy(self.approach_options)

        input_vars['Contracted gradation input options'] = self.input['Contracted gradation input options']
        if input_vars['Contracted gradation input options'].get_val() == 'Single D50':
            input_vars['Contracted D50'] = self.input['Contracted D50']
            input_vars['Unit weight of soil (γs)'] = self.input['Unit weight of soil (γs)']
            input_vars['Angle of repose (Θ)'] = self.input['Angle of repose (Θ)']
            approach_options.remove('Gradations')
        elif input_vars['Contracted gradation input options'].get_val() == 'Single borehole':
            input_vars['Boreholes'] = self.input['Single borehole']
        else:
            input_vars['Boreholes'] = self.input['Boreholes']

        input_vars['Approach gradation input options'] = self.input['Approach gradation input options']
        input_vars['Approach gradation input options'].value_options = approach_options
        if input_vars['Approach gradation input options'].get_val() == 'None':
            pass
        elif input_vars['Approach gradation input options'].get_val() == 'Single D50':
            input_vars['Approach D50'] = self.input['Approach D50']
        else:
            input_vars['Approach gradations'] = self.input['Approach gradations']

        return input_vars

    # def update_elevation_data(self):
    #     """Update the elevation data."""
    #     set = self.input['Boreholes'].value
    #     set.x_data = self.x_data
    #     set.y_data = self.y_data
    #     set.min_y = self.min_y

    #     set.compute_data()

    #     set = self.input['Approach gradations'].value
    #     set.x_data = self.x_data
    #     set.y_data = self.y_data
    #     set.min_y = self.min_y

    #     set.compute_data()

    def get_gradation_dict(self, location):
        """Get the list of gradation sets."""
        # self.update_elevation_data()
        if location == 'Approach cross-section':
            if self.input['Approach gradation input options'].get_val() == 'Single D50':
                return None
            else:
                return self.input['Approach gradations'].value

        else:
            if self.input['Contracted gradation input options'].get_val() == 'Single D50':
                return None
            else:
                return self.input['Boreholes'].value

    def get_results_tab_group(self, unknown=None):
        """Get the results tab group.

        Args:
            unknown (variable): unknown variable

        Returns:
            results_vars (list of variables): results group of variables
        """
        results_vars = {}

        results_vars['All gradations'] = {}
        results_vars['Approach gradations'] = {}
        results_vars['Contraction gradations'] = {}

        results_vars['All gradations']['Approach gradations'] = self.results['Approach gradations']
        results_vars['All gradations']['Contraction gradations'] = self.results['Contraction gradations']
        results_vars['Approach gradations'] = self.results['Approach gradations']
        results_vars['Contraction gradations'] = self.results['Contraction gradations']

        return results_vars
