"""CalcData for performing Abutment Scour calculations."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.plot.plot_options import PlotOptions
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.abutment_scour_calc import AbutmentScourCalc
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.scour_base_data import ScourBaseData


class AbutmentScourData(ScourBaseData):
    """A class that defines a contraction scour at a bridge contraction."""
    # # 8.10 from HEC-18 (previously α_wingwall_mainchannel)
    # α_mainchannel = [
    #     1.0006, 1.0185, 1.0345, 1.048, 1.0655, 1.091, 1.118, 1.142, 1.17, 1.2,
    #     1.225, 1.25, 1.3, 1.355, 1.406, 1.462, 1.501, 1.537, 1.584, 1.648,
    #     1.699, 1.772, 1.8615, 2.0, 2.1, 2.25, 2.5, 2.75, 3.0
    # ]
    # α_mainchannel = [
    #     1.2, 1.3, 1.4, 1.465, 1.53, 1.6, 1.653, 1.687, 1.716, 1.737, 1.749,
    #     1.755, 1.757, 1.741, 1.708, 1.659, 1.618, 1.58, 1.53, 1.467, 1.423,
    #     1.367, 1.31, 1.244, 1.209, 1.169, 1.129, 1.109, 1.102
    # ]

    # # 8.11 from HEC-18 (previously α_spillthrough_floodplain)
    # α_floodplain = [
    #     1.0, 1.019, 1.051, 1.0713, 1.1059, 1.155, 1.2, 1.25, 1.3, 1.35, 1.4,
    #     1.45, 1.5, 1.55, 1.6, 1.65, 1.7, 1.75, 1.8, 1.85, 2.0, 2.25, 2.5, 2.75,
    #     3.0, 3.25, 3.5, 3.75, 4.0, 4.6, 5.0
    # ]
    # α_floodplain = [
    #     1.0, 1.25, 1.49, 1.6, 1.75, 1.9, 1.995, 2.071, 2.122, 2.156, 2.1756,
    #     2.184, 2.185, 2.179, 2.168, 2.153, 2.137, 2.116, 2.095, 2.073, 2.002,
    #     1.876, 1.749, 1.626, 1.509, 1.4108, 1.34, 1.282, 1.235, 1.159, 1.134
    # ]

    def __init__(self, clear_my_own_results=True, app_data=None, model_name=None, project_uuid=None):
        """Initializes the GVF calculator.

        Args:
            clear_my_own_results (bool): Should the class clean up its results?  If Standalone, yet. otherwise, no.
            app_data (AppData): The application data.
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        _, null_data = self.get_setting('Null data')
        _, zero_tol = self.get_setting('Zero tolerance')

        self.calculator = AbutmentScourCalc(null_data=null_data, zero_tol=zero_tol)
        # self.α_mainchannel_interpolate = Interpolation(AbutmentScourData.α_mainchannel,
        #                                                AbutmentScourData.α_mainchannel, app_data=self.app_data,
        #                                                null_data=null_data, zero_tol=zero_tol)
        # self.α_floodplain_interpolate = Interpolation(AbutmentScourData.α_floodplain,
        #                                               AbutmentScourData.α_floodplain, app_data=self.app_data,
        #                                               null_data=null_data, zero_tol=zero_tol)

        self.name = "Abutment Scour Calculator"
        self.type = "AbutmentScourData"
        self.class_name = 'Abutment Scour Calculator'

        self.calc_support_dual_input = False

        condition_options = ['Compute', 'Live bed', 'Clear water']
        # location_options = ['Commpute', 'Type a (main channel)', 'Type b (overbank)']
        location_options = ['Type a (main channel)', 'Type b (overbank)']
        # abutment_options = ['Vertical walls', 'Vertical-wall abutment with wing walls',
        #                     'Spill-through abutment']

        # Input
        self.input['Scour condition'] = Variable('Scour condition', "list", 0,
                                                 condition_options)

        self.input['Scour location'] = Variable('Scour location', "list", 0,
                                                location_options)

        # self.input['Abutment Type (k1)'] = Variable('Abutment Type (k1)',
        #                                             "list", 0,
        #                                             abutment_options)

        self.input['Unit discharge, upstream (q1)'] = Variable(
            'Unit Discharge, Upstream in Active, Approach Overbank Area (q1)', 'float', 0.0, [], precision=2,
            unit_type=['unit flow'], native_unit='ft^3/s/ft', us_units=['ft^3/s/ft'], si_units=['m^3/s/m'])

        self.input['Unit discharge, contracted (q2)'] = Variable(
            'Unit Discharge, in contracted section (q2)', 'float', 0.0, [], precision=2, unit_type=['unit flow'],
            native_unit='ft^3/s/ft', us_units=['ft^3/s/ft'], si_units=['m^3/s/m'])

        self.input['Average depth upstream (y1)'] = Variable(
            'Average depth upstream of contraction (y1)', 'float', 0.0, [], precision=2, unit_type=['length'],
            native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.input['Depth prior to scour in contracted section (y0)'] = Variable(
            'Depth prior to scour in contracted section (y0)', 'float', 0.0, [], precision=2, unit_type=['length'],
            native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length, note='Depth at Abutment Toe')

        # Cohesive
        # self.input['Average velocity in contracted section (v2)'] = Variable(
        #     'Average velocity in contracted section (v2)',
        #     'float',
        #     0.0, [],
        #     precision=2,
        #     unit_type=['Velocity'],
        #     native_unit='ft/s',
        #     us_units=self.us_velocity,
        #     si_units=self.si_velocity)

        self.input["Manning's n (n)"] = Variable(
            "Manning's n (n)", 'float', 0.0, [], precision=4, unit_type=['coefficient'])

        _, gamma_w = self.get_setting('Unit weight of water (γw)')
        self.input['Unit weight of water (γw)'] = Variable(
            'Unit weight of water (γw)', 'float', gamma_w, precision=4, unit_type=["specific_weight"],
            native_unit="lb/ft^3", us_units=[["lb/ft^3"]], si_units=[["kN/m^3", "N/m^3"]])

        # Shear Decay
        self.input['Alpha value for local shear decay'] = Variable(
            'Alpha value for local shear decay (α)', 'float', 4.37, [],
            limits=(-sys.float_info.max, sys.float_info.max), precision=2, complexity=2)

        self.input['Beta value for local shear decay'] = Variable(
            'Beta value for local shear decay (β)', 'float', 1.33, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2, complexity=2)

        self.unknown = None

        # Intermediate
        # self.compute_prep_functions = []
        self.intermediate_to_copy.extend([
            'cross_section_x', 'cross_section_y', 'abutment_toe_station', 'wse_x', 'wse_y', 'scour_depth',
            'contraction_scour_depth', 'ltd', 'ymax', 'streambed_elev_prior_to_scour', 'streambed_elev_post_scour',
            'con_decay_x', 'con_decay_y', 'con_decay_x_markers', 'con_decay_y_markers', 'is_computing_shear_decay',
            'centerline', 'centerline_streambed', 'channel_location', 'scour_reference_point', 'bh_dict', 'bh_uuid',
            'bh', 'bh_layers', 'layer_index', 'surface_d50', 'abutment_toe_station'])
        self.cross_section_x = None
        self.cross_section_y = None
        self.abutment_toe_station = None
        self.wse_x = None
        self.wse_y = None
        self.contraction_scour_depth = 0.0
        self.ltd = 0.0

        self.ymax = 0.0

        self.streambed_elev_prior_to_scour = 0.0
        self.streambed_elev_post_scour = 0.0

        self.con_decay_x = []
        self.con_decay_y = []
        self.con_decay_x_markers = []
        self.con_decay_y_markers = []

        # Results
        self.clear_my_own_results = clear_my_own_results

        self.results['Average scour depth (y2)'] = Variable(
            'Average scour depth (y2)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Amplification factor (α)'] = Variable(
            'Amplification factor (α)', 'float', 0.0, precision=2, limits=(0.5, 0.69), unit_type=['coefficient'])
        self.results['q2 / q1'] = Variable(
            'q2 / q1', 'float', 0.0, precision=2, limits=(0.0, sys.float_info.max), unit_type=['coefficient'])

        self.results['Governing scour method'] = Variable(
            'Governing scour method', "string", 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Flow depth including scour (yc)'] = Variable(
            'Flow depth including scour (yc)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Max flow depth including scour (ymax)'] = \
            Variable('Max flow depth including scour (ymax)', 'float', 0.0, [], precision=2, unit_type=['length'],
                     native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Flow depth including scour using clear water method (yc)'] = \
            Variable('Flow depth including scour using clear water method (yc)', 'float', 0.0, [], precision=2,
                     unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Max flow depth including scour using clear water method (ymax)'] = \
            Variable('Max flow depth including scour using clear water method (ymax)', 'float', 0.0, [],
                     precision=2, unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
                     si_units=self.si_mid_length)

        self.results['Longterm degradation'] = Variable(
            'Longterm degradation', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Scour hole depth (yc)'] = Variable(
            'Scour hole depth (yc)', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Shear decay results']['Shear decay alpha'] = Variable(
            'Shear decay alpha coefficient', 'float', 0.0, [], precision=4, unit_type=['coefficient'])

        self.results['Shear decay results']['Shear decay beta'] = Variable(
            'Shear decay beta coefficient', 'float', 0.0, [], precision=4, unit_type=['coefficient'])

        self.warnings = []

        # # Reference Manuals
        # self.reference_pdfs['HEC-18'] = (
        #     'HEC-18, 5th Edition - Evaluating Scour at Bridges.pdf', 215)

        # # plot
        # self.plots['profile'] = {}
        # self.plots['profile']['Plot name'] = "Profile"
        # self.plots['profile']['Legend'] = 'best'
        # self.plots['profile']['index'] = 1

        shear_var = Variable(
            'Shear stress', 'float_list', 0, [0.0], precision=2, limits=(0.0, sys.float_info.max),
            unit_type=['stress'], native_unit='psf', us_units=[['psf']], si_units=[['pa']],)

        elevation_var = Variable(
            'Elevation', 'float_list', 0, [0.0], precision=2, limits=(-sys.float_info.max, sys.float_info.max),
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        shear_vs_elevation = {}
        shear_vs_elevation['Shear stress'] = shear_var
        shear_vs_elevation['Elevation'] = elevation_var

        input_dict = {}
        input_dict['Shear decay'] = shear_vs_elevation

        self.input['Plot options'] = {}
        self.input['Plot options']['Shear decay'] = Variable(
            'Shear decay', 'class', PlotOptions('Shear decay', input_dict=input_dict, show_series=True,
                                                app_data=app_data, model_name=model_name, project_uuid=project_uuid),
            complexity=1)

        name = 'Shear decay'
        shear_decay_color = (128, 0, 128)  # Deep Violet
        # Other options: Deep Violet: (128, 0, 128), Amber/Gold: (255, 170, 0), Orange-Red: (255, 85, 0),
        # Magenta: (220, 20, 140)
        self.input['Plot options'][name].get_val().set_plot_series_options(
            related_index=0, index=0, name=name, x_axis='Shear stress', y_axis='Elevation',
            line_color=shear_decay_color, linetype='solid', line_width=1.5, fill_below_line=False,
            fill_color=shear_decay_color)  # need x_var - it is None

        self.input['Plot options'][name].get_val().set_plot_point_options(
            related_index=0, index=0, name=name, point_color=shear_decay_color,)
        #    point_marker=point_marker, marker_size=marker_size)

        grey_rgb = (127, 127, 127)
        self.input['Plot options'][name].get_val().set_plot_line_options(
            index=0, name='Y max', line_alignment='horizontal',
            line_color=grey_rgb, linetype='dotted', line_width=1.5, text_color=grey_rgb,
            labels='Y max')

        self.plot_names = ['Shear decay']
        self.plots = {}
        for name in self.plot_names:
            self.plots[name] = {}
            self.plots[name]['Plot name'] = name
            self.plots[name]['Legend'] = "best"

    def check_for_needed_data(self):
        """Checks for needed data for the calculator to run.

        Else, show popup with warnings and do not run dialog.
        """
        self.warnings = []
        failed_list = []
        result = True

        if self.gradations is None:
            return result

        # Check for the cross-section
        if self.calculator.cross_section_x is None or len(self.calculator.cross_section_x) <= 0:
            # self.warnings.append("Please enter a cross-section before defining the abutment scour.")
            failed_list.append("Bridge cross-section")
            result = False
        # Check for abutment toe station
        if self.calculator.abutment_toe_station is None or self.calculator.abutment_toe_station == 0.0:
            # self.warnings.append("Please enter the abutment toe station before defining the abutment scour.")
            failed_list.append("Abutment toe station")
            result = False
        # Check for WSE
        if self.calculator.wse_x is None or len(self.calculator.wse_x) == 0 or len(self.calculator.wse_y) == 1 and \
                self.calculator.wse_y[0] == 0.0:
            # self.warnings.append("Please enter the scenario WSE before defining the abutment scour.")
            failed_list.append("Scenario WSE")
            result = False

        if len(failed_list) > 0:
            if len(failed_list) > 2:
                failed_text = ', '.join(failed_list[:-1]) + ', and ' + failed_list[-1]
            elif len(failed_list) > 1:
                failed_text = ' '.join(failed_list[:-1]) + ' and ' + failed_list[-1]
            else:
                failed_text = failed_list[0]
            warning_message = f"Please enter the following before defining the abutment scour: {failed_text}."
            self.warnings.append(warning_message)

        if self.is_computing_shear_decay:
            self.calc_support_plot = True
        else:
            self.calc_support_plot = False

        return result

    def get_input_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and left out of the input dictionary)

        Returns:
              input_vars (dictionary of variables): the input variables
        """
        input_vars = {}

        self.update_gradation_lists()

        input_vars = copy.deepcopy(self.input)

        input_vars.pop('Critical shear stress (τc)')

        # cohesive
        if self.d50 <= 0.0 and self.critical_shear_stress > 0.0:
            pass
        else:
            # input_vars.pop('Average velocity in contracted section (v2)')
            input_vars.pop("Manning's n (n)")

        if self.clear_my_own_results:
            # This means we are a stand alone calculator and we pop the gradation selections
            # input_vars.pop('Approach gradation')
            input_vars.pop('Selected approach location')
            # input_vars.pop('Contracted gradation')
            input_vars.pop('Selected borehole')
        else:
            # This means we are part of a bridge project and we pop the D50s
            if 'Approach D50' in input_vars:
                input_vars.pop('Approach D50')
            if 'Contracted D50' in input_vars:
                input_vars.pop('Contracted D50')
            if 'Critical shear stress (τc)' in input_vars:
                input_vars.pop('Critical shear stress (τc)')
            # sel_approach_uuid = self.input['Approach gradation'].get_val()
            # if sel_approach_uuid is not None and sel_approach_uuid != 'None':
            #     approach_locations = self.approach_gradation_dict[sel_approach_uuid].input[
            #         'Approach locations'].get_val()
            #     if approach_locations == 'Single gradation for entire cross-section':
            #         input_vars.pop('Selected approach location')
            # else:
            #     input_vars.pop('Selected approach location')
            if self.gradations.input['Approach gradation input options'].get_val() == 'Single D50' or \
                    self.gradations.input['Approach gradations'].get_val().input['Approach locations'].get_val() == \
                    'Single gradation for entire cross-section':
                input_vars.pop('Selected approach location')
            if self.gradations.input['Contracted gradation input options'].get_val() == 'Single D50' or \
                    self.gradations.input['Contracted gradation input options'].get_val() == 'Single borehole':
                input_vars.pop('Selected borehole')

        # Move the scour hole geometry to the end
        input_vars.pop('Scour hole geometry')
        input_vars['Scour hole geometry'] = copy.deepcopy(self.input['Scour hole geometry'])

        if not self.is_computing_shear_decay:
            # if 'Use CSU equation' in input_vars:
            #     input_vars.pop('Use CSU equation')
            if 'Alpha value for local shear decay' in input_vars:
                input_vars.pop('Alpha value for local shear decay')
            if 'Beta value for local shear decay' in input_vars:
                input_vars.pop('Beta value for local shear decay')
        else:
            # input_vars['Use CSU equation'] = self.input['Use CSU equation']
            input_vars['Alpha value for local shear decay'] = self.input['Alpha value for local shear decay']
            input_vars['Beta value for local shear decay'] = self.input['Beta value for local shear decay']

        self.pop_property_vars(input_vars)

        return input_vars

    def get_results_tab_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
              result_vars (dictionary of variables): the input variables
        """
        results_vars = {}

        basic_results = {}
        complete_results = {}

        can_compute = self.can_compute  # Sets the D50s

        if self.gradations:
            # Cohesive
            if self.d50 <= 0.0 and self.critical_shear_stress > 0.0:
                basic_results['Critical shear stress (τc)'] = self.results['Results']['Critical shear stress (τc)']
                complete_results['Critical shear stress (τc)'] = self.results['Results']['Critical shear stress (τc)']
            else:
                basic_results['Approach D50'] = self.results['Results']['Approach D50']
                complete_results['Approach D50'] = self.results['Results']['Approach D50']
                basic_results['Contracted D50'] = self.results['Results']['Contracted D50']
                complete_results['Contracted D50'] = self.results['Results']['Contracted D50']

        if not can_compute:
            results_vars['Basic Results'] = basic_results
            results_vars['Complete Results'] = complete_results
            return results_vars

        basic_results['Amplification factor (α)'] = self.results[
            'Amplification factor (α)']
        complete_results['Amplification factor (α)'] = self.results[
            'Amplification factor (α)']
        complete_results['q2 / q1'] = self.results['q2 / q1']
        complete_results['Governing scour method'] = self.results[
            'Governing scour method']

        basic_results['Flow depth including scour (yc)'] = self.results[
            'Flow depth including scour (yc)']
        complete_results['Flow depth including scour (yc)'] = self.results[
            'Flow depth including scour (yc)']

        basic_results['Max flow depth including scour (ymax)'] = \
            self.results['Max flow depth including scour (ymax)']
        complete_results['Max flow depth including scour (ymax)'] = \
            self.results['Max flow depth including scour (ymax)']

        contraction_method = self.results['Governing scour method'].get_val()
        if contraction_method == 'Clear water':
            complete_results[
                'Flow depth including scour using clear water method (yc)'] = self.results[
                    'Flow depth including scour using clear water method (yc)']
            complete_results[
                'Max flow depth including scour using clear water method (ymax)'] = self.results[
                    'Max flow depth including scour using clear water method (ymax)']

        longterm_degradation = 0.0
        if not self.stand_alone_calc:
            self.results['longterm degradation'].set_val(longterm_degradation)

        basic_results['Scour hole depth (yc)'] = self.results[
            'Scour hole depth (yc)']
        complete_results['Scour hole depth (yc)'] = self.results[
            'Scour hole depth (yc)']

        # if self.input['Scour hole geometry'].get_val().can_compute:
        basic_results['Bottom width'] = self.input['Scour hole geometry'].get_val().results['Bottom width']
        complete_results['Bottom width'] = self.input['Scour hole geometry'].get_val().results['Bottom width']

        basic_results['Top width'] = self.input['Scour hole geometry'].get_val().results['Top width']
        complete_results['Top width'] = self.input['Scour hole geometry'].get_val().results['Top width']

        results_vars['Basic Results'] = basic_results
        results_vars['Complete Results'] = complete_results

        return results_vars

    # def assign_results(self):
    #     """Assigns the results from a computational run to the appropriate locations."""
    #     # Assign the results
    #     _, gamma = self.get_setting('Unit weight of water (γw)')

    def check_warnings(self):
        """Checks for warnings that are given during computations or a check if we can compute (get_can_compute).

        Returns:
            list of str: The warnings found (if any)
        """
        return self.warnings

    def clear_results(self):
        """Clears the results and those of subclasses to prepare for computation."""
        if self.clear_my_own_results:
            self.warnings = []
