"""CalcData for performing Pier Scour calculations."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import math
import sys

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.scour_base_calc import ScourBaseCalc
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation


class PierScourCalc(ScourBaseCalc):
    """A class that defines a pier scour at a bridge contraction."""

    def _get_can_compute(self):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Args:
            unknown (string): variable that is unknown and being calculated.

        Returns:
            bool: True if can compute
        """
        result = True

        _, zero_tol = self.get_data('Zero tolerance')

        if self.gradations:
            self.update_gradation_lists()
        else:
            _, self.d50 = self.get_data('Contracted D50')
            # self.upstream_d50 = self.get_data('Approach D50')
            _, self.critical_shear_stress = self.get_data('Critical shear stress (τc)')

        # self.results['Results']['Approach D50'].set_val(self.upstream_d50)
        if 'Results' not in self.results:
            self.results['Results'] = {}
        self.results['Results']['Contracted D50'] = self.d50
        self.results['Results']['Critical shear stress (τc)'] = self.critical_shear_stress

        # cohesive
        _, computation_method = self.get_data('Computation method')
        if self.d50 <= 0.0 and self.critical_shear_stress > 0.0:
            _, shape = self.get_data('Pier shape (K1)', prior_keys=['Pier stem parameters'])
            _, var = self.get_data('Pier width (ap)', prior_keys=['Pier stem parameters'])
            if not var > zero_tol:
                result = False
                self.warnings['Pier width (ap)'] = 'Please enter the Pier width (ap)'

            _, var = self.get_data('Pier length (Lp)', prior_keys=['Pier stem parameters'])
            if not var > zero_tol and shape not in ['Circular Cylinder', 'Group of Cylinders']:
                result = False
                self.warnings['Pier length (Lp)'] = 'Please enter the Pier length (Lp)'

            result = self.check_float_vars_to_greater_zero(
                ['Velocity upstream of pier (v1)', 'Critical velocity (vc)'])

        else:
            vars_to_check = ['Depth upstream of pier (y1)', 'Velocity upstream of pier (v1)']
            result = self.check_float_vars_to_greater_zero(vars_to_check)

            self.input_dict['calc_data']['Pier stem parameters']['calculator'].is_hec18_complex = \
                computation_method in ['HEC-18 Complex Pier']
            if computation_method in ["'Rule of Thumb' Simple Pier"]:
                vars_to_check = ['Pier width (ap)']
                if not self.check_float_vars_to_greater_zero(vars_to_check):
                    result = False
                _, shape = self.get_data('Pier shape (K1)', prior_keys=['Pier stem parameters'])
                if shape in ['Square Nose', 'Sharp Nose']:
                    result = False
                    self.warnings['Aligned round nose piers'] = \
                        "This method only applies to round nose piers aligned with the flow."
                elif shape == 'Round Nose':
                    _, angle_of_attack = self.get_data('Angle of attack (K2)')
                    if angle_of_attack > zero_tol:
                        result = False
                        self.warnings['Aligned round nose piers'] = \
                            "This method only applies to round nose piers aligned with the flow."
            elif not self.input_dict['calc_data']['Pier stem parameters']['calculator'].get_can_compute_with_subdict(
                    self.input_dict, self.input_dict['calc_data']['Pier stem parameters'])[0]:
                result = False
                self.warnings.update(self.input_dict['calc_data']['Pier stem parameters']['calculator'].warnings)

            # if computation_method in ['HEC-18 Simple Pier']:
            #     if self.get_data('Apply wide pier factor (Kw)'):
            #         if self.d50 < zero_tol:
            #             self.warnings.append("Please enter the contracted D50 to apply the wide pier factor")
            #             result = False

            if computation_method in ['HEC-18 Complex Pier']:
                if not self.input_dict['calc_data']['Pile cap parameters']['calculator'].get_can_compute_with_subdict(
                        self.input_dict, self.input_dict['calc_data']['Pile cap parameters'])[0]:
                    result = False
                    self.warnings.update(self.input_dict['calc_data']['Pile cap parameters']['calculator'].warnings)

                if self.scour_case in ['Case 1', 'Both']:
                    can_compute, warnings = self.input_dict['calc_data']['Pile group parameters']['calculator']. \
                        get_can_compute_with_subdict(self.input_dict,
                                                     self.input_dict['calc_data']['Pile group parameters'])
                    # self.get_data('Pile group parameters').get_can_compute()
                    self.warnings.update(warnings)

            if computation_method in ['FDOT Simple Pier', 'FDOT Complex Pier']:
                _, d50 = self.get_data('Contracted D50')
                _, d50_in_mm = self.unit_converter.convert_units('ft', 'mm', d50)
                if not self.check_float_vars_to_greater_zero(['Contracted D50']):
                    result = False
                    self.warnings['D50 for FDOT'] = "Please enter the Contracted D50 to compute the FDOT method"
                elif 0.0 < d50_in_mm < 0.1:
                    self.warnings['D50 too small for FDOT'] = "D50 is too small for FDOT method"
                    result = False

            if computation_method in ['FDOT Complex Pier']:
                _, compute_pier_stem = self.get_data('Compute pier stem', prior_keys=['Pier stem parameters'])
                _, compute_pile_cap = self.get_data('Compute pile cap', prior_keys=['Pile cap parameters'])
                _, compute_pile_group = self.get_data('Compute pile group', prior_keys=['Pile group parameters'])

                if not compute_pier_stem and not compute_pile_cap and not compute_pile_group:
                    self.warnings['Contracted D50 for FDOT'] = 'Please select some scour components to compute'
                    result = False

                if compute_pier_stem and not compute_pile_cap and not compute_pile_group:
                    self.warnings['FDOT simple'] = 'Please use the FDOT simple method to compute pier stem only - ' \
                        'otherwise, compute pile cap and/or pile group'
                    result = False

                if not compute_pier_stem and compute_pile_cap and not compute_pile_group:
                    self.warnings['FDOT simple 2'] = 'Please use the FDOT simple method to compute pile cap scour ' \
                        'component by treating the pile cap as a pier stem'
                    result = False

                if compute_pier_stem and not compute_pile_cap and compute_pile_group:
                    self.warnings['Add pile cap'] = 'Please add a pile cap to join the pier stem to the pile group'
                    result = False

                if compute_pier_stem:
                    can_compute, warnings = self.input_dict['calc_data']['Pier stem parameters']['calculator']. \
                        get_can_compute_with_subdict(self.input_dict,
                                                     self.input_dict['calc_data']['Pier stem parameters'])
                    if not can_compute:
                        result = False
                    self.warnings.update(warnings)

                if compute_pile_cap:
                    can_compute, warnings = self.input_dict['calc_data']['Pile cap parameters']['calculator']. \
                        get_can_compute_with_subdict(self.input_dict,
                                                     self.input_dict['calc_data']['Pile cap parameters'])
                    if not can_compute:
                        result = False
                    self.warnings.update(warnings)

                if compute_pile_group:
                    can_compute, warnings = self.input_dict['calc_data']['Pile group parameters']['calculator']. \
                        get_can_compute_with_subdict(self.input_dict,
                                                     self.input_dict['calc_data']['Pile group parameters'])
                    if not can_compute:
                        result = False
                    self.warnings.update(warnings)

        return result

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        # Document the source of the equation (if possible):
        #   Give publication, date, page number and link (if there's a bug, that it can be traced back to the source).
        # Assign new variables from the input_dict in similar names as the equation and document units

        self.scour_depth = 0.0
        # gradation lists updated in get_can_compute
        # self.update_gradation_lists()

        _, computation_method = self.get_data('Computation method')

        # Check if the soil is cohesive
        if self.d50 <= 0.0 and self.critical_shear_stress > 0.0:
            self.compute_cohesive()
            self.compute_partial_scour_depth()
        else:
            if computation_method in ["'Rule of Thumb' Simple Pier"]:
                self.compute_rule_of_thumb()
            elif computation_method in ['HEC-18 Simple Pier', 'HEC-18 Complex Pier']:
                self.compute_hec18()
                if computation_method in ['HEC-18 Complex Pier']:
                    self.compute_hec18_complex()

            elif computation_method in ['FDOT Simple Pier']:
                self.compute_fdot()
            elif computation_method in ['FDOT Complex Pier']:
                self.compute_fdot_complex()

        if self.is_computing_shear_decay:
            self.compute_shear_decay()

        self.compute_scour_hole_geometry(self.get_data('Pier width (ap)', prior_keys=['Pier stem parameters'])[1],
                                         is_pier_scour_hole=True)
        return True

    def compute_cohesive(self):
        """Compute the cohesive soil scour depth."""
        # C++ cohesive scour method
        _, g = self.get_data('Gravity')
        _, length = self.get_data('Pier length (Lp)', prior_keys=['Pier stem parameters'])
        _, ap = self.get_data('Pier width (ap)', prior_keys=['Pier stem parameters'])
        _, shape = self.get_data('Pier shape (K1)', prior_keys=['Pier stem parameters'])
        _, v1 = self.get_data('Velocity upstream of pier (v1)')
        _, vc = self.get_data('Critical velocity (vc)')

        k1 = self._compute_shape_factor(shape)
        self.results['Results']['Pier shape factor (k1)'] = k1

        k2 = self._compute_angle_of_attack_factor(length, ap, shape)
        self.results['Results']['Pier angle of attack factor (k2)'] = k2

        mcf = self.compute_column_group_factors()

        if (2.6 * v1 - vc) > 0.0:
            self.cohesive_scour_depth = 2.2 * k1 * k2 * ap ** 0.65 * ((2.6 * v1 - vc) / math.sqrt(g)) ** 0.7 * mcf
        else:
            self.cohesive_scour_depth = 0.0
        self.scour_depth = self.cohesive_scour_depth
        self.results['Cohesive scour depth (ys)'] = self.cohesive_scour_depth
        self.results['Results']['Scour depth (ys)'] = self.cohesive_scour_depth

        return self.cohesive_scour_depth

    def compute_column_group_factors(self):
        """Compute the factors for a column group."""
        # if __debug__:
        #     raise NotImplementedError("We don't currently support entering in the number of cylinders")
        return 1.0

        # shape = self.get_data('Pier stem parameters').input['Pier shape (K1)')
        # angle_of_attack_of_flow = self.get_data('Angle of attack (K2)')
        # m_pierWidth = self.get_data('Pier stem parameters').input['Pier width (ap)')

        # s_over_a = m_pileColSpacing / m_pierWidth

        # multiple_column_factor = 1.0

        # if shape == 'Group of Cylinders' and s_over_a > 5.0 and angle_of_attack_of_flow > 0.01:
        #     multiple_column_factor = 1.2
        # elif shape == 'Group of Cylinders' and s_over_a <= 5.0 and angle_of_attack_of_flow > 0.01:
        #     # TODO: We don't currenty support entering in the number of cylinders
        #     pass
        #     # L = m_numberOfPilesInAColumn * m_pierWidth
        #     # angle_in_radians = UnitsConversion::ForceConvertUnits("degrees", "radians", angle_of_attack_of_flow)
        #     # m_factorForAngleOfAttackOfFlow = pow(cos(angle_in_radians) + L / m_pierWidth * sin(angle_in_radians),
        # 0.65)

        # return multiple_column_factor

    def compute_partial_scour_depth(self):
        """Compute the partial scour depth for cohesive soils."""
        # TODO: Refactor this in accordance with the example at 7.39
        _, t = self.get_data('Duration of flow event (t)', prior_keys=['Scour from single event'])
        _, n = self.get_data("Manning's n (n)", prior_keys=['Scour from single event'])
        _, y1 = self.get_data('Depth upstream of pier (y1)')
        _, v1 = self.get_data('Velocity upstream of pier (v1)')

        self.initialStress = 0.0
        self.partial_scour_depth = 0.0

        if y1 > 0.0 and v1 > 0.0 and n > 0.0:
            _, shape = self.get_data('Pier shape (K1)', prior_keys=['Pier stem parameters'])
            # ['Square Nose', 'Round Nose', 'Sharp Nose', 'Circular Cylinder', 'Group of Cylinders',]
            k = 1.5  # for circular piers
            if shape in ['Square Nose', 'Round Nose', 'Sharp Nose']:
                k = 1.7
            _, ku = self.get_data('Manning constant', 1.486)
            _, gamma_w = self.get_data('Unit weight of water (γw)')

            self.shear_stress = gamma_w / (y1 ** (1.0 / 3.0)) * (n * k * v1 / ku) ** 2.0
            self.results['Initial shear stress at the pier (τpier)'] = self.shear_stress

            _, initial_erosion_rate = self.get_data('Initial erosion rate', prior_keys=['Scour from single event'])

            if t > 0.0 and self.cohesive_scour_depth > 0.0 and initial_erosion_rate > 0.0:
                self.partial_scour_depth = t / (1 / initial_erosion_rate + t / self.cohesive_scour_depth)
                self.results['Partial scour depth'] = self.partial_scour_depth

    def compute_shear_decay(self):
        """Compute the shear decay for contraction scour."""
        if self.gradations is None:
            self.warnings['Gradations not defined'] = 'Gradations must be defined to compute shear decay.'
            return
        # B=self.B     # Pier width (ft)
        _, ap = self.get_data('Pier width (ap)', prior_keys=['Pier stem parameters'])
        # lp = self.get_data('Pier stem parameters').input['Pier length (Lp)']
        # K1=self.K1   # Correction Factor: Pier Nose Shape (Square 1.1, Round 1, Sharp Nose 0.9)
        k1 = self.results['Results']['Pier shape factor (k1)']
        # K2=self.K2   # Correction Factor: Angle of Attack (Update using formulas in HEC-18 later)
        # k2 = self.results['Case1']['Pile cap angle of attack factor (k2)')
        k2 = self.results['Results']['Pier angle of attack factor (k2)']
        k3 = self.results['Results']['Bed form factor (k3)']

        # at Local Approach
        # at highest unit q if scour reference point = thalweg, or directly upstream of pier if local
        _, y1 = self.get_data('Depth upstream of pier (y1)')
        _, v1 = self.get_data('Velocity upstream of pier (v1)')
        qpmax = v1 * y1

        thalweg = self.local_streambed_after_cs_ltd
        original_streambed = self.streambed_elev_prior_to_scour
        cs_and_ltd = original_streambed - thalweg
        if cs_and_ltd < 0.0:
            cs_and_ltd = 0.0
        wse = original_streambed + y1
        # Use the WSE, if we have it
        if self.wse_x is not None and max(self.wse_x) > self.centerline and min(self.wse_x) < self.centerline:
            wse = float(np.interp(self.centerline, self.wse_x, self.wse_y))

        decay_x = []  # Shear
        decay_y = []  # Elevation
        marker_x = []
        marker_y = []

        _, include_contraction_shear = self.get_data('Include contraction shear')
        if include_contraction_shear:
            decay_x = copy.deepcopy(self.con_decay_x)
            decay_y = copy.deepcopy(self.con_decay_y)
            marker_x = copy.deepcopy(self.con_decay_x_markers)
            marker_y = copy.deepcopy(self.con_decay_y_markers)

        # Enter decay function parameters (Later provide a lookup table based on Pier Shape)
        shear_alpha, shear_beta = self.compute_alpha_beta_for_pier_shape()

        _, g = self.get_data('Gravity')

        # Calculate flow conditions after contraction scour
        yp = y1 + cs_and_ltd  # New depth after contraction scour
        v2 = qpmax / yp
        fr2 = (v2) / (g * yp) ** 0.5

        if 'Shear decay results' not in self.results:
            self.results['Shear decay results'] = {}
        self.results['Shear decay results']['Unit discharge'] = qpmax
        self.results['Shear decay results']['Shear decay alpha'] = shear_alpha
        self.results['Shear decay results']['Shear decay beta'] = shear_beta
        self.results['Shear decay results']['New depth after cs and ltd (yp)'] = yp
        self.results['Shear decay results']['New velocity after cs and ltd (v2)'] = v2

        _, use_csu = self.get_data('Use CSU equation')

        self.compute_local_shear_decay(
            decay_x, decay_y, marker_x, marker_y, ap, k1, k2, k3, fr2, shear_alpha,
            shear_beta, wse, v2, yp, thalweg, g, qpmax, use_csu)

    def compute_local_shear_decay(self, decay_x, decay_y, marker_x, marker_y, ap, k1, k2, k3, fr2, shear_alpha,
                                  shear_beta, wse, v2, yp, thalweg, g, qpmax, use_csu):
        """Compute the shear decay for contraction scour.

        Args:
            decay_x (list): The x values for the decay curve.
            decay_y (list): The y values for the decay curve.
            marker_x (list): The x values for the markers.
            marker_y (list): The y values for the markers.
            ap (float): The pier width.
            k1 (float): The angle of attack.
            k2 (float): The bed condition.
            k3 (float): The contraction scour factor.
            fr2 (float): The Froude number.
            shear_alpha (float): The shear alpha.
            shear_beta (float): The shear beta.
            wse (float): The water surface elevation.
            v2 (float): The velocity.
            yp (float): The new depth after cs and ltd.
            thalweg (float): The thalweg elevation.
            g (float): The gravity.
            qpmax (float): The maximum flow.
            use_csu (bool): Whether to use the CSU equation or the Hager equation for ymax.
        """
        # Already done in scour_base class
        # bh_uuid = self.get_data('Selected borehole')
        # bh = self.bh_dict[bh_uuid]
        # self.bh_layers = bh.input['Layers').item_list

        # Get starting soil layer
        self.layer_index, cur_layer_top, cur_layer_bottom, cur_critical_shear = self.get_layer_index_from_elevation(
            thalweg)
        mannings_n = self.bh_layers[self.layer_index]['calculator'].manning_n
        cur_mannings_n = mannings_n
        lower_layer_n, lower_layer_shear, lower_layer_critical = self.get_next_layer_details(self.layer_index, wse,
                                                                                             qpmax)
        if len(self.bh_layers) == 1:
            cur_layer_top = thalweg
            cur_layer_bottom = -sys.float_info.max

        cur_decay_y = thalweg
        shear = self.compute_shear(wse, cur_decay_y, cur_mannings_n, qpmax)
        _, decay_inc = self.get_data('Shear decay curve increment', 0.5)

        _, min_thickness = self.get_data('Min layer thickness', 2.0)

        # Add Hager Equation as alternative y_max (g=9.81 m/s^2, Sg=2.65 ) --- Noncohesive soils
        d50 = self.bh_layers[self.layer_index]['calculator'].d50
        # sigma = self.bh_layers[self.layer_index]['calculator'].results['Shear decay parameters'][
        #     'Sediment gradation coefficient (σ)']
        sigma = self.bh_layers[self.layer_index]['calculator'].tau
        gamma_s = self.bh_layers[self.layer_index]['Unit weight of soil (γs)']
        cur_hager, ymax_hager = self.compute_hager(v2, g, yp, d50, sigma, ap, k1, k2, gamma_s)
        ymax_csu = self.compute_csu(yp, k1, k2, k3, fr2, ap)

        self.results['Shear decay results']['Computed Hager number'] = cur_hager
        self.results['Shear decay results']['Max depth computed from Hager number'] = ymax_hager
        self.results['Shear decay results']['Max depth computed from CSU equation'] = ymax_csu
        self.results['Shear decay results']['Unadjusted computed local shear'] = shear

        ymax_depth = ymax_hager
        _, use_csu = self.get_data('Use CSU equation')
        if use_csu:
            ymax_depth = ymax_csu
        ymax = thalweg - ymax_depth

        # Calculate amplified tau at the pier.
        self.tau_local = shear_alpha * shear
        cur_shear = self.tau_local

        # Enter zero depth data (adjustment blows up with zero increment)
        scour_depth = 0.0
        decay_x.append(self.tau_local)
        decay_y.append(cur_decay_y)
        marker_x.append(self.tau_local)
        marker_y.append(cur_decay_y)

        # Increment data
        scour_depth += decay_inc
        cur_decay_y -= decay_inc
        cur_thickness = cur_decay_y - cur_layer_bottom

        # Check if current layer is getting too thin to trust
        cur_thickness = cur_decay_y - cur_layer_bottom
        lower_layer_weaker = False
        if lower_layer_n is not None:
            lower_layer_weaker = lower_layer_critical < cur_critical_shear and cur_thickness < min_thickness and \
                lower_layer_shear > lower_layer_critical

        layers_too_thin = []
        k_roughness = 1.0

        change_layer = False

        # Determine shear stress until criteria is met (converge on critical shear, or layer is too thin)
        while cur_shear > cur_critical_shear and cur_decay_y > ymax or \
                (cur_shear <= cur_critical_shear and lower_layer_weaker):
            if not cur_shear > cur_critical_shear and cur_decay_y > ymax and self.layer_index not in \
                    layers_too_thin:
                layers_too_thin.append(self.layer_index)
                self.warnings[f'Layer {str(self.layer_index)} is too thin'] = f'Layer {str(self.layer_index)} is too ' \
                    'thin to trust; continuing the shear decay curve.'
            change_layer = False
            if not cur_layer_top >= cur_decay_y >= cur_layer_bottom:
                change_layer = True
                # Determine the shear stress at the bottom of the layer
                step = thalweg - cur_layer_bottom
                cur_shear = float(self.tau_local * k_roughness * np.exp(-1 * shear_beta * (step / ymax_depth)))
                decay_x.append(cur_shear)
                decay_y.append(cur_layer_bottom)
                marker_x.append(cur_shear)
                marker_y.append(cur_layer_bottom)
                # Update data from the layers
                self.layer_index, cur_layer_top, cur_layer_bottom, cur_critical_shear = \
                    self.get_layer_index_from_elevation(cur_decay_y)
                cur_mannings_n = self.bh_layers[self.layer_index]['calculator'].manning_n
                lower_layer_n, lower_layer_shear, lower_layer_critical = self.get_next_layer_details(
                    self.layer_index, wse, qpmax)

                # K_roughness=(self.soil_summary.loc[i,'Manning n']/self.soil_summary.loc[idx,'Manning n'])**2
                k_roughness = (cur_mannings_n / mannings_n) ** 2

                # Determine the shear stress at the top of the new layer
                # step = thalweg - cur_layer_top
                cur_shear = float(self.tau_local * k_roughness * np.exp(-1 * shear_beta * (step / ymax_depth)))
                decay_x.append(cur_shear)
                decay_y.append(cur_layer_top)
                marker_x.append(cur_shear)
                marker_y.append(cur_layer_top)

            if cur_shear > cur_critical_shear:
                cur_shear = float(self.tau_local * k_roughness * np.exp(-1 * shear_beta * (scour_depth / ymax_depth)))

                decay_x.append(cur_shear)
                decay_y.append(cur_decay_y)
                # If we overshot, correct it:
                if cur_shear < cur_critical_shear:
                    _, null_data = self.get_data('Null data')
                    _, zero_tol = self.get_data('Zero tolerance')
                    shear_interp = Interpolation(decay_x, decay_y, null_data=null_data, zero_tol=zero_tol)
                    decay_y[-1], _ = shear_interp.interpolate_y(cur_critical_shear)
                    decay_x[-1] = cur_critical_shear

            if cur_shear > cur_critical_shear or not change_layer:
                # Update the current elevation
                scour_depth += decay_inc
                cur_decay_y -= decay_inc

            # Check for thin layer
            cur_thickness = cur_decay_y - cur_layer_bottom
            if lower_layer_n is not None:  # Check if current layer is getting too thin to trust
                lower_layer_weaker = lower_layer_critical < cur_critical_shear and cur_thickness < min_thickness \
                    and lower_layer_shear > lower_layer_critical

        if marker_y[-1] != decay_y[-1]:
            marker_x.append(decay_x[-1])
            marker_y.append(decay_y[-1])

        self.results['Shear decay results']['Shear decay curve shear'] = decay_x
        self.results['Shear decay results']['Shear decay curve elevation'] = decay_y
        self.results['Shear decay results']['Shear decay curve shear markers'] = marker_x
        self.results['Shear decay results']['Shear decay curve elevation markers'] = marker_y

        # Set the determined scour depth
        self.scour_depth = thalweg - decay_y[-1]

        self.set_layer_plot_data(decay_x, decay_y, marker_x, marker_y, ymax, use_csu=use_csu)

    def compute_hager(self, v2, g, y2, d50, sigma, ap, k1, k2, gamma_s):
        """Compute the max scour depth using the Hager equation.

        Args:
            v2 (float): The velocity.
            g (float): The gravity.
            y2 (float): The depth of the flow.
            d50 (float): The D50 value.
            sigma (float): The sediment gradation coefficient.
            ap (float): The pier width.
            k1 (float): The angle of attack.
            k2 (float): The bed condition.

        Returns:
            float: The shear stress at the given elevation
        """
        _, gamma_w = self.get_data('Unit weight of water (γw)')
        s = gamma_s / gamma_w
        # Add Hager Equation as alternative y_max (g=9.81 m/s^2, Sg=2.65 ) --- Noncohesive soils
        # H=(self.V2*0.3048)/(9.81*(2.65-1)*(self.soil_summary.loc[idx,'D50 (mm)']/1000))**0.5
        hager = 0.0
        if d50 > 0.0 and s > 1.0:
            hager = v2 / (g * (s - 1) * d50) ** 0.5
        # self.ymaxHager = np.around((ap*0.3048)**0.62*(self.y2*0.3048)**0.38*(1.32*K1*K2*np.tanh(H**2/(
        #     1.97*self.soil_summary.loc[idx,'Sigma']**1.5)))*3.281,decimals=2)
        ymax_hager = 0.0
        if hager > 0.0:
            ymax_hager = ap ** 0.62 * (y2) ** 0.38 * (1.32 * k1 * k2 * math.tanh(hager ** 2.0 / (1.97 * sigma ** 1.5)))

        return hager, ymax_hager

    def compute_csu(self, y2, k1, k2, k3, fr2, ap):
        """Compute the max scour depth using the CSU equation.

        Args:
            y2 (float): The depth of the flow.
            k1 (float): The angle of attack.
            k2 (float): The bed condition.
            k3 (float): The contraction scour factor.
            fr2 (float): The Froude number.
            ap (float): The pier width.

        Returns:
            float: The shear stress at the given elevation
        """
        ymax_csu = y2 * (2.0 * k1 * k2 * k3 * (ap / y2) ** 0.65 * fr2 ** 0.43)
        return ymax_csu

    def compute_rule_of_thumb(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        _, self.y1 = self.get_data('Depth upstream of pier (y1)')
        _, self.v1 = self.get_data('Velocity upstream of pier (v1)')
        _, a = self.get_data('Pier width (ap)', prior_keys=['Pier stem parameters'])

        _, g = self.get_data('Gravity')

        fr1 = self.v1 / (g * self.y1)**0.5
        self.results['Results']['Upstream Froude number (Fr1)'] = fr1

        check_factor = 2.4
        if fr1 > 0.8:
            check_factor = 3.0
        max_scour_depth = check_factor * a

        self.results['Results']['Scour factor'] = check_factor
        self.results['Results']['Scour depth (ys)'] = max_scour_depth
        self.scour_depth = max_scour_depth
        return True

    def compute_fdot(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        self.scour_depth = 0.0
        _, shape = self.get_data('Pier shape (K1)', prior_keys=['Pier stem parameters'])

        _, y1 = self.get_data('Depth upstream of pier (y1)')

        self.ks, self.kp, d_star = self._compute_ks_kp_dstar(shape)
        if 'FDOT Complex' not in self.results:
            self.results['FDOT Complex'] = {}
        self.results['FDOT Complex']['Shape factor (ks)'] = self.ks
        self.results['FDOT Complex']['Projection factor (kp)'] = self.kp
        self.results['FDOT Complex']['Shape and projection factors (d*)'] = d_star

        ysmax = self._compute_fdot_simple_pier_with_shape(y1, d_star)

        self.scour_depth = ysmax
        self.ys = ysmax
        self.results['Results']['Scour depth (ys)'] = ysmax

        self.scour_depth = ysmax

        return True

    def _compute_fdot_simple_pier_with_shape(self, y, d_star):
        """Computes the ysmx value for the simple pier.

        Args:
            shape (str): The shape of the pier

        Returns:
            float: The ysmx value
        """
        _, zero_tol = self.get_data('Zero tolerance')
        if d_star < zero_tol:
            return 0.0
        _, v1 = self.get_data('Velocity upstream of pier (v1)')

        v1p, vc, f1, f2, f3 = self._compute_v1_vc_f1_f2_f3_factors(v1, y, d_star)
        if 'FDOT Complex' not in self.results:
            self.results['FDOT Complex'] = {}
        self.results['FDOT Complex']['Live-bed peak velocity (v1p)'] = v1p
        self.results['FDOT Complex']['Critical velocity (vc)'] = vc
        self.results['FDOT Complex']['FDOT factor 1 (f1)'] = f1
        self.results['FDOT Complex']['FDOT factor 2 (f2)'] = f2
        self.results['FDOT Complex']['FDOT factor 3 (f3)'] = f3

        v1_over_vc = v1 / vc
        v1p_over_vc = v1p / vc
        if 0.4 <= v1_over_vc <= 1:
            ysmax = d_star * f1 * f2 * f3 * 2.5
        elif v1_over_vc > 1 and v1_over_vc <= v1p_over_vc:
            x = 2.2 * (v1_over_vc - 1) / (v1p_over_vc - 1)
            y = 2.5 * f3 * (v1p_over_vc - v1_over_vc) / (v1p_over_vc - 1)
            ysmax = d_star * f1 * (x + y)
        elif v1_over_vc > v1p_over_vc:
            ysmax = d_star * f1 * 2.2
        else:
            ysmax = 0.0
            self.warnings['velocity ratio'] = "The velocity ratio is out of range for the FDOT method"

        self.scour_depth = ysmax
        return ysmax

    def compute_fdot_complex(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        # Gather and Adjust data
        _, compute_pier_stem = self.get_data('Compute pier stem', prior_keys=['Pier stem parameters'])
        _, compute_pile_cap = self.get_data('Compute pile cap', prior_keys=['Pier stem parameters'])
        _, compute_pile_group = self.get_data('Compute pile group', prior_keys=['Pier stem parameters'])

        _, angle_of_attack = self.get_data('Angle of attack (K2)')

        kpp = 0.0
        kppc = 0.0
        kppg = 0.0
        _, y1 = self.get_data('Depth upstream of pier (y1)')
        _, h0 = self.get_data('Initial height of the pile cap above bed (h0)', prior_keys=['Pile cap parameters'])
        _, t = self.get_data('Pile cap thickness (T)', prior_keys=['Pile cap parameters'])
        y0 = y1
        y0_changed = True
        pier_angle_of_attack = angle_of_attack
        pc_angle_of_attack = angle_of_attack
        pg_angle_of_attack = angle_of_attack

        if not compute_pile_cap:
            t = 0.0
        hp = h0 + t

        if compute_pier_stem:
            _, ap = self.get_data('Pier width (ap)', prior_keys=['Pier stem parameters'])
            _, lp = self.get_data('Pier length (Lp)', prior_keys=['Pier stem parameters'])
            _, p_shape = self.get_data('Pier shape (K1)', prior_keys=['Pier stem parameters'])
            if angle_of_attack > 45:
                ap, lp = lp, ap
                if p_shape == 'Sharp Nose':
                    p_shape = 'Square Nose'
                pier_angle_of_attack = 90 - angle_of_attack

        if compute_pile_cap:
            _, apc = self.get_data('Pile cap width (apc)', prior_keys=['Pile cap parameters'])
            _, lpc = self.get_data('Pile cap length (Lpc)', prior_keys=['Pile cap parameters'])
            _, pc_shape = self.get_data('Pile cap shape (K1)', prior_keys=['Pile cap parameters'])
            if angle_of_attack > 45:
                apc, lpc = lpc, apc
                if pc_shape == 'Sharp Nose':
                    pc_shape = 'Square Nose'
                pc_angle_of_attack = 90 - angle_of_attack

        if compute_pile_group:
            _, m = self.get_data('Pile group number of rows (m)', prior_keys=['Pile group parameters'])
            _, n = self.get_data('Pile group number of columns (n)', prior_keys=['Pile group parameters'])
            _, sm = self.get_data('Pile group spacing between rows (sm)', prior_keys=['Pile group parameters'])
            _, sn = self.get_data('Pile group spacing between columns (sn)', prior_keys=['Pile group parameters'])
            _, pg_shape = self.get_data('Pile nose shape (K1)', prior_keys=['Pile group parameters'])
            if angle_of_attack > 45:
                m, n, sm, sn = n, m, sn, sm
                pg_angle_of_attack = 90 - angle_of_attack
            _, a = self.get_data('Pile width (a)', prior_keys=['Pile group parameters'])

        dp_star = 0.0
        dpc_star = 0.0
        dpgi_star = 0.0

        ksp = 0.0
        kspc = 0.0
        kpgs = 0.0

        while y0_changed:
            if compute_pier_stem:
                kpp = self._compute_pier_or_pile_cap_projection_factor(ap, lp, p_shape, pier_angle_of_attack)
                ksp = self._compute_pier_or_pile_cap_shape_factor(pier_angle_of_attack, ap, lp, p_shape)
                khip = self._compute_vertical_position_factors(hp, y0)
                dp_star = 1.8 * kpp * ksp * khip

            if compute_pile_cap:
                kppc = self._compute_pier_or_pile_cap_projection_factor(apc, lpc, pc_shape, pc_angle_of_attack)
                kspc = self._compute_pier_or_pile_cap_shape_factor(pc_angle_of_attack, apc, lpc, pc_shape)
                khipc = self._compute_vertical_position_factors(h0, y0)
                kti = min(t / y0, 1)
                dpc_star = 1.8 * kppc * kspc * khipc * kti

            if compute_pile_group:
                kppg = self._compute_pile_group_projection_factor(a, m, n, pg_angle_of_attack)
                kpgs = 1.4
                khipg = self._compute_pile_group_vertical_position_factors(h0, y0, kppg)
                dpgi_star = 3.5 * kppg * kpgs * khipg

            y0_max_i = 8 * max(kpp, kppc, kppg)
            y0 = min(y0_max_i, y0)
            if y0 != y0_max_i:
                y0_changed = False

        # End of loop
        dcsi_star = dp_star + dpc_star + dpgi_star
        y_big = min(y0, 8 * dcsi_star)
        ysmax = self._compute_fdot_simple_pier_with_shape(y_big, dcsi_star)

        y = y_big + ysmax
        hp = hp + ysmax
        hpc = h0 + ysmax
        hpg = h0 + ysmax

        klobp = 1.0
        if compute_pier_stem:
            klobp = self._compute_lob_factor(ap, lp, pier_angle_of_attack, ysmax, y, hp)
        dp_star_alone = ksp * kpp * klobp

        _, zero_tol = self.get_data('Zero tolerance')
        if dp_star_alone < zero_tol:
            ysmax_alone = ysmax
        else:
            ysmax_alone = self._compute_fdot_simple_pier_with_shape(y, dp_star_alone)

        if ysmax - ysmax_alone >= hpc + t:
            dcs_star = dp_star_alone
            self.ys = ysmax_alone
            # Done (scour did not pass pile cap)
            if 'FDOT Complex' not in self.results:
                self.results['FDOT Complex'] = {}
            if 'Pier stem' not in self.results['FDOT Complex']:
                self.results['FDOT Complex']['Pier stem'] = {}
            self.results['FDOT Complex']['Pier stem']['Pier projection factor (kpp)'] = kpp
            self.results['FDOT Complex']['Pier stem']['Pier shape factor (ksp)'] = ksp
            self.results['FDOT Complex']['Pier stem']['Pier length to width factor (klobp)'] = klobp
            self.results['FDOT Complex']['Pier stem']['Pier effective diameter (dp*)'] = dcs_star
        else:
            khp = self._compute_second_vertical_position_for_pier_stem_factors(hp, y, ysmax)
            kf = 1.0
            f_ratio = 0.0
            if compute_pile_cap:
                if compute_pier_stem:
                    f_ratio, kf = self._compute_pile_cap_effect_on_pier_factor(lpc, lp, apc, ap, hp, y, ysmax,
                                                                               pc_angle_of_attack, kpp)
                else:
                    kf = 1.0
                    f_ratio = 0.0
                khpc = self._compute_second_vertical_position_for_pile_cap_factors(hpc, y, t, ysmax)
                klobpc = self._compute_lob_factor(apc, lpc, pc_angle_of_attack, ysmax, y, hpc)
                dpc_star = kspc * kppc * khpc * klobpc
            else:
                kf = 1.0
                khp = 1.0
                klobp = 1.0
                dpc_star = 0.0
                khpc = 0.0
                klobpc = 0.0

            dp_star = ksp * kpp * khp * kf * klobp

            if compute_pile_group:
                kpgp, kpgp1 = self._projected_area(a, sm, sn, m, n, pg_angle_of_attack, pg_shape, 1)
                kpgpe = self._compute_effective_projection_of_pile_group(a, sm, sn, m, n, pg_angle_of_attack, pg_shape,
                                                                         kpgp, kpgp1)
                khpg = self._compute_vertical_position_for_pile_group_factors(ysmax, y, hpg)
                dpg_star = kpgs * kpgpe * khpg
            else:
                dpg_star = 0.0
                kpgp = 0.0
                kpgpe = 0.0
                khpg = 0.0

            dcs_star = dp_star + dpc_star + dpg_star

            _, sf = self.get_data('Safety factor (SF)')
            ysmax2 = self._compute_fdot_simple_pier_with_shape(y1, sf * dcs_star)

            ysmax3 = -1 * sf * (hp - ysmax)

            if dpc_star > 0:
                ysmax_final = max(ysmax2, ysmax3)
            elif dpc_star == 0:
                ysmax_final = ysmax2

            self.ys = ysmax_final

            # Pier Stem
            if 'Pier stem' not in self.results['FDOT Complex']:
                self.results['FDOT Complex']['Pier stem'] = {}
            self.results['FDOT Complex']['Pier stem']['Pier projection factor (kpp)'] = kpp
            self.results['FDOT Complex']['Pier stem']['Pier shape factor (ksp)'] = ksp
            self.results['FDOT Complex']['Pier stem']['Pier stem vertical position factor (khp)'] = khp
            self.results['FDOT Complex']['Pier stem']['Pier length to width factor (klobp)'] = klobp
            self.results['FDOT Complex']['Pier stem']['Pile cap effect on pier stem factor (kf)'] = kf
            self.results['FDOT Complex']['Pier stem']['Ratio of overhang to projected pier width (f_ratio)'] = f_ratio
            self.results['FDOT Complex']['Pier stem']['Pier effective diameter (dp*)'] = dp_star

            # Pile Cap
            if 'Pile cap' not in self.results['FDOT Complex']:
                self.results['FDOT Complex']['Pile cap'] = {}
            self.results['FDOT Complex']['Pile cap']['Pile cap projection factor (kppc)'] = kppc
            self.results['FDOT Complex']['Pile cap']['Pile cap shape factor (kspc)'] = kspc
            self.results['FDOT Complex']['Pile cap']['Pile cap vertical position factor (khpc)'] = khpc
            self.results['FDOT Complex']['Pile cap']['Pile cap length to width factor (klobpc)'] = klobpc
            self.results['FDOT Complex']['Pile cap']['Pile cap effective diameter (dpc*)'] = dpc_star

            # Pile Group
            if 'Pile group' not in self.results['FDOT Complex']:
                self.results['FDOT Complex']['Pile group'] = {}
            self.results['FDOT Complex']['Pile group']['Pile group shape factor (kpgs)'] = kpgs
            self.results['FDOT Complex']['Pile group']['Pile group projection factor (kpgp)'] = kpgp
            self.results['FDOT Complex']['Pile group']['Pile group effective projection factor (kpgpe)'] = kpgpe
            self.results['FDOT Complex']['Pile group']['Pile group vertical position factor (khpg)'] = khpg
            self.results['FDOT Complex']['Pile group']['Pile group effective diameter (dpg*)'] = dpg_star

            self.results['FDOT Complex']['Safety factor for complex pier effective diameter (sf)'] = sf
            self.results['FDOT Complex']['Complex pier effective diameter (dcs*)'] = dcs_star

        self.results['Results']['Scour depth (ys)'] = self.ys

    def _compute_effective_projection_of_pile_group(self, apg, sm, sn, m, n, angle_of_attack, shape, kpgp, kpgp1):
        """Computes the effective projection of the pile group.

        Args:
            apg (float): The width of the pile group
            sm (float): The spacing between rows
            sn (float): The spacing between columns
            m (int): The number of rows
            n (int): The number of columns
            angle_of_attack (float): The angle of attack
            shape (str): The shape of the pier
            kpgp (float): The pile group projection factor
            kpgp1 (float): The pile group projection factor 1

        Returns:
            float: The effective projection of the pile group
        """
        _, pi = self.get_data('Pi')
        api = angle_of_attack / 180
        if n == 1 and m > 1:
            c_s = -8.4 * api + 9.3
        elif n > 1 and m == 1:
            c_s = 8.4 * api + 5.1
        elif m > 1 and n > 1:
            c_s = 9.3

        s = np.min([sm, sn]).item()

        if shape == 'Round Nose':
            bp = apg
        else:  # shape == 'square':
            alpha = angle_of_attack / 180 * pi
            bp = apg * (np.cos(alpha).item() + np.sin(alpha).item())
        soa = s / bp
        fa = c_s * (kpgp - bp) / ((c_s - 1) * kpgp)
        fb = (c_s * bp - kpgp1) / ((c_s - 1) * kpgp)
        kpgpe = kpgp * (fa / soa + fb)

        return kpgpe

    def _compute_vertical_position_for_pile_group_factors(self, ysmax, y, hpg):
        """Computes the vertical position factor for the pile group.

        Args:
            ysmax (float): The maximum scour depth
            y (float): The depth upstream of the pier
            hpg (float): The vertical position of the pile group

        Returns:
            float: The vertical position factor for the pile group
        """
        if hpg > y:
            kh = 1.0
        elif hpg >= ysmax and hpg <= y:
            kh = 1.0 - 0.65 * ((y - hpg) / (y - ysmax))**2
        elif hpg >= 0.0 and hpg < ysmax:
            kh = 0.35 * (hpg / ysmax)
        elif hpg < 0:
            kh = 0.0
        return kh

    def _compute_second_vertical_position_for_pile_cap_factors(self, hpc, y, t, ysmax):
        """Computes the vertical position factor for the pile cap.

        Args:
            hpc (float): The vertical position of the pile cap
            y (float): The depth upstream of the pier
            t (float): The thickness of the pile cap
            ysmax (float): The maximum scour depth

        Returns:
            float: The vertical position factor for the pile cap
        """
        if hpc > y:
            fpch = 0.0
        elif hpc >= ysmax and hpc <= y:
            fpch = 0.65 * ((y - hpc) / (y - ysmax))**2
        elif hpc >= 0.0 and hpc < ysmax:
            fpch = 1.0 - 0.35 * (hpc / ysmax)**2
        elif hpc < 0.0:
            fpch = 1.0

        hpc_t = hpc + t
        if hpc_t > y:
            fpch_t = 0.0
        elif hpc_t >= ysmax and hpc_t <= y:
            fpch_t = 0.65 * ((y - hpc_t) / (y - ysmax))**2
        elif hpc_t >= 0 and hpc_t < ysmax:
            fpch_t = 1.0 - 0.35 * (hpc_t / ysmax)**2
        elif hpc_t < 0:
            fpch_t = 1.0

        kh = fpch - fpch_t

        return kh

    def _compute_pile_cap_effect_on_pier_factor(self, lpc, lp, apc, ap, hp, y, ysmax, angle_of_attack, kpp):
        """Computes the pile cap effect on pier factor.

        Args:
            lpc (float): The length of the pile cap
            lp (float): The length of the pier stem
            apc (float): The width of the pile cap
            ap (float): The width of the pier stem
            hp (float): The vertical position of the pier
            y (float): The depth upstream of the pier
            ysmax (float): The maximum scour depth
            angle_of_attack (float): The angle of attack
            kpp (float): The pile cap projection factor

        Returns:
            float: The pile cap effect on pier factor
        """
        _, pi = self.get_data('Pi')
        f1 = 0.5 * (lpc - lp)
        f2 = 0.5 * (apc - ap)
        f = np.max([2 / pi * (f2 - f1) * angle_of_attack / 180 * pi + f1, 0]).item()
        f_ratio = f / kpp
        if hp > y:
            ff = 0.3
        elif hp >= ysmax and hp <= y:
            ff = 1.3 * ((y - hp) / (y - ysmax))**2 + 0.3
        elif hp < ysmax:
            ff = 1.6

        if f_ratio >= 0 and f_ratio <= ff:
            kf = 1.0 - f_ratio / ff
        elif f_ratio > ff:
            kf = 0.0
        return f_ratio, kf

    def _compute_second_vertical_position_for_pier_stem_factors(self, h, y, ysmax):
        """Computes the vertical position factor (second version).

        Args:
            h (float): The vertical position of the pier
            y (float): The depth upstream of the pier
            ysmax (float): The maximum scour depth

        Returns:
            float: The vertical position factor
        """
        if h > y:
            kh = 0.0
        elif h >= ysmax and h <= y:
            kh = 0.65 * ((y - h) / (y - ysmax))**3
        elif h >= 0.0 and h < ysmax:
            kh = 1.0 - 0.35 * (h / ysmax)**2
        elif h < 0:
            kh = 1.0
        return kh

    def _compute_lob_factor(self, a, length, angle_of_attack, ysmax, y, hp):
        """Computes the lob factor for the pier or pile cap.

        Args:
            a (float): The width of the pier stem or pile cap
            length (float): The length of the pier stem or pile cap
            angle_of_attack (float): The angle of attack
            ysmax (float): The maximum scour depth
            y (float): The depth upstream of the pier
            hp (float): The vertical position of the pier

        Returns:
            float: The lob factor
        """
        _, pi = self.get_data('Pi')
        alpha = angle_of_attack / 180 * pi
        lob_e = (length * np.cos(alpha).item() + a * np.sin(alpha).item()) / (a * np.cos(alpha).item() + length
                                                                              * np.sin(alpha).item())
        if lob_e <= 1 and lob_e >= 0:
            flob = 0.5 * lob_e**3 - 0.75 * lob_e**2 + 1.25
        elif lob_e > 1:
            flob = 1.0

        ymm = ysmax + 0.2 * y
        if hp > ymm:
            fh = 1.0 / flob
        elif hp >= ysmax and hp <= ymm:
            fh = 1.0 + (1.0 - flob) / flob * (hp - ysmax) / (ymm - ysmax)
        elif hp < ysmax:
            fh = 1.0
        klob = flob * fh

        return klob

    def _compute_pile_group_vertical_position_factors(self, h_pg, y0, kppg):
        """Computes the d* value for the pier.

        Args:
            h_pg (float): The vertical position of the pier
            y0 (float): The depth upstream of the pier
            kppg (float): The shape factor

        Returns:
            khi (float): The shape factor
        """
        if h_pg > y0:
            khi = 1
        elif h_pg > -1 * kppg and h_pg <= y0:
            khi = (h_pg + 3 * kppg) / (y0 + 3 * kppg)
        else:
            khi = 0
        return khi

    def _compute_pier_or_pile_cap_projection_factor(self, ap, lp, shape, angle_of_attack):
        """Computes the d* value for the pier.

        Args:
            ap (float): The width of the pier stem or pile cap
            lp (float): The length of the pier stem or pile cap
            shape (str): The shape of the pier stem or pile cap
            angle_of_attack (float): The angle of attack

        Returns:
            kpp (float): Porjection factor for pier stem or pile cap
        """
        if shape == 'Round Nose':
            kpp = ap + (lp - ap) * np.sin(np.radians(angle_of_attack)).item()
        elif shape in ['Square Nose', 'Sharp Nose']:
            kpp = ap * np.cos(np.radians(angle_of_attack)).item() + lp * np.sin(np.radians(angle_of_attack)).item()
        return kpp

    def _compute_pile_group_projection_factor(self, a, m, n, angle_of_attack):
        """Computes the d* value for the pier.

        Args:
            a (float): The pile width
            m (float): The number of rows in the pile group
            n (float): The number of columns in the pile group
            angle_of_attack (float): The angle of attack

        Returns:
            kppg (float): The shape factor
        """
        kppg = a * (n * np.cos(np.radians(angle_of_attack)).item() + m * np.sin(np.radians(angle_of_attack)).item())
        return kppg

    def _compute_pier_or_pile_cap_shape_factor(self, angle_of_attack, ap, lp, shape):
        """Computes the d* value for the pier.

        Args:
            angle_of_attack (float): The angle of attack
            ap (float): The width of the pier stem or pile cap
            lp (float): The length of the pier stem or pile cap
            shape (str): The shape of the pier stem or pile cap

        Returns:
            ks (float): The shape factor
        """
        lob = lp / ap
        api = angle_of_attack / 180

        if lob < 1:
            lob = 1 / lob
            api = 0.5 - angle_of_attack / 180

        if shape == 'Round Nose':
            if lob > 2:
                lob = 2
                ks = -0.8 * api**2 + 0.8 * (lob - 1) * api + 1
            else:
                ks = 0.8 * (1 - lob) * api**2 + 0.8 * (lob - 1) * api + 1
        elif shape == 'Square Nose':
            if lob > 2:
                lob = 2
                ks = 1.2
            else:
                ks = (9.6 - 4.8 * lob) * api**2 - (4.8 - 2.4 * lob) * api + 1.2
        elif shape == 'Sharp Nose':
            if lob > 2:
                lob = 2
                ks = 1.2
            elif lob <= 2 and lob >= 1:
                ks = -4.8 * api**2 + 2.4 * api + 0.9

        return ks

    def _compute_vertical_position_factors(self, hp, y0):
        """Computes the d* value for the pier.

        Args:
            hp (float): The vertical position of the pier
            y0 (float): The depth upstream of the pier

        Returns:
            khi (float): The shape factor
        """
        if hp > y0:
            khi = 0
        elif hp <= y0 and hp >= 0:
            khi = 1 - hp / y0
        else:
            khi = 1

        return khi

    def _compute_fdot_critical_velocity(self, y1, d50_mm, d50):
        """Computes the critical velocity for the pier.

        Args:
            y1 (float): The depth upstream of the pier
            d50_mm (float): The D50 value, in mm
            d50 (float): The D50 value, in m

        Returns:
            float: The critical velocity
        """
        _, y1_m = self.unit_converter.convert_units('ft', 'm', y1)
        if 0.1 <= d50_mm and d50_mm <= 0.6:
            vc = 0.066 / (1 - 500 * d50) * np.log10(3 * y1_m / d50).item()
        elif 0.6 < d50_mm and d50_mm <= 1.14:
            vc = -0.041 / (1 + 0.44 * np.log10(d50).item()) * np.log10(5 * y1_m / d50).item()
        elif 1.14 < d50_mm and d50_mm <= 5.7:
            vc = (-0.085 + 6.7 * d50**0.5) * np.log10(4.4 * y1_m / d50).item()
        elif d50_mm > 5.7:
            vc = 5.62 * d50**0.5 * np.log10(4 * y1_m / d50).item()

        _, vc_ft = self.unit_converter.convert_units('m', 'ft', vc)
        return vc_ft

    def _compute_ks_kp_dstar(self, shape):
        """Computes the d* value for the pier.

        Args:
            shape (str): The shape of the pier

        Returns:
            ks (float): The shape factor
            kp (float): The projection factor, in feet
            d_star (float): The d* value, in feet
        """
        _, angle_of_attack = self.get_data('Angle of attack (K2)')
        _, ap = self.get_data('Pier width (ap)', prior_keys=['Pier stem parameters'])
        _, lp = self.get_data('Pier length (Lp)', prior_keys=['Pier stem parameters'])
        _, pi = self.get_data('Pi')

        alpha = np.radians(angle_of_attack).item()
        lob = lp / ap
        if shape == 'Round Nose':
            kp = ap + (lp - ap) * np.cos(alpha).item()
            if lob > 2:
                ks = -0.8 * (alpha / pi)**2 + 0.8 * (lob - 1) * (alpha / pi) + 1
            else:
                ks = 0.8 * (1 - lob) * (alpha / pi)**2 + 0.8 * (lob - 1) * (alpha / pi) + 1
        elif shape == 'Square Nose':
            kp = ap * np.cos(alpha).item() + lp * np.sin(alpha).item()
            if lob > 2:
                ks = 1.2
            else:
                ks = (9.6 - 4.8 * lob) * (alpha / pi)**2 - (4.8 - 2.4 * lob) * (alpha / pi) + 1.2
        elif shape == 'Sharp Nose':
            kp = ap * np.cos(alpha).item() + lp * np.sin(alpha).item()
            if lob > 2:
                ks = 1.2
            else:
                if angle_of_attack <= 45:
                    ks = -4.8 * (alpha / pi)**2 + 2.4 * (alpha / pi) + 0.9
                else:
                    ks = (1.2 * lob - 2.4) * (alpha / pi) + 1.8 - 0.3 * lob
        d_star = ks * kp
        return ks, kp, d_star

    def _compute_v1_vc_f1_f2_f3_factors(self, v1, y1, d_star):
        """Computes the f1, f2, and f3 factors for the pier.

        Args:
            v1 (float): The velocity upstream of the pier
            y1 (float): The depth upstream of the pier
            d_star (float): The d* value

        Returns:
            v1p (float): greater of v2 or v3
            vc (float): critical velocity
            f1 (float): The f1 factor
            f2 (float): The f2 factor
            f3 (float): The f3 factor
        """
        _, d50 = self.get_data('Contracted D50')
        _, d50_mm = self.unit_converter.convert_units('ft', 'mm', d50)
        d50_m = d50_mm / 1000
        # _, d50_ft = self.unit_converter.convert_units('mm', 'ft', d50_mm)
        _, g = self.get_data('Gravity')

        vc = self._compute_fdot_critical_velocity(y1, d50_mm, d50_m)
        v3 = 5 * vc
        v2 = 0.6 * (g * y1)**0.5
        v1p = np.max([v3, v2]).item()
        f1 = np.tanh((y1 / d_star)**0.4).item()
        f2 = 1 - 1.2 * (np.log(v1 / vc).item())**2
        f3a = d_star / d50
        f3b = 0.4 * (d_star / d50)**1.2
        f3c = 10.6 * (d_star / d50)**(-0.13)
        f3 = f3a / (f3b + f3c)

        return v1p, vc, f1, f2, f3

    def compute_hec18(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        self.scour_depth = 0.0

        _, length = self.get_data('Pier length (Lp)', prior_keys=['Pier stem parameters'])
        _, a = self.get_data('Pier width (ap)', prior_keys=['Pier stem parameters'])
        _, shape = self.get_data('Pier shape (K1)', prior_keys=['Pier stem parameters'])
        k1 = self._compute_shape_factor(shape)
        self.results['Results']['Pier shape factor (k1)'] = k1
        k2 = self._compute_angle_of_attack_factor(length, a, shape)
        self.results['Results']['Pier angle of attack factor (k2)'] = k2
        self.k3 = self._compute_bed_factor()

        _, a = self.get_data('Pier width (ap)', prior_keys=['Pier stem parameters'])
        _, self.y1 = self.get_data('Depth upstream of pier (y1)')
        _, self.v1 = self.get_data('Velocity upstream of pier (v1)')

        _, g = self.get_data('Gravity')

        fr1 = 0.0
        if self.y1 > 0.0:
            fr1 = self.v1 / (g * self.y1)**0.5
        self.results['Results']['Upstream Froude number (Fr1)'] = fr1

        kw = 1.0
        if self.get_data('Apply wide pier factor (Kw)')[1]:
            _, kw = self._compute_wide_pier_factor(self.y1, self.v1, a)
        self.results['Results']['Wide pier factor (kw)'] = kw

        self.ys = 0.0
        if self.y1 > 0:
            self.ys = (2.0 * k1 * k2 * self.k3 * kw * (a / self.y1)**0.65 * fr1**0.43) * self.y1

        self.results['Results']['Scour depth (ys)'] = self.ys
        self.scour_depth = self.ys
        return True

    def compute_hec18_complex(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        self.scour_depth = 0.0

        khpier = self._compute_kh_pier()
        self.ys_pier = self.ys * khpier

        if 'Case1' not in self.results:
            self.results['Case1'] = {}
        if 'Case2' not in self.results:
            self.results['Case2'] = {}

        self.results['Case1']['Pier stem correction factor (khpier)'] = khpier
        self.results['Case1']['Pier stem scour component (yspier)'] = self.ys_pier
        self.results['Case2']['Pier stem correction factor (khpier)'] = khpier
        self.results['Case2']['Pier stem scour component (yspier)'] = self.ys_pier

        self.results['Results']['Pier stem scour component (yspier)'] = self.ys_pier

        # Determine Scour Case
        scour_case = self.determine_scour_case()
        if scour_case == 'Case 1':
            # CASE 1 Perfom Pile Group Calculations
            self.results['Results']['Scour case'] = 'Case 1'
            self._compute_hec18_case2()  # Compute Case 2 for complete results
            can_compute, warnings = self.input_dict['calc_data']['Pile group parameters'][
                'calculator'].get_can_compute_with_subdict(self.input_dict['calc_data'],
                                                           self.input_dict['calc_data']['Pile group parameters'])
            self.warnings.update(warnings)
            if can_compute:
                self.ys = self._compute_hec18_case1()

        elif scour_case == 'Case 2':
            # CASE 2 Do not perform Pile Group Calculations
            self.results['Results']['Scour case'] = 'Case 2'
            can_compute, warnings = self.input_dict['calc_data']['Pile group parameters'][
                'calculator'].get_can_compute_with_subdict(self.input_dict['calc_data'],
                                                           self.input_dict['calc_data']['Pile group parameters'])
            if can_compute:
                self.ys = self._compute_hec18_case1()  # Compute Case 1 for complete results
            # Do not include warnings because Pile Group is bonus calculations in Case 2
            self.ys = self._compute_hec18_case2()

        self.results['Results']['Scour depth (ys)'] = self.ys
        self.scour_depth = self.ys

        return True

    def determine_scour_case(self):
        """Determines the scour case.

        Returns:
            str: The scour case
        """
        _, h0 = self.get_data('Initial height of the pile cap above bed (h0)', prior_keys=['Pile cap parameters'])
        height_of_pile_cap_after_pier_stem_scour = h0 + self.ys_pier / 2.0  # Hydraulic Toolbox logic
        self.results['Results']['Height of the pile cap after stem scour (h2)'] = \
            height_of_pile_cap_after_pier_stem_scour
        if height_of_pile_cap_after_pier_stem_scour >= 0:
            self.scour_case = 'Case 1'
            self.results['Case1']['Height of the pile cap after stem scour (h2)'] = \
                height_of_pile_cap_after_pier_stem_scour
        else:
            self.scour_case = 'Case 2'
            self.results['Case2']['Height of the pile cap after stem scour (h2)'] = \
                height_of_pile_cap_after_pier_stem_scour

        return self.scour_case

    def _compute_hec18_case1(self):
        """Computes the data possible; stores results in self.

        Returns:
            ys_pc (float): The pile cap scour component case 1
        """
        _, shape = self.get_data('Pile cap shape (K1)', prior_keys=['Pile cap parameters'])
        k1 = self._compute_shape_factor(shape)
        self.results['Case1']['Pile cap shape factor (k1)'] = k1

        _, h0 = self.get_data('Initial height of the pile cap above bed (h0)', prior_keys=['Pile cap parameters'])
        _, lpc = self.get_data('Pile cap length (Lpc)', prior_keys=['Pile cap parameters'])
        _, apc = self.get_data('Pile cap width (apc)', prior_keys=['Pile cap parameters'])
        h2 = h0 + self.ys_pier / 2.0
        y2 = self.y1 + self.ys_pier / 2.0
        v2 = self.v1 * (self.y1 / y2)
        self.results['Case1']['Adjusted depth of flow upstream (y2)'] = y2
        self.results['Case1']['Adjusted velocity of flow upstream (v2)'] = v2
        apc_star = self._compute_equivalent_pile_cap_width(h2, y2, apc)
        self.results['Case1']['Pile cap equivalent width (apc*)'] = apc_star
        max_y2 = 3.5 * apc
        y2 = min(y2, max_y2)
        self.results['Case1']['Max adjusted depth of flow upstream (max y2)'] = max_y2

        fr, kw = self._compute_wide_pier_factor(y2, v2, apc_star)
        self.results['Case1']['Pile cap Froude number (Fr)'] = fr
        self.results['Case1']['Wide pile cap factor (kw)'] = kw

        k2 = self._compute_angle_of_attack_factor(lpc, apc_star, shape)
        self.results['Case1']['Pile cap angle of attack factor (k2)'] = k2

        self.ys_pc = 0.0
        if y2 > 0:
            self.ys_pc = 2.0 * k1 * k2 * self.k3 * kw * (apc_star / y2)**0.65 * fr**0.43 * y2
        self.results['Case1']['Pile cap scour component (yspc)'] = self.ys_pc
        self.results['Results']['Pile cap scour component (yspc)'] = self.ys_pc

        # Pile group Calculations
        _, shape = self.get_data('Pile nose shape (K1)', prior_keys=['Pile group parameters'])
        k1 = self._compute_shape_factor(shape)
        self.results['Case1']['Pile nose shape factor (k1)'] = k1

        apg_star = self._compute_equivalent_pile_group_width()
        self.results['Case1']['Pile group equivalent width (apg*)'] = apg_star

        h3 = h0 + self.ys_pier / 2 + self.ys_pc / 2
        y3 = self.y1 + self.ys_pier / 2 + self.ys_pc / 2
        v3 = self.v1 * self.y1 / y3
        self.results['Case1']['Adjusted depth for pile group computations (y3)'] = y3
        self.results['Case1']['Adjusted velocity for pile group computations (v3)'] = v3
        max_y3 = 3.5 * apg_star
        y3 = min(y3, max_y3)
        self.results['Case1']['Max adjusted depth for pile group computations (max y3)'] = max_y3

        khpg = self._compute_pile_group_height_factor(h3, y3, apg_star)
        self.results['Case1']['Pile group height factor (khpg)'] = khpg

        _, g = self.get_data('Gravity')
        ys_pg = 0.0
        if y3 > 0:
            ys_pg = khpg * (2 * k1 * self.k3 * (apg_star / y3)**0.65 * (v3 / math.sqrt(g * y3))**0.43) * y3
        self.results['Case1']['Pile group scour component (yspg)'] = ys_pg
        self.results['Results']['Pile group scour component (yspg)'] = ys_pg

        self.ys = self.ys_pier + self.ys_pc + ys_pg

        self.results['Case1']['Scour depth (ys)'] = self.ys

        return self.ys

    def _compute_hec18_case2(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        _, shape = self.get_data('Pile cap shape (K1)', prior_keys=['Pile cap parameters'])
        k1 = self._compute_shape_factor(shape)
        self.results['Case2']['Pile cap shape factor (k1)'] = k1
        ks = self._compute_grain_roughness()
        self.results['Case2']['Grain roughness (ks)'] = k1
        _, ks_ft = self.unit_converter.convert_units('mm', 'ft', ks)
        y2 = self.y1 + self.ys_pier / 2.0
        v2 = self.v1 * (self.y1 / y2)
        self.results['Case2']['Adjusted depth of flow upstream (y2)'] = y2
        self.results['Case2']['Adjusted velocity of flow upstream (v2)'] = v2

        yf = self.h1 + self.ys_pier / 2.0
        vf = 0.0
        if ks_ft > 0 and y2 > 0 and yf > 0:
            vf = (math.log(10.93 * yf / ks_ft + 1) / math.log(10.93 * y2 / ks_ft + 1)) * v2
        self.results['Case2']['distance from bed to top of footing (yf)'] = yf
        self.results['Case2']['Average velocity below footing (vf)'] = vf

        _, apc = self.get_data('Pile cap width (apc)', prior_keys=['Pile cap parameters'])
        _, lpc = self.get_data('Pile cap length (Lpc)', prior_keys=['Pile cap parameters'])
        fr, kw = self._compute_wide_pier_factor(yf, vf, apc)
        self.results['Case2']['Wide pile cap factor (kw)'] = kw

        k2 = self._compute_angle_of_attack_factor(lpc, apc, shape)

        self.ys_pc = 2.0 * k1 * k2 * self.k3 * kw * (apc / yf)**0.65 * (fr)**0.43 * yf

        self.results['Case2']['Pile cap scour component (yspc)'] = self.ys_pc
        self.results['Results']['Pile cap scour component (yspc)'] = self.ys_pc

        self.ys = self.ys_pier + self.ys_pc
        self.results['Case2']['Scour depth (ys)'] = self.ys

        return self.ys

    def _compute_grain_roughness(self):
        """Computes the grain roughness.

        Args:

        Returns:
            ks (float): the grain roughness
        """
        # TODO: We do not follow the HEC_18 manual for this calculation
        # Ideally, we woud allow the user to specify the largest sand size or
        # largest gravel size and then calculate the grain roughness based on that value
        # HEC-18 would have us use D84 for sand materials and 3.5 * D84 for gravel or coarser materials
        # Argonne National Laboratory would have us use D50 for sand materials and 2.0 * D50
        # Because D50 is easier, we are using that method

        # Determine if grain size is sand or gravel (or larger)
        # if sand:
        #     sand_gradation =
        #     sand_grad_factor =
        #     ks = sand_grad_factor * sand_grad
        # d50 = self.get_data('Contracted D50')
        ks = 2.0 * self.d50
        return ks

    def _compute_equivalent_pile_group_width(self):
        """Compute the equivalent pile group width."""
        # width of the equivalent pier, m
        _, a = self.get_data('Pile width (a)', prior_keys=['Pile group parameters'])
        _, m = self.get_data('Pile group number of rows (m)', prior_keys=['Pile group parameters'])
        _, n = self.get_data('Pile group number of columns (n)', prior_keys=['Pile group parameters'])
        _, sm = self.get_data('Pile group spacing between rows (sm)', prior_keys=['Pile group parameters'])
        _, sn = self.get_data('Pile group spacing between columns (sn)', prior_keys=['Pile group parameters'])
        _, angle_of_attack = self.get_data('Angle of attack (K2)')
        _, shape = self.get_data('Pile nose shape (K1)', prior_keys=['Pile group parameters'])
        a_prog, _ = self._projected_area(a, sm, sn, m, n, angle_of_attack, shape, 2)
        # TODO: Add prog to results in feet units

        ksp = self._compute_coefficient_of_pile_spacing(a, a_prog, sn)
        if 'FDOT Complex' not in self.results:
            self.results['FDOT Complex'] = {}
        self.results['FDOT Complex']['Coefficient of pile spacing (ksp)'] = ksp

        km = self._compute_coefficient_of_aligned_rows()
        self.results['FDOT Complex']['Coefficient of aligned rows (km)'] = km

        apg_star = a_prog * ksp * km

        return apg_star

    def _compute_pile_group_height_factor(self, h3, y3, aproj_star):
        """Compute the pile group height factor.

        Args:
            h3 (float): The pile group height after pier stem and pile cap scour.
            y3 (float): Adjusted flow depth for the pile group.
            aproj_star (float): The equivalent pile group width.

        Returns:
            float: The pile group height factor.
        """
        y3max = np.min([y3, 3.5 * aproj_star]).item()
        y3 = y3max
        x = h3 / y3
        a1 = 3.08 * x
        a2 = -5.23 * x**2
        a3 = 5.25 * x**3
        a4 = -2.10 * x**4
        khpg = (a1 + a2 + a3 + a4)**(1 / 0.65)
        return khpg

    def _compute_coefficient_of_pile_spacing(self, a, a_proj, s, ):
        """Compute the coefficient of pile spacing.

        Args:
            a (float): Pile width, m.
            a_proj (float): Projected area of pile group, m^2.
            s (float): Pile spacing, m.
        """
        x = a_proj / a
        y = s / a
        ksp = 1 - 4 / 3.0 * (1 - 1 / x) * (1 - y**(-0.6))
        return ksp

    def _compute_coefficient_of_aligned_rows(self):
        """Compute the coefficient of aligned rows."""
        _, m = self.get_data('Pile group number of rows (m)', prior_keys=['Pile group parameters'])
        _, stag = self.get_data('Staggered pile rows', prior_keys=['Pile group parameters'])
        _, angle_of_attack = self.get_data('Angle of attack (K2)')
        _, a = self.get_data('Pile width (a)', prior_keys=['Pile group parameters'])
        _, s = self.get_data('Pile group spacing between columns (sn)', prior_keys=['Pile group parameters'])
        y = s / a
        if angle_of_attack != 0.0 or stag:
            km = 1
        else:
            m1 = np.min([6, m]).item()
            m = m1
            a1 = 0.1 * m
            a2 = -0.0714 * (m - 1) * (2.4 - 1.1 * y + 0.1 * y**2)
            km = 0.9 + a1 + a2
        return km

    def _projected_area(self, a, sm, sn, m, n, angle_of_attack, shape, num_rows):
        """Calculate the projected area of a pile group on a plane normal to flow.

        Args:
            a (float): Pile spacing, m.
            sm (float): Pile group spacing between rows
            sn (float): Pile group spacing between columns
            m (float): number of rows
            n (float): number of columns
            angle_of_attack (float): Angle of attack, degrees.
            shape (str): Shape of the pile group.
            num_rows (int): Number of rows.

        Returns:
            float: The projected area of a pile group on a plane normal to flow.
        """
        _, pi = self.get_data('Pi')
        theta = angle_of_attack / 180 * pi
        if shape == 'Round Nose':
            l_m = (m - 1) * a
            l_n = (n - 1) * a
            aproj1 = l_n * np.cos(theta).item() + l_m * np.sin(theta).item() + a
        else:
            l_m = m * a
            l_n = n * a
            aproj1 = l_n * np.cos(theta).item() + l_m * np.sin(theta).item()

        # l_m = sm*(m-1)+a
        # l_n = sn*(n-1)+a
        # PileN = m+n-1
        pile_ids = []
        _, pi = self.get_data('Pi')
        theta = angle_of_attack / 180 * pi

        # Pile ID
        for i in range(m):
            pile_ids.append([0, i])
        for k in range(num_rows):
            for j in range(n - 1):
                pile_ids.append([j + 1, k])

        # locs = []
        p_ws = []
        for pile_id in pile_ids:
            loc = []
            loc = self._pile_location(pile_id, a, sm, sn, angle_of_attack)  # pile center location
            yc = loc[0]

            p_w = []  # project vector on the normal direction of water flow
            if shape == 'Round Nose':
                p_w = [yc - 0.5 * a, yc + 0.5 * a]
            elif shape == 'Square Nose':
                l_y = 0.5 * a * (np.cos(theta).item() + np.sin(theta).item())
                p_w = [yc - l_y, yc + l_y]
            p_ws.append(p_w)

        pws_sorted = sorted(p_ws, key=lambda x: (x[0], x[1]))

        units = self._unit_function(pws_sorted)
        aporj = 0

        # sum of non-overlapping projected widths of piles
        for unit in units:
            aporj += unit[1] - unit[0]
        return aporj, aproj1

    def _pile_location(self, pile_id, a=0, sm=0, sn=0, angle_of_attack=0):
        """Calculate the pile center locations rotated by angle of attack.

        Args:
            pile_id (list): Pile ID.
            a (float): Pile spacing, m.
            sm (float): Pile group spacing between rows
            sn (float): Pile group spacing between columns
            angle_of_attack (float): Angle of attack, degrees.

        Returns:
            list: The pile center location.
        """
        xid, yid = pile_id
        xc = -1 * xid * sn
        yc = yid * sm
        loc = [xc, yc]
        loc = self._point_rotate(loc, angle_of_attack)
        return loc

    def _point_rotate(self, point, angle_of_attack=0):
        """Rotates the pile group for angle of attack.

        Args:
            point (list of x, y): locations of all pile centers
            angle_of_attack (float): Angle of attack, degrees.

        Return:
            loc (list of x, y): new locations of pile centers.
        """
        _, pi = self.get_data('Pi')
        theta = (angle_of_attack) / 180 * pi
        rotation = np.array([[np.cos(theta), np.sin(theta)], [-np.sin(theta), np.cos(theta)]])
        rotate_p = np.dot(rotation, np.array(point))
        loc = rotate_p.tolist()
        return loc

    # unit the project vector
    def _unit_function(self, pws):
        """This function handles overlapping projected width of piles.

        Args:
            pws (list): pile projected widths

        Return:
            units (list): projected widths without overlapping.
        """
        units = []
        unit = pws[0][:]
        units.append(unit)
        for i in range(len(pws) - 1):
            if unit[1] >= pws[i + 1][0]:
                unitx1 = min(unit[0], pws[i + 1][0])
                unitx2 = max(unit[1], pws[i + 1][1])
                unit = [unitx1, unitx2]
                units.pop()
            else:
                unit = pws[i + 1][:]
            units.append(unit)
        return units

    def _compute_wide_pier_factor(self, y, v, a):
        """Computes the wide pier factor.

        Args:
            fr (float): Froude number
            y (float): depth
            v (float): velocity
            a (float): pier width

        Returns:
            kw (float): wide pier factor
        """
        _, g = self.get_data('Gravity')
        fr = 0.0
        if y > 0.0:
            fr = v / (g * y)**0.5
        _, d50 = self.get_data('Contracted D50')
        _, d50_mm = self.unit_converter.convert_units('ft', 'mm', d50)
        kw = 1.0
        if y < 0.8 * a and fr < 1.0 and a > 50 * d50_mm:
            _, y_si = self.unit_converter.convert_units('ft', 'm', y)  # Convert to SI to match vc equation
            vc = 6.19 * y_si**(1 / 6.0) * (d50_mm / 1000)**(1 / 3)
            _, vc_us = self.unit_converter.convert_units('m', 'ft', vc)  # Convert vc to US units
            self.results['Results']['Critical velocity (vc)'] = vc_us
            if v > 0.0 and vc > 0.0 and v / vc_us < 1:
                kw = 2.58 * (y / a) ** 0.34 * fr ** 0.65
            else:
                kw = 1.0 * (y / a) ** 0.13 * fr ** 0.25

        return fr, kw

    def _compute_equivalent_pile_cap_width(self, h2, y2, apc):
        """Compute the equivalent pile cap width."""
        # width of the equivalent pier, m
        if y2 > 3.5 * apc:
            y2 = 3.5 * apc
        _, t = self.get_data('Pile cap thickness (T)', prior_keys=['Pile cap parameters'])

        x = t / y2
        y = h2 / y2

        a1 = 0.0
        if x > 0.0:
            a1 = -2.705 + 0.51 * np.log(x) - 2.783 * y**3 + 1.751 / np.exp(y)
        apc_star = apc * np.exp(a1).item()

        return apc_star

    def _compute_kh_pier(self):
        """Calculate the correction factor for pier stem.

        Args:

        Returns:
            float: The correction factor for pier stem.
        """
        _, h0 = self.get_data('Initial height of the pile cap above bed (h0)', prior_keys=['Pile cap parameters'])
        _, t = self.get_data('Pile cap thickness (T)', prior_keys=['Pile cap parameters'])
        self.h1 = h0 + t

        _, ap = self.get_data('Pier width (ap)', prior_keys=['Pile cap parameters'])
        _, f = self.get_data('Distance between front edge (f)', prior_keys=['Pile cap parameters'])

        x = f / ap
        y = self.h1 / ap
        a1 = (0.4075 - 0.0669 * x)
        a2 = -1 * (0.4271 - 0.0778 * x) * y
        a3 = (0.1615 - 0.0455 * x) * y**2
        a4 = -1 * (0.0269 - 0.012 * x) * y**3
        return a1 + a2 + a3 + a4

    def _compute_shape_factor(self, shape):
        """Compute the K1 value (shape factor)."""
        k1 = 1.0
        if shape == 'Square Nose':
            k1 = 1.1
        if shape in ['Round Nose', 'Circular Cylinder', 'Group of Cylinders']:
            k1 = 1.0
        if shape == 'Sharp Nose':
            k1 = 0.9
        # HEC-18, page 7.5
        # The correction factor K1 for pier nose shape should be determined using Table 7.1 for angles
        # of attack up to 5 degrees. For greater angles, K2 dominates and K1 should be considered
        # as 1.0."
        _, angle_of_attack = self.get_data('Angle of attack (K2)')
        if angle_of_attack >= 5.0:
            k1 = 1.0

        return k1

    def _compute_bed_factor(self):
        """Compute the K3 value (bed factor)."""
        bed = self.get_data('Bed condition (K3)')[1].lower()
        k3 = 1.1
        if bed in ['clear-water scour', 'plane bed and antidune flow', 'small dunes']:
            k3 = 1.1
        elif bed in ['medium dunes']:
            k3 = 1.15
        elif bed in ['large dunes']:
            k3 = 1.3

        self.results['Results']['Bed form factor (k3)'] = k3

        return k3

    def _compute_angle_of_attack_factor(self, length, a, shape):
        """Computes the factor for the angle of attack.

        Args:
            length (float): The length of the pier
            a (float): The width of the pier
            shape (str): The shape of the pier

        Returns:
            float: The angle of attack factor
        """
        length_over_width = 0.0
        if a > 0.0:
            length_over_width = length / a
        if length_over_width > 12.0:
            length_over_width = 12.0

        self.results['Results']['Pier length over width (L/a)'] = length_over_width

        _, angle_of_attack = self.get_data('Angle of attack (K2)')
        if shape == 'Circular Cylinder':
            angle_of_attack = 0.0  # No angle of attack on a circular pier
        _, angle_in_radians = self.unit_converter.convert_units(
            '°', 'radians', angle_of_attack)

        # Equation 7.4 on pdf page 170 of HEC-18, April 2012
        k2 = (math.cos(angle_in_radians) + length_over_width * math.sin(angle_in_radians))**0.65
        self.results['Results']['Angle of attack factor (k2)'] = k2

        return k2
