"""EmbedmentData Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.variable_group import VariableGroup
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.gradations.gradation_layer_data import GradationLayerData
from xms.HydraulicToolboxCalc.hydraulics.channel_geometry.embedment_calc import EmbedmentCalc


class EmbedmentData(VariableGroup):
    """Provides a class that will define the site data of a culvert barrel."""

    def __init__(self, rise, open_shape, app_data=None, model_name=None, project_uuid=None):
        """Initializes theEmbedment Data class.

        Args:
            rise (float): rise of the channel
            open_shape (bool): whether the channel is open
            app_data (AppData): The application data.
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'Embedment Data'
        self.type = 'EmbedmentData'

        self.calculator = EmbedmentCalc(rise, open_shape)

        # Input
        self.rise = rise
        self.open_shape = open_shape

        self.embedment_method_list = []
        self.build_embedment_method_list()

        # self.input['Name'] = Variable('Name', 'string', self.name)
        self.input['Embedment entry'] = Variable('Embedment entry', "list", 0,
                                                 self.embedment_method_list)

        self.input['Embedment depth'] = Variable('Embedment depth',
                                                 'float',
                                                 0, [],
                                                 precision=2,
                                                 unit_type=['length'],
                                                 native_unit='ft',
                                                 us_units=self.us_mid_length,
                                                 si_units=self.si_mid_length)

        self.input['Embedment percent'] = Variable(
            'embedment percentage of rise',
            'float',
            0.0, [],
            limits=[0.0, 90.0],
            precision=1,
            unit_type=['%'],
            native_unit='%',
            us_units=[['%']],
            si_units=[['%']])

        self.input['Low flow channel depth'] = Variable(
            'Low flow channel depth',
            'float',
            0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.input['Low flow channel side slope'] = Variable(
            'Low flow channel side slope',
            'float',
            1.0, [],
            precision=2,
            unit_type=['slope'],
            native_unit='H:1V',
            us_units=[['H:1V']],
            si_units=[['H:1V']])

        self.input['Specify gradation'] = Variable('Specify gradation', "bool",
                                                   0, False)
        self.input['Gradation layer'] = Variable(
            'Gradation layer', "class", GradationLayerData([0.5, 0.85], app_data=app_data, model_name=model_name,
                                                           project_uuid=project_uuid))

        self.compute_prep_functions.extend([])
        self.intermediate_to_copy.extend(['rise', 'open_shape',])
        self.compute_finalize_functions = [self.transfer_inputs]

        self.warnings = []
        self.results = {}
        # self.results['result'] = Variable('result', 'float_list', 0.0, [], precision=precision,
        #                                   unit_type=unit_type, native_unit=native_unit,
        #                                   us_units=us_units, si_units=si_units)

    def transfer_inputs(self):
        """Transfer inputs from the calculator to the input variable."""
        self.input['Embedment depth'].set_val(self.calculator.input_dict['calc_data']['Embedment depth'])
        self.input['Embedment percent'].set_val(self.calculator.input_dict['calc_data']['Embedment percent'])

    def build_embedment_method_list(self):
        """Build the list of embedment methods."""
        # build list for the ways to specify manning's n value
        self.calculator.open_shape = self.open_shape

        self.calculator.embedment_method_list = self.embedment_method_list
        self.calculator.build_embedment_method_list()
        self.embedment_method_list = self.calculator.embedment_method_list
        if 'Embedment entry' in self.input:
            self.input['Embedment entry'].set_list(self.embedment_method_list)

    def get_val(self):
        """Computes and returns the results.

        Returns:
            self.results['result'] (variable): result of the computations.
        """
        self.compute_data()
        embed = self.calculator.input_dict['calc_data']['Embedment depth']
        embed_percent = self.calculator.input_dict['calc_data']['Embedment percent']
        self.input['Embedment depth'].set_val(embed)
        self.input['Embedment percent'].set_val(embed_percent)
        return embed

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Args:
            unknown (str): unknown variable that we are computing

        Returns
            input_vars (list of variables): input group of variables
        """
        input_vars = {}

        _, null_data = self.get_setting('Null data')

        # update variable lists
        self.build_embedment_method_list()

        input_vars['Embedment entry'] = self.input['Embedment entry']
        # Gather the input options
        if self.input['Embedment entry'].get_val() in ['embed by percentage of rise']:
            input_vars['Embedment percent'] = self.input['Embedment percent']
        elif self.input['Embedment entry'].get_val() in ['embed by depth', 'embed evenly across channel']:
            input_vars['Embedment depth'] = self.input['Embedment depth']

        if self.get_embedment_depth() > 0.0 and not self.open_shape:
            input_vars['Low flow channel depth'] = self.input[
                'Low flow channel depth']
            embed_depth = self.input['Embedment depth'].get_val()
            input_vars['Low flow channel depth'].limits = (0.0, embed_depth)

            if self.input['Low flow channel depth'].get_val() > 0.0:
                input_vars['Low flow channel side slope'] = self.input['Low flow channel side slope']

        if self.get_embedment_depth() > 0.0:
            input_vars['Specify gradation'] = self.input['Specify gradation']
            if self.input['Specify gradation'].get_val():
                input_vars['Gradation layer'] = self.input['Gradation layer']

        return input_vars

    def get_embedment_depth(self):
        """Get the embedment depth.

        Returns:
            float: embedment depth
        """
        self.compute_data()
        return self.input['Embedment depth'].get_val()

    # def set_no_embedment_depth(self):
    #     """Set the embedment depth to zero."""
    #     self.input['Embedment depth'].set_val(0.0)
    #     self.input['Embedment percent'].set_val(0.0)
