"""CalcData for performing Culvert Barrel operations."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator import Calculator

# 4. Local modules


class CulvertBarrelCalc(Calculator):
    """A class that defines a channel and performs Culvert barrel computations."""

    def set_name(self, index_of_culvert):
        """Set the name of the culvert barrel.

        Args:
            index_of_culvert (int): used to initialize the name to make it unique.
        """
        if index_of_culvert is None:
            self.name = 'Culvert barrel calc'
        elif index_of_culvert >= 0:
            self.name = f'Culvert {index_of_culvert + 1}'

    def update_name(self):
        """Update the name from the input variable."""
        self.name = self.input['Name'].get_val()

    def update_input_name(self):
        """Update the name from the input variable."""
        self.input['Name'].set_val(self.name)

    def assign_results(self):
        """Assigns the results from a computational run to the appropriate locations."""
        # Assign the results
        _, gamma = self.get_data('Unit weight of water (γw)')

        multiple = False
        if multiple:
            pass
        else:
            self.boundary_shear_stress = []
            self.max_shear_stress = []

            for energy_slope, hyd_rad, depth in zip(self.energy_slope,
                                                    self.hydraulic_radius,
                                                    self.y):
                self.boundary_shear_stress.append(gamma * hyd_rad * energy_slope)
                self.max_shear_stress.append(gamma * depth * energy_slope)

            self.results['Station'].set_val(self.wse_stations)
            self.results['WSE'].set_val(self.wse_elevations)
            self.results['Distance from inlet'].set_val(self.x)
            self.results['Depth'].set_val(self.y)
            self.results['Flow area'].set_val(self.flow_area)
            self.results['Wetted perimeter'].set_val(self.wetted_perimeter)
            self.results['Top width'].set_val(self.top_width)
            self.results['Hydraulic radius'].set_val(self.hydraulic_radius)
            self.results['Manning n'].set_val(self.manning_n)
            self.results['Velocity'].set_val(self.velocity)
            self.results['Energy'].set_val(self.energy)
            self.results['Energy loss'].set_val(self.energy_loss)
            self.results['Energy slope'].set_val(self.energy_slope)
            self.results['Boundary shear stress'].set_val(
                self.boundary_shear_stress)
            self.results['Max shear stress'].set_val(self.max_shear_stress)

            if self.input_dict['calc_data']['Display channel slope as flat']:
                self.plot_x = self.x
                self.plot_y = self.y
            else:
                self.plot_x = self.wse_stations
                self.plot_y = self.wse_elevations

            if self.hyd_jump_swept_out:
                self.warnings['Hydraulic jump'] = (
                    'An hydraulic jump forms near the culvert outlet but may be swept into the tailwater channel'
                )

    def _get_can_compute(self, unknown=None):
        """Determines if there is enough data to make a computation and if there isn't, add a warning for each reason.

        Args:
            unknown (string): the variable that is unknown and is being calculated

        Returns:
            bool: True if can compute
        """
        result = True

        if unknown is None:
            if self.unknown is None:
                self.unknown = 'Head'
            unknown = self.unknown
        else:
            self.unknown = unknown

        found = False
        flows = self.input_dict['calc_data']['Flows']
        if hasattr(flows, '__len__') and len(flows) < 1:
            self.warnings['Flows'] = "Please enter a flow"
            result = False
        else:
            for flow in flows:
                if flow > 0.0:
                    found = True
                    break
            if not found:
                self.warnings['Flows'] = "Please enter a positive, non-zero flow"
                result = False

        if not self.input_dict['calc_data']['Geometry']['calculator']._get_can_compute(unknown):
            self.warnings.update(self.input_dict['calc_data']['Geometry']['calculator'].warnings)
            result = False

        if not self.input_dict['calc_data']['Site data']['calculator']._get_can_compute():
            slope = 0.0
            self.warnings.update(self.input_dict['calc_data']['Site data']['calculator'].warnings)
            result = False
        else:
            self.input_dict['calc_data']['Site data']['calculator'].compute_data()
            # TODO: Update this to get the slope of the first barrel, so we can determine the inlet control
            slope = self.input_dict['calc_data']['Site data']['calculator'].get_slope_of_final_barrel()

        boundary_result, _ = self.input_dict['calc_data']['Downstream water depth']['calculator'].get_can_compute()
        if not boundary_result:
            return False

        # Initialize inlet calc
        self.manning_n_calc.input_dict['calc_data']['Slope'] = slope

        self.manning_n_calc.input_dict['calc_data']['Calculate'] = unknown
        self.manning_n_calc.input_dict['calc_data']['Flows'] = self.input_dict['calc_data']['Flows']
        self.manning_n_calc.input_dict['calc_data']['Composite n'] = \
            self.input_dict['calc_data']['Composite n']
        self.manning_n_calc.input_dict['calc_data']['Geometry'] = self.input_dict['calc_data']['Geometry']
        inlet_calc = self.input_dict['calc_data']['Inlet data']['calculator']
        inlet_calc.initialize_inlet_data_for_culvert_with_manning_n_calc(self.manning_n_calc)

        if not inlet_calc._get_can_compute():
            self.warnings.update(inlet_calc.warnings)
            result = False

        _, horizontal_slope = self.get_data('Horizontal tolerance')
        final_slope = self.input_dict['calc_data']['Site data']['calculator'].get_slope_of_final_barrel()
        if final_slope < horizontal_slope:
            slope_type = 'adverse slope'
            if abs(final_slope) < horizontal_slope:
                slope_type = 'horizontal slope'
            if self.input_dict['calc_data']['Downstream water depth']['calculator'].input_dict['calc_data'][
                    'Select'] == 'normal depth':
                self.warnings['Downstream boundary'] = (
                    'The downstream boundary condition is set to normal depth, but the slope is '
                    f'{slope_type}')
                return False

        return result

    def clear_results(self):
        """Clears the results and those of subclasses to prepare for computation."""
        self.input_dict['calc_data']['Geometry']['calculator'].clear_results()

        # Intermediate Results
        # self.high_elev = 0.0
        # self.low_elev = 0.0

    def _compute_data(self, tw_elevation=None, tw_velocity=None):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        self.reset_note_flags()

        culvert_type = self.input_dict['calc_data']['Site data']['calculator'].input_dict['calc_data']['Culvert type']

        self.min_embedment_depth = self.input_dict['calc_data']['Geometry']['calculator'].min_embedment_depth
        self.input_dict['calc_data']['Site data']['calculator'].input_dict['calc_data'][
            'calculator'].embedment_depth = self.min_embedment_depth

        # if we have broken-back culvert, do not allow embedment
        if culvert_type != 'straight':
            self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data'][
                'Embedment'].set_no_embedment_depth()

        # Determine the number of identical (parallel barrels)
        num_identical = self.input_dict['calc_data']['Site data']['calculator'].input_dict['calc_data'][
            'Identical barrels']
        flows = self.input_dict['calc_data']['Flows']
        flow_per_barrel = [flow / num_identical for flow in flows]

        # Setup the GVF
        num_of_gvf_segments = 1
        if culvert_type == 'single broken-back':
            num_of_gvf_segments = 2
        elif culvert_type == 'double broken-back':
            num_of_gvf_segments = 3

        self.rise = self.input_dict['calc_data']['Geometry']['calculator'].rise_to_crown

        # used to set the basic segments with an entrance loss coefficient
        self.input_dict['calc_data']['Inlet data']['calculator'].set_ke()
        ke = self.input_dict['calc_data']['Inlet data']['calculator'].input_dict['calc_data']['KE']

        # Setup a culvert barrel that we will copy to each segment
        self.input_dict['calc_data']['Geometry']['calculator'].compute_data()
        basic_segment = copy.copy(self.gvf_base_calc)
        basic_segment.m_ke = ke
        basic_segment.input_dict['calc_data']['Flows'] = copy.deepcopy(self.input_dict['calc_data']['Flows'])
        basic_segment.input_dict['calc_data']['Flows'] = flow_per_barrel
        basic_segment.input_dict['calc_data']['Composite n'] = self.input_dict['calc_data']['Composite n']
        basic_segment.manning_n_calc = self.manning_n_calc  # Be sure this is a mann n calc, not complex n calc!
        basic_segment.input_dict['calc_data']['Downstream water depth']['calculator'].input_dict['calc_data'][
            'Select'] = 'normal depth'

        # Deepcopy the barrel to each segment
        self.barrel_segments = []
        for _ in range(num_of_gvf_segments):
            self.barrel_segments.append(copy.copy(basic_segment))

        # Setup the data for each segment
        self.critical_depths = [0.0] * num_of_gvf_segments
        self.critical_flowareas = [0.0] * num_of_gvf_segments
        self.critical_velocities = [0.0] * num_of_gvf_segments
        self.normal_depths = [0.0] * num_of_gvf_segments
        self.normal_velocities = [0.0] * num_of_gvf_segments
        self.wse_stations = [0.0] * num_of_gvf_segments
        self.wse_elevations = [0.0] * num_of_gvf_segments

        self.inlet_invert_elevation = self.input_dict['calc_data']['Site data']['calculator'].input_dict['calc_data'][
            'Inlet invert elevation']
        self.outlet_invert_elevation = self.input_dict['calc_data']['Site data']['calculator'].input_dict['calc_data'][
            'Outlet invert elevation']

        self.barrel_segments[0].input_dict['calc_data']['Upstream invert station'] = self.input_dict['calc_data'][
            'Site data']['calculator'].input_dict['calc_data']['Inlet invert station']
        self.barrel_segments[0].input_dict['calc_data']['Upstream invert elevation'] = self.inlet_invert_elevation
        if culvert_type == 'single broken-back':
            self.barrel_segments[0].input_dict['calc_data']['Downstream invert station'] = self.input_dict[
                'calc_data']['Site data']['calculator'].input_dict['calc_data']['Break invert station']
            self.barrel_segments[0].input_dict['calc_data']['Downstream invert elevation'] = self.input_dict[
                'calc_data']['Site data']['calculator'].input_dict['calc_data']['Break invert elevation']
            self.barrel_segments[1].input_dict['calc_data']['Upstream invert station'] = self.input_dict[
                'calc_data']['Site data']['calculator'].input_dict['calc_data']['Break invert station']
            self.barrel_segments[1].input_dict['calc_data']['Upstream invert elevation'] = self.input_dict[
                'calc_data']['Site data']['calculator'].input_dict['calc_data']['Break invert elevation']
        elif culvert_type == 'double broken-back':
            self.barrel_segments[0].input_dict['calc_data']['Downstream invert station'] = self.input_dict[
                'calc_data']['Site data']['calculator'].input_dict['calc_data']['Upper break invert station']
            self.barrel_segments[0].input_dict['calc_data']['Downstream invert elevation'] = self.input_dict[
                'calc_data']['Site data']['calculator'].input_dict['calc_data']['Upper break invert elevation']
            self.barrel_segments[1].input_dict['calc_data']['Upstream invert station'] = self.input_dict[
                'calc_data']['Site data']['calculator'].input_dict['calc_data']['Upper break invert station']
            self.barrel_segments[1].input_dict['calc_data']['Upstream invert elevation'] = self.input_dict[
                'calc_data']['Site data']['calculator'].input_dict['calc_data']['Upper break invert elevation']

            self.barrel_segments[1].input_dict['calc_data']['Downstream invert station'] = self.input_dict[
                'calc_data']['Site data']['calculator'].input_dict['calc_data']['Lower break invert station']
            self.barrel_segments[1].input_dict['calc_data']['Downstream invert elevation'] = self.input_dict[
                'calc_data']['Site data']['calculator'].input_dict['calc_data']['Lower break invert elevation']
            self.barrel_segments[2].input_dict['calc_data']['Upstream invert station'] = self.input_dict[
                'calc_data']['Site data']['calculator'].input_dict['calc_data']['Lower break invert station']
            self.barrel_segments[2].input_dict['calc_data']['Upstream invert elevation'] = self.input_dict[
                'calc_data']['Site data']['calculator'].input_dict['calc_data']['Lower break invert elevation']

        self.barrel_segments[-1].input_dict['calc_data']['Downstream invert station'] = self.input_dict[
            'calc_data']['Site data']['calculator'].input_dict['calc_data']['Outlet invert station']
        self.barrel_segments[-1].input_dict['calc_data']['Downstream invert elevation'] = self.input_dict[
            'calc_data']['Site data']['calculator'].input_dict['calc_data']['Outlet invert elevation']
        self.barrel_segments[-1].input_dict['calc_data']['Downstream water depth'] = copy.copy(
            self.input_dict['calc_data']['Downstream water depth'])

        # TODO Add multiple flows!
        flow_index = 0

        downstream_velocities = [0.0 for x in range(len(flow_per_barrel))]

        # Initialize the inlet control calculator
        geom = self.input_dict['calc_data']['Geometry']['calculator']
        shape = geom.input_dict['calc_data']['Shape']
        self.span = geom.input_dict['calc_data']['Span']
        if shape == 'circle':
            self.span = self.rise
        geom.input_dict['calc_data']['Depths'] = [self.rise]
        geom.compute_data()
        self.full_flow_area = geom.full_flow_area
        self.slope = self.input_dict['calc_data']['Site data']['calculator'].get_slope_of_final_barrel()
        self.input_dict['calc_data']['Inlet data']['calculator'].initialize_inlet_data_for_culvert(
            shape, self.span, self.rise, self.full_flow_area, self.slope, self.barrel_segments[0].manning_n_calc)

        for flow in flow_per_barrel:
            # Determine slopes and calculation order
            # Put the steep slopes first from upstream to downstream, put the mild & less last
            self.flow = flow
            calculation_order = []
            mild_flat_or_horizontal_slopes = []
            lower_depth = []
            lower_depth_mild_or_less = []
            for index in range(num_of_gvf_segments):
                self.barrel_segments[index].input_dict['calc_data']['Geometry'] = self.input_dict['calc_data'][
                    'Geometry']
                self.barrel_segments[index].determine_critical_and_normal_depths_and_slope(index)
                self.normal_depths[index] = self.barrel_segments[index].normal_depth
                self.critical_depths[index] = self.barrel_segments[index].critical_depth
                self.critical_flowareas[index] = self.barrel_segments[index].critical_flowarea
                if self.barrel_segments[index].slope_type == 'steep':
                    calculation_order.append(index)
                    lower_depth.append(min(self.barrel_segments[index].normal_depth,
                                           self.barrel_segments[index].critical_depth))
                else:
                    mild_flat_or_horizontal_slopes.append(index)
                    lower_depth_mild_or_less.append(min(self.barrel_segments[index].normal_depth,
                                                        self.barrel_segments[index].critical_depth))
            calculation_order.extend(mild_flat_or_horizontal_slopes)
            lower_depth.extend(lower_depth_mild_or_less)

            # Determine the inlet depths
            self.inlet_water_velocity = 0.0  # Assume dead pool upstream of culvert barrel
            self.determine_inlet_control_depth(flow_per_barrel, self.critical_depths[-1], self.critical_flowareas[-1])

            self.tw_depth, self.tw_velocity = self.input_dict['calc_data']['Downstream water depth']['calculator']. \
                get_starting_water_depth_and_velocity(self.normal_depths[-1], self.normal_velocities[-1],
                                                      self.critical_depths[-1], self.critical_velocities[-1], self.flow)

            # Determine the Outlet Control Depth
            #   Determine the upstream water depth
            change = sys.float_info.max
            _, zero_tol = self.get_data('Zero tolerance')
            _, num_iterations = self.get_data('Max number of iterations')
            count = 0
            jump_swept_count = 0
            while change > zero_tol and count < num_iterations:
                change = 0.0
                upstream_depths = [0.0]
                upstream_velocities = [0.0]
                downstream_depths = [0.0]
                for calc_index in range(len(calculation_order)):
                    index = calculation_order[calc_index]
                    # Update the upstream water depth
                    up_depth_list, up_vel_list = self.barrel_segments[index].input_dict['calc_data'][
                        'Upstream water depth']['calculator'].get_specified_depths_and_velocities()
                    if len(up_depth_list):
                        upstream_depths = up_depth_list
                        upstream_velocities = up_vel_list
                    if index > 0:
                        upstream_depths = [self.barrel_segments[index - 1].downstream_depth]
                        # if upstream_depths[0] < lower_depth[index]:
                        #     upstream_depths = [lower_depth[index]]
                        upstream_velocities = [self.barrel_segments[index - 1].downstream_velocity]
                    else:
                        # If we are computing the upstream segment, set the inlet control depth as the upstream boundary
                        upstream_depths[flow_index] = self.inlet_water_depth
                        upstream_velocities[flow_index] = self.inlet_water_velocity
                    self.barrel_segments[index].input_dict['calc_data']['Upstream water depth']['calculator']\
                        .set_specified_depths_and_velocities(upstream_depths, upstream_velocities)

                    # Update the downstream water depth
                    down_depth_list, down_vel_list = self.barrel_segments[index].input_dict['calc_data'][
                        'Downstream water depth']['calculator'].get_specified_depths_and_velocities()
                    if len(down_depth_list):
                        downstream_depths = down_depth_list
                        downstream_velocities = down_vel_list
                    if index < len(self.barrel_segments) - 1:
                        if count > 0:
                            up_y = self.barrel_segments[index + 1].upstream_depth
                            up_p = self.barrel_segments[index + 1].upstream_pressure_head
                            downstream_depths = [up_y + up_p]
                        # Do not try a downstream boundary condition less than normal depth
                        if downstream_depths[index] < self.barrel_segments[index].normal_depth:
                            downstream_depths[index] = self.barrel_segments[index].normal_depth
                        downstream_velocities = [self.barrel_segments[index + 1].upstream_velocity]
                    else:
                        downstream_depths[flow_index] = self.tw_depth
                        downstream_velocities[flow_index] = self.tw_velocity
                    self.barrel_segments[index].input_dict['calc_data']['Downstream water depth']['calculator'].\
                        set_specified_depths_and_velocities(downstream_depths, downstream_velocities)

                    # If we have run a few calc rounds, the segment upstream has a jump swept out, enforce that depth
                    if jump_swept_count > 20 and index > 0:
                        jump_swept_out = self.barrel_segments[
                            index - 1].hyd_jump_swept_out
                        if self.barrel_segments[index - 1].solution == 'downstream direction' and \
                                self.barrel_segments[index - 1].downstream_depth < \
                                self.barrel_segments[index].upstream_depth:
                            jump_swept_out = True
                        if jump_swept_out:
                            self.barrel_segments[index].enforce_upstream_depth = True
                            jump_swept_count = 0
                    # Compute and find the largest change
                    self.barrel_segments[index].inlet_control_depths = self.inlet_control_depth
                    self.barrel_segments[index].inlet_control_depth = self.inlet_control_depth[0]
                    self.barrel_segments[index].compute_data()

                for index in range(len(self.barrel_segments) - 1):
                    difference = self.barrel_segments[index].downstream_depth - \
                        self.barrel_segments[index + 1].upstream_depth
                    change = max(abs(change), abs(difference))
                count += 1
                jump_swept_count += 1

            self.handle_hydraulic_jumps()

            # Transfer outlet data
            self.outlet_depth = self.barrel_segments[-1].downstream_depth
            self.outlet_velocity = self.barrel_segments[-1].downstream_velocity

            self.complete_the_outlet_control_depth()

            # Determine HW by the greater of ICD and OCD
            self.headwaterDepth = max(self.inlet_control_depth[0], self.outlet_control_depth)
            self.hw = self.headwaterDepth + self.barrel_segments[0].input_dict['calc_data'][
                'Upstream invert elevation']

        return True

    def complete_the_outlet_control_depth(self):
        """Complete the outlet control depth for the culvert.
        """
        # Update the OCD
        upstream_surface_depth = self.barrel_segments[0].upstream_depth + \
            self.barrel_segments[0].upstream_pressure_head

        if upstream_surface_depth < self.barrel_segments[0].critical_depth + self.barrel_segments[0].increment:
            tw_velocity_for_ocd = 0.0
            _, usu_exitloss = self.get_data('USU Exit loss method')
            if usu_exitloss:
                tw_velocity_for_ocd = self.tw_velocity
            exit_loss_coefficient = self.input_dict['calc_data']['Inlet data']['calculator'].input_dict['calc_data'][
                'KE']
            velocity = self.barrel_segments[0].upstream_velocity
            _, g = self.get_data('Gravity')
            velocity_head = velocity**2 / (2 * g)  # Loss at entrance, with entrance loss coefficient
            velocity_difference = velocity - tw_velocity_for_ocd
            if velocity_difference < 0.0:
                velocity_difference = 0.0
            exit_loss = velocity_difference**2 / (2 * g)  # Loss for velocity discharged at TW
            self.outlet_control_depth = self.barrel_segments[0].critical_depth + \
                velocity_head * exit_loss_coefficient + exit_loss
        else:
            self.outlet_control_depth = upstream_surface_depth

    def handle_hydraulic_jumps(self):
        """Handle hydraulic jumps in the culvert."""
        # Handle hydraulic jumps!
        # Check for hydraulic jumps at break in slopes
        for index in range(len(self.barrel_segments) - 1):
            if self.barrel_segments[index].hyd_jump_swept_out:
                # Stitch in the Hydraulic jump!
                # TODO Need to recompute the jump length!
                station_between = self.barrel_segments[index].wse_stations[-1]
                hyd_jump_station = self.barrel_segments[index].hyd_jump_station
                hyd_jump_depth = self.barrel_segments[index].hyd_jump_depth
                hyd_jump_length = self.barrel_segments[index].hyd_jump_length
                hyd_jump_sequent_depth = self.barrel_segments[index].hyd_jump_sequent_depth
                x_up, y_up, x_down, y_down = self.barrel_segments[index].combine_wses_with_hydraulic_jump(
                    True, hyd_jump_station, hyd_jump_depth, hyd_jump_length, hyd_jump_sequent_depth,
                    self.barrel_segments[index].results, self.barrel_segments[index].intermediate_results,
                    self.barrel_segments[index + 1].results, station_between)
                self.barrel_segments[index].wse_stations = x_up
                self.barrel_segments[index].wse_stations = y_up
                self.barrel_segments[index + 1].wse_stations = x_down
                self.barrel_segments[index + 1].wse_stations = y_down
                self.barrel_segments[index].hyd_jump_exists = True
        for index in range(len(self.barrel_segments)):
            if self.barrel_segments[index].hyd_jump_exists:
                self.hyd_jump_exists = True
                self.hyd_jump_station_absolute.append(self.barrel_segments[index].hyd_jump_station_absolute)
                self.hyd_jump_final_station_absolute.append(self.barrel_segments[index].hyd_jump_final_station_absolute)
                self.hyd_jump_elevation.append(self.barrel_segments[index].hyd_jump_elevation)
                self.hyd_jump_sequent_elevation.append(self.barrel_segments[index].hyd_jump_sequent_elevation)
                self.hyd_jump_depth.append(self.barrel_segments[index].hyd_jump_depth)
                self.hyd_jump_sequent_depth.append(self.barrel_segments[index].hyd_jump_sequent_depth)
        # Finalize results and gather plot data
        for index in range(len(self.barrel_segments)):
            self.critical_depths[index] = self.barrel_segments[index].critical_depth
            self.critical_flowareas[index] = self.barrel_segments[index].critical_flowarea
            self.critical_velocities[index] = self.barrel_segments[index].critical_velocity
            self.normal_depths[index] = self.barrel_segments[index].normal_depth
            self.normal_velocities[index] = self.barrel_segments[index].normal_velocity
            self.wse_stations[index] = self.barrel_segments[index].wse_stations
            self.wse_elevations[index] = self.barrel_segments[index].wse_elevations
        # Transfer jump swept out from last barrel
        self.hyd_jump_swept_out = self.barrel_segments[-1].hyd_jump_swept_out

    def determine_inlet_control_depth(self, flow_per_barrel, critical_depth=None, critical_flowarea=None):
        """Determine the inlet control depth for the culvert.

        Args:
            flow_per_barrel (list): The flow per barrel.
            critical_depth (float): The critical depth.
            critical_flowarea (float): The critical flow area.
        """
        # Determine the Inlet Control Depth
        self.input_dict['calc_data']['Inlet data']['calculator'].mannings_n_calc = self.manning_n_calc
        self.input_dict['calc_data']['Inlet data']['calculator'].input_dict['calc_data']['Flows'] = flow_per_barrel
        self.inlet_control_depth = self.input_dict['calc_data']['Inlet data']['calculator']._compute_data(
            critical_depth, critical_flowarea)
        self.inlet_water_depth = max(critical_depth, self.inlet_control_depth[0])

    def compute_vena_contracta_depth_and_length(self, shape, ):
        """Compute the vena contracta depth and length for the culvert.

        Args:
            shape (str): The shape of the culvert.

        Returns:
            tuple: The vena contracta length and depth.
        """
        coefficient = 0.7
        slope_correction_term = 0.5
        shape = self.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Shape']
        if shape in ['box']:  # Low profile box, conspan, south dakota box
            coefficient = 0.8
        inlet_water_depth = self.rise

        if 0.0 < self.inlet_control_depth < self.rise:
            inlet_water_depth = self.inlet_control_depth

        vena_contracta_length = slope_correction_term * self.rise
        vena_contracta_depth = coefficient * inlet_water_depth

        return vena_contracta_length, vena_contracta_depth

    def reset_note_flags(self):
        """Reset the flags for the notes."""
        self.show_oscillitary_flow_note = False
        self.show_depth_note = False
        self.show_mild_inlet_control = False
