"""CalcData for performing Culvert Barrel operations."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import CalcData
from xms.FhwaVariable.core_data.variables.user_array import UserArray
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.channel_geometry.channel_geometry_data import ChannelGeometryData
from xms.HydraulicToolboxCalc.hydraulics.culvert.culvert_barrel_calc import CulvertBarrelCalc
from xms.HydraulicToolboxCalc.hydraulics.culvert.inlet_control_data import InletControlData
from xms.HydraulicToolboxCalc.hydraulics.culvert.site_data import SiteData
from xms.HydraulicToolboxCalc.hydraulics.GVF.gvf_data import GVFData
from xms.HydraulicToolboxCalc.hydraulics.GVF.specify_boundary_data import SpecifyBoundaryData
from xms.HydraulicToolboxCalc.hydraulics.manning_n.composite_manning_n_data import CompositeManningNData
from xms.HydraulicToolboxCalc.hydraulics.manning_n.manning_n_data import ManningNData


class CulvertBarrelData(CalcData):
    """A class that defines a channel and performs Culvert barrel computations."""

    def __init__(self, clear_my_own_results=True, in_crossing=False, app_data=None,
                 model_name=None, project_uuid=None):
        """Initializes the Manning's n calculator.

        Args:
            clear_my_own_results (bool): Should the class clean up its results?  If Standalone, yet. otherwise, no.
            in_crossing (bool): Is this a crossing?
            app_data (AppData): application data (settings, etc.).
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = "Culvert barrel Calculator"
        self.type = "CulvertBarrelCalc"
        self.class_name = 'Culvert Barrel Calculator'

        self.calculator = CulvertBarrelCalc()

        self.calc_support_dual_input = False

        self.in_crossing = in_crossing

        # self.set_name(index_of_culvert)

        # Input
        self.input['Name'] = Variable('Name', 'string', self.name)
        self.input['Flows'] = Variable(
            'Flow(s)', 'UserArray', UserArray(2, ['flow'], 'cfs', us_units=self.us_flow, si_units=self.si_flow,
                                              select_name='specify flow(s) as:', name_append='flow'))
        self.input['Geometry'] = Variable(
            'Geometry', 'class',
            ChannelGeometryData(stand_alone_calc=False, hide_depth=True, allowed_shapes='closed', app_data=app_data,
                                model_name=model_name, project_uuid=project_uuid))
        self.input['Inlet data'] = Variable(
            'Inlet data', 'class', InletControlData(app_data=app_data, model_name=model_name,
                                                    project_uuid=project_uuid))
        self.input['Composite n'] = Variable(
            "Manning's n", 'class',
            CompositeManningNData(shape='circle', open_shape=False, embedment_present=False, gradation_present=False,
                                  total_embedment=0.0, stations=[], elevations=[], app_data=app_data,
                                  model_name=model_name, project_uuid=project_uuid))
        self.input['Composite n'].get_val().input['Channel n entry'].set_val('specify material')
        self.input['Composite n'].get_val().input['n entry'].set_val('specify material')

        self.input['Site data'] = Variable('Site data', 'class', SiteData(
            0.0, app_data=app_data, model_name=model_name, project_uuid=project_uuid))

        self.input['Downstream water depth'] = Variable(
            'Downstream water depth', 'class', SpecifyBoundaryData(
                boundary_name='downstream', allow_normal_depth=True, app_data=app_data, model_name=model_name,
                project_uuid=project_uuid))

        # Maybe?
        self.input['Display channel slope as flat'] = Variable(
            'Display channel slope as flat', 'bool', False)

        self.unknown = None

        # Intermediate
        self.compute_prep_functions.extend([])
        self.compute_finalize_functions.extend([])
        self.intermediate_to_copy.extend([
            'in_crossing', 'unknown', 'manning_n_calc', 'min_embedment_depth', 'inlet_invert_elevation',
            'outlet_invert_elevation', 'single_barrel_flow', 'flowing_full', 'max_open_channel_flow',
            'inlet_control_depth', 'outlet_control_depth', 'headwaterDepth', 'hw', 'inlet_water_depth',
            'inlet_water_velocity', 'full_flow_area', 'full_flow_hydraulic_radius', 'full_flow_velocity',
            'length_flowing_full', 'full_flow_n', 'span', 'rise', 'slope', 'critical_depth', 'normal_depth',
            'show_oscillitary_flow_note', 'oscillitary_flow_note', 'show_depth_note', 'depth_note',
            'show_mild_inlet_control', 'mild_inlet_control', 'barrel_segments', 'critical_depths', 'critical_flowareas',
            'critical_velocities', 'normal_depths', 'normal_velocities', 'wse_stations', 'wse_elevations', 'tw_depth',
            'tw_velocity', 'outlet_depth', 'outlet_velocity', 'flow', 'flows', 'hyd_jump_exists', 'hyd_jump_swept_out',
            'hyd_jump_station_absolute', 'hyd_jump_final_station_absolute', 'hyd_jump_elevation',
            'hyd_jump_sequent_elevation', 'hyd_jump_depth', 'hyd_jump_sequent_depth', 'intermediate_results', 'depth',
            'flow_area', 'wetted_perimeter', 'top_width', 'hydraulic_radius', 'manning_n', 'velocity', 'energy',
            'energy_loss', 'energy_slope', 'friction_slope_ave', 'delta_x', 'x', 'y', 'sequent_depth', 'froude',
            'boundary_shear_stress', 'max_shear_stress', 'results_upstream', 'results_downstream', 'plot_x', 'plot_y',
            'compute_vena_contracta', 'ignore_energy_in_full_flow_calcs', 'hyd_jump_type_b_over_break',
            'hyd_jump_exists', 'hyd_jump_swept_out', 'froude_jump', 'join_curves', 'gvf_base_calc'
        ])
        self.theme = self.get_theme()
        manning_n_data = ManningNData(standalone_calc=False, app_data=app_data, model_name=model_name,
                                      project_uuid=project_uuid)
        manning_n_data.prepare_for_compute()
        self.manning_n_calc = manning_n_data.calculator
        self.manning_n_calc.input_dict, self.manning_n_calc.plot_dict = manning_n_data.prepare_input_dict()

        gvf_data = GVFData(clear_my_own_results=False, app_data=app_data, model_name=model_name,
                           project_uuid=project_uuid)
        gvf_data.prepare_for_compute()
        self.gvf_base_calc = gvf_data.calculator
        self.gvf_base_calc.input_dict, self.gvf_base_calc.plot_dict = gvf_data.prepare_input_dict()

        self.min_embedment_depth = 0.0

        self.inlet_invert_elevation = 0.0
        self.outlet_invert_elevation = 0.0

        self.single_barrel_flow = 0.0
        self.flowing_full = False
        self.max_open_channel_flow = 0.0

        self.inlet_control_depth = 0.0
        self.outlet_control_depth = 0.0
        self.headwaterDepth = 0.0
        self.hw = 0.0

        self.inlet_water_depth = 0.0
        self.inlet_water_velocity = 0.0  # Assume dead pool upstream of culvert barrel

        # Full flow parameters
        self.full_flow_area = 0.0
        self.full_flow_hydraulic_radius = 0.0
        self.full_flow_velocity = 0.0
        self.length_flowing_full = 0.0
        self.full_flow_n = 0.0
        self.span = 0.0
        self.rise = 0.0
        self.slope = 0.0

        self.critical_depth = 0.0
        self.normal_depth = 0.0

        # items that need messages
        self.show_oscillitary_flow_note = False  # m_showOscilateNote
        self.oscillitary_flow_note = 'An unsteady, oscillatory hydraulic jump is possible, but full flow is likely.'
        self.show_depth_note = False  # m_showDepthNote
        self.depth_note = '* inlet control elevation is below inlet invert.'
        self.show_mild_inlet_control = False  # m_showMildInletControlNote
        self.mild_inlet_control = 'Inlet control is shown, but flow profile is substantially FF.'

        self.barrel_segments = []
        self.critical_depths = []
        self.critical_flowareas = []
        self.critical_velocities = []
        self.normal_depths = []
        self.normal_velocities = []
        self.wse_stations = []
        self.wse_elevations = []
        self.tw_depth = 0.0
        self.tw_velocity = 0.0
        self.outlet_depth = 0.0
        self.outlet_velocity = 0.0

        self.flow = 0.0
        self.flows = [0.0]

        self.hyd_jump_exists = False
        self.hyd_jump_swept_out = False
        self.hyd_jump_station_absolute = []
        self.hyd_jump_final_station_absolute = []
        self.hyd_jump_elevation = []
        self.hyd_jump_sequent_elevation = []
        self.hyd_jump_depth = []
        self.hyd_jump_sequent_depth = []

        # lists for direct step point data
        self.intermediate_results = {
            'Depth': [],
            'flow_area': [],
            'wetted_perimeter': [],
            'top_width': [],
            'hydraulic_radius': [],
            'manning_n': [],
            'Velocity': [],
            'Energy': [],
            'energy_loss': [],
            'energy_slope': [],
            'friction_slope_ave': [],
            'delta_x': [],
            'x': [],
            'y': [],
            'sequent_depth': [],
            'froude': [],
            'boundary_shear_stress': [],
            'max_shear_stress': []
        }

        self.depth = []
        self.flow_area = []
        self.wetted_perimeter = []
        self.top_width = []
        self.hydraulic_radius = []
        self.manning_n = []
        self.velocity = []
        self.energy = []
        self.energy_loss = []
        self.energy_slope = []
        self.friction_slope_ave = []
        self.delta_x = []
        self.x = []
        self.y = []
        self.sequent_depth = []
        self.froude = []
        self.boundary_shear_stress = []
        self.max_shear_stress = []

        self.results_upstream = {}
        self.results_downstream = {}

        self.plot_x = []
        self.plot_y = []

        # Vena Contracta
        self.compute_vena_contracta = False

        # Hydraulic Jump
        self.ignore_energy_in_full_flow_calcs = False
        self.hyd_jump_type_b_over_break = False
        self.hyd_jump_exists = False
        self.hyd_jump_swept_out = False
        self.froude_jump = False
        self.join_curves = False

        # Results
        self.clear_my_own_results = clear_my_own_results
        self.results = {}

        self.results['Station'] = Variable(
            'Station', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max), precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['WSE'] = Variable(
            'WSE', 'float_list', 0.0, [], limits=(-sys.float_info.max, sys.float_info.max), precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Distance from inlet'] = Variable(
            'Distance from inlet', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Depth'] = Variable(
            'Depth', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Flow area'] = Variable(
            'Flow area', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft^2',
            us_units=[['yd^2', 'ft^2', 'in^2']], si_units=[['m^2', 'mm^2']])
        self.results['Wetted perimeter'] = Variable(
            'Wetted perimeter', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Top width'] = Variable(
            'Top width', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Hydraulic radius'] = Variable(
            'Hydraulic radius', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Manning n'] = Variable(
            'Critical depth', 'float_list', 0.0, [], precision=4, unit_type=['coefficient'], native_unit='coefficient',
            us_units=[['coefficient']], si_units=[['coefficient']])
        self.results['Velocity'] = Variable(
            'Velocity', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft/s',
            us_units=[['yd/s', 'ft/s', 'in/s']], si_units=[['m/s', 'mm/s']])
        self.results['Energy'] = Variable(
            'Energy', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Energy loss'] = Variable(
            'Energy loss', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Energy slope'] = Variable(
            'Energy slope', 'float_list', 0.0, limits=[-10.0, 10.0], precision=6, unit_type=['slope'],
            native_unit='ft/ft', us_units=[['ft/ft']], si_units=[['m/m']])
        self.results['Boundary shear stress'] = Variable(
            'Boundary shear stress', 'float_list', 0.0, precision=2, unit_type=['stress'], native_unit='psf',
            us_units=[['psf']], si_units=[['pa']])
        self.results['Max shear stress'] = Variable(
            'Max shear stress', 'float_list', 0.0, precision=2, unit_type=['stress'], native_unit='psf',
            us_units=[['psf']], si_units=[['pa']])

        self.warnings = []

        # plot
        self.plots['profile'] = {}
        self.plots['profile']['Plot name'] = "Cross-section"
        self.plots['profile']['Legend'] = 'best'
        self.plots['profile']['index'] = 1

    # def set_name(self, index_of_culvert):
    #     """Set the name of the culvert barrel.

    #     Args:
    #         index_of_culvert (int): used to initialize the name to make it unique.
    #     """
    #     if index_of_culvert is None:
    #         self.name = 'Culvert barrel calc'
    #     elif index_of_culvert >= 0:
    #         self.name = f'Culvert {index_of_culvert + 1}'

    # def update_name(self):
    #     """Update the name from the input variable."""
    #     self.name = self.input['Name'].get_val()

    # def update_input_name(self):
    #     """Update the name from the input variable."""
    #     self.input['Name'].set_val(self.name)

    def get_input_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and left out of the input dictionary)

        Returns:
              input_vars (dictionary of variables): the input variables
        """
        input_vars = {}

        if not self.in_crossing:
            input_vars['Flows'] = self.input['Flows']
        else:
            self.name = self.input['Name'].get_val()
            input_vars['Name'] = self.input['Name']

        input_vars['Geometry'] = copy.copy(self.input['Geometry'])
        culvert_type = self.input['Site data'].get_val().input['Culvert type'].get_val()
        if culvert_type == 'straight':
            input_vars['Geometry'].get_val().allow_embedment = True
        else:
            input_vars['Geometry'].get_val().allow_embedment = False

        # input_vars['Geometry'] = self.input['Geometry'].value.get_input_group()

        # banned_items = ['Calculate', 'Head', 'Flows']
        # for item in banned_items:
        #     if item in input_vars['Geometry'].value.input:
        #         input_vars['Geometry'].value.input.pop(item)

        input_vars['Composite n'] = self.input['Composite n']

        input_vars['Inlet data'] = self.input['Inlet data']

        input_vars['Site data'] = self.input['Site data']

        if not self.in_crossing:
            input_vars['Downstream water depth'] = self.input[
                'Downstream water depth']

            input_vars['Display channel slope as flat'] = self.input[
                'Display channel slope as flat']

        return input_vars

    def get_results_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
              result_vars (dictionary of variables): the input variables
        """
        result_vars = {}

        if not self.get_can_compute():
            return result_vars

        if self.input['Display channel slope as flat'].get_val():
            result_vars['Distance from inlet'] = self.results[
                'Distance from inlet']
            result_vars['Depth'] = self.results['Depth']
        else:
            result_vars['Station'] = self.results['Station']
            result_vars['WSE'] = self.results['WSE']
        result_vars['Flow area'] = self.results['Flow area']
        result_vars['Wetted perimeter'] = self.results['Wetted perimeter']
        result_vars['Top width'] = self.results['Top width']
        result_vars['Hydraulic radius'] = self.results['Hydraulic radius']
        # result_vars['Manning n'] = self.results['Manning n']
        result_vars['Velocity'] = self.results['Velocity']
        # result_vars['Energy'] = self.results['Energy']
        # result_vars['Energy loss'] = self.results['Energy loss']
        result_vars['Energy slope'] = self.results['Energy slope']
        result_vars['Boundary shear stress'] = self.results[
            'Boundary shear stress']
        result_vars['Max shear stress'] = self.results['Max shear stress']

        return result_vars

    def assign_results(self):
        """Assigns the results from a computational run to the appropriate locations."""
        # Assign the results
        _, gamma = self.get_setting('Unit weight of water (γw)')

        multiple = False
        if multiple:
            pass
        else:
            self.boundary_shear_stress = []
            self.max_shear_stress = []

            for energy_slope, hyd_rad, depth in zip(self.energy_slope,
                                                    self.hydraulic_radius,
                                                    self.y):
                self.boundary_shear_stress.append(gamma * hyd_rad * energy_slope)
                self.max_shear_stress.append(gamma * depth * energy_slope)

            self.results['Station'].set_val(self.wse_stations)
            self.results['WSE'].set_val(self.wse_elevations)
            self.results['Distance from inlet'].set_val(self.x)
            self.results['Depth'].set_val(self.y)
            self.results['Flow area'].set_val(self.flow_area)
            self.results['Wetted perimeter'].set_val(self.wetted_perimeter)
            self.results['Top width'].set_val(self.top_width)
            self.results['Hydraulic radius'].set_val(self.hydraulic_radius)
            self.results['Manning n'].set_val(self.manning_n)
            self.results['Velocity'].set_val(self.velocity)
            self.results['Energy'].set_val(self.energy)
            self.results['Energy loss'].set_val(self.energy_loss)
            self.results['Energy slope'].set_val(self.energy_slope)
            self.results['Boundary shear stress'].set_val(
                self.boundary_shear_stress)
            self.results['Max shear stress'].set_val(self.max_shear_stress)

            if self.input['Display channel slope as flat'].get_val():
                self.plot_x = self.x
                self.plot_y = self.y
            else:
                self.plot_x = self.wse_stations
                self.plot_y = self.wse_elevations

            if self.hyd_jump_swept_out:
                self.warnings.append(
                    'An hydraulic jump forms near the culvert outlet but may be swept into the tailwater channel'
                )

    def check_warnings(self):
        """Checks for warnings that are given during computations or a check if we can compute (get_can_compute).

        Returns:
            list of str: The warnings found (if any)
        """
        return self.warnings

    def clear_results(self):
        """Clears the results and those of subclasses to prepare for computation."""
        self.input['Geometry'].get_val().clear_results()
