"""Provides a class that will Handle user-selected for boundary conditions."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math

# 2. Third party modules
from sortedcontainers import SortedDict

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator import Calculator

# 4. Local modules
from xms.HydraulicToolboxCalc.util.interpolation import Interpolation


class InletControlCalc(Calculator):
    """Provides a class that will Handle user-selected for boundary conditions.

    Boundary Conditions: Normal depth, critical depth, constant depth, rating curve, specified depths.
    """

    def _get_can_compute(self):
        """Determine whether we have enough data to compute.

        Returns:
            True, if we can compute; otherwise, False
        """
        result = True

        full_flow_area = self.input_dict['calc_data']['Full flow area']
        span = self.input_dict['calc_data']['Span']
        rise = self.input_dict['calc_data']['Rise']
        slope = self.input_dict['calc_data']['Slope']

        if full_flow_area <= 0.0:
            self.warnings['Full flow area'] = 'Full flow area is not specified.'
            result = False
        if span <= 0.0:
            self.warnings['Span'] = 'Span is not specified.'
            result = False
        if rise <= 0.0:
            self.warnings['Rise'] = 'Rise is not specified.'
            result = False
        if slope <= 0.0:
            self.warnings['Slope'] = 'Slope is not specified.'
            result = False

        if self.input_dict['calc_data']['inlet type'] == 'manual polynomial':
            if self.input_dict['calc_data']['A'] == 0.0:
                self.warnings['A'] = 'Polynomial coefficients A value is not specified.'
                result = False
            if self.input_dict['calc_data']['BS'] == 0.0:
                self.warnings['BS'] = 'Polynomial coefficients BS value is not specified.'
                result = False
            if self.input_dict['calc_data']['C'] == 0.0:
                self.warnings['C'] = 'Polynomial coefficients C value is not specified.'
                result = False
            if self.input_dict['calc_data']['DIP'] == 0.0:
                self.warnings['DIP'] = 'Polynomial coefficients DIP value is not specified.'
                result = False
            if self.input_dict['calc_data']['EE'] == 0.0:
                self.warnings['EE'] = 'Polynomial coefficients EE value is not specified.'
                result = False
            if self.input_dict['calc_data']['F'] <= 0.0:
                self.warnings['F'] = 'Polynomial coefficients F value is not specified.'
                result = False
            # Conspan culverts do not specify a slope correction coefficient
            # if self.input_dict['calc_data']['SR'] < 0.0:
            #     self.warnings['SR'] = 'Polynomial coefficients SR value is not specified.'
            #     result = False
            return result
        # elif self.input_dict['calc_data']['inlet type'] == 'interpolation: Compute':
        #     self.warnings['Interpolation: Compute'] = \
        #         'Interpolation: Compute is not implemented. Please use the other inlet options.'
        #     return False

        return result

    def _compute_data(self, critical_depth=None, critical_flowarea=None):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        flows = self.input_dict['calc_data']['Flows']
        # full_flow_area = self.input['Full flow area'].get_val()
        # span = self.input['Span'].get_val()
        # rise = self.input['Rise'].get_val()
        # slope = self.input['Slope'].get_val()

        # flows = flow_var.get_val().get_result()
        self.hw = []
        for flow in flows:
            hw = self.compute_inlet_head_for_flow(flow, critical_depth, critical_flowarea)
            # hw = self.compute_data_for_flow(flow, full_flow_area, span, rise, slope)
            self.hw.append(hw)

        self.results['hw'] = self.hw
        return self.hw

    def set_ke(self):
        """Sets the KE value, if not specified by the user."""
        if self.input_dict['calc_data']['inlet type'] in ['best available', 'manual polynomial']:
            # ke value is user-entered
            return None
        else:
            if self.input_dict['calc_data']['inlet type'] == 'interpolation: circular or elliptical':
                if self.input_dict['calc_data']['Inlet configuration'] == 'beveled edge':
                    self.input_dict['calc_data']['KE'] = self.user_cir_beveled_ke
                elif self.input_dict['calc_data']['Inlet configuration'] == 'mitered to conform to slope':
                    self.input_dict['calc_data']['KE'] = self.user_cir_mitered_ke
                elif self.input_dict['calc_data']['Inlet configuration'] == 'square edge with headwall':
                    self.input_dict['calc_data']['KE'] = self.user_cir_square_ke
                elif self.input_dict['calc_data']['Inlet configuration'] == 'thin edge projecting':
                    self.input_dict['calc_data']['KE'] = self.user_cir_projecting_ke
            elif self.input_dict['calc_data']['inlet type'] == 'interpolation: arch or embedded':
                if self.input_dict['calc_data']['Inlet configuration'] == 'beveled edge':
                    self.input_dict['calc_data']['KE'] = self.user_arch_beveled_ke
                elif self.input_dict['calc_data']['Inlet configuration'] == 'mitered to conform to slope':
                    self.input_dict['calc_data']['KE'] = self.user_arch_mitered_ke
                elif self.input_dict['calc_data']['Inlet configuration'] == 'square edge with headwall':
                    self.input_dict['calc_data']['KE'] = self.user_arch_square_ke
                elif self.input_dict['calc_data']['Inlet configuration'] == 'thin edge projecting':
                    self.input_dict['calc_data']['KE'] = self.user_arch_projecting_ke
        return self.input_dict['calc_data']['KE']

    def initialize_inlet_data_for_culvert_with_manning_n_calc(self, mannings_n_calc):
        """Compute and set coefficient and data for the culvert to perform inlet computations.

        Args:
            mannings_n_calc (ManningsNCalculator): The Manning's n calculator to use for computations.
        """
        mannings_n_calc.get_can_compute()
        mannings_n_calc.input_dict['calc_data']['Geometry']['calculator'].initialize_geometry()
        shape = mannings_n_calc.input_dict['calc_data']['Geometry']['Shape']
        span = mannings_n_calc.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Span']
        rise = mannings_n_calc.input_dict['calc_data']['Geometry']['calculator'].input_dict['calc_data']['Rise']
        fullflow_area = mannings_n_calc.input_dict['calc_data']['Geometry']['calculator'].full_flow_area
        slope = mannings_n_calc.input_dict['calc_data']['Slope']

        if mannings_n_calc.can_compute:
            self.initialize_inlet_data_for_culvert(shape, span, rise, fullflow_area, slope, mannings_n_calc)

    def initialize_inlet_data_for_culvert(self, shape, span, rise, fullflow_area, slope, mannings_n_calc):
        """Compute and set coefficient and data for the culvert to perform inlet computations.

        Args:
            shape (str): the shape of the culvert
            span (float): the span of the culvert
            rise (float): the rise of the culvert
            fullflow_area (float): the full flow area of the culvert
            slope (float): the slope of the culvert
            mannings_n_data (ManningsNData): the Manning's n data for the culvert

        Returns:
            bool: True if successful

        """
        self.flow_at_half_rise = 0.0
        self.flow_at_triple_rise = 0.0

        _, g = self.get_data('Gravity')

        self.mannings_n_calc = mannings_n_calc
        self.mannings_n_calc.clear_results()

        # Set given data
        self.input_dict['calc_data']['Shape'] = shape
        self.input_dict['calc_data']['Rise'] = rise
        self.input_dict['calc_data']['Span'] = span
        self.input_dict['calc_data']['Full flow area'] = fullflow_area
        self.input_dict['calc_data']['Slope'] = slope

        # create variables for transition points
        half_rise = rise / 2.0
        triple_rise = 3.0 * rise

        _, null_data = self.get_data('Null data')
        _, zero_tol = self.get_data('Zero tolerance')
        # Add Zero (floor of computation)
        self.flow_unk_sorted = SortedDict({0.0: 0.0})
        self.flow_interp = Interpolation([], [], null_data=null_data, zero_tol=zero_tol)

        # Compute a range of flows with the regression equations to enable our interpolation routines to find the flow
        # at specific headwaters
        new_flow = 1.0
        ih = 0.0
        while (ih < triple_rise):
            ih = self.compute_hw_regression_equations_for_flow(new_flow)
            self.flow_unk_sorted[ih] = new_flow
            new_flow *= 2.0

        # The following methods come from:
        # "Hydraulic Analysis of Culverts by Microcomputer," A Thesis in Civil Engineering by Arthur C. Parola,
        # Jr., May 1987, 84p. Chapter 3, Program Computations, page 20

        # Compute the lower limit velocity head coefficient (kelow) at half rise
        # The velocity head coefficient (kelow) is determined by setting the minimum energy equation equal to the
        # regression equation at one half the rise, which is the lower limit of the regression equation.
        self.flow_at_half_rise = self.compute_flow_for_given_inlet_hw(half_rise)

        # Determine flow conditions at critical depth for given discharge
        self.mannings_n_calc.input_dict['calc_data']['Flows'] = [self.flow_at_half_rise]
        self.mannings_n_calc.initialize_geometry()
        critical_depth_with_flow_at_half_rise = self.mannings_n_calc._compute_critical_depth_from_flow()[0]
        critical_flow_area_with_flow_at_half_rise = self.mannings_n_calc.results['Critical flow area'][0]

        if self.flow_at_half_rise == 0.0 or critical_flow_area_with_flow_at_half_rise == 0.0:
            self.kelow = 0.0
        else:
            self.kelow = ((half_rise - critical_depth_with_flow_at_half_rise)) * (2.0 * g) / \
                (self.flow_at_half_rise ** 2 / critical_flow_area_with_flow_at_half_rise ** 2) - 1.0

        # Compute the upper limit coefficient at 3 * rise
        # The coefficient of contraction (cdahi) is determined by setting the orifice equation equal to the regression
        # equation at the upper limit of the regression equation.
        self.flow_at_triple_rise = self.compute_flow_for_given_inlet_hw(triple_rise)

        self.cdahi = 0.0
        if rise > 0.0:
            self.cdahi = self.flow_at_triple_rise / math.sqrt(2.5 * rise)

    def compute_inlet_head_for_flow(self, flow, critical_depth=None, critical_flowarea=None):
        """Computes the inlet head for the given flow.

        Returns:
            bool: True if successful
        """
        ih = 0.0

        _, g = self.get_data('Gravity')

        # The following methods come from:
        # "Hydraulic Analysis of Culverts by Microcomputer," A Thesis in Civil Engineering by Arthur C. Parola, Jr.,
        # May 1987, 84p. Chapter 3, Program Computations, page 20
        # if flow is less than half the rise (below experimental data from regression curves), use the energy
        # equation to determine the inlet head control depth
        if flow < self.flow_at_half_rise:
            # Method to adjust the coefficients to keep flow area from going to zero as we approach zero depth
            shape = self.input_dict['calc_data']['Shape']

            # Determine flow conditions at critical depth for given discharge
            self.mannings_n_calc.input_dict['calc_data']['Flows'] = [flow]
            if critical_depth is None:
                critical_depth = self.mannings_n_calc._compute_critical_depth_from_flow()[0]
                critical_flowarea = self.mannings_n_calc.results['Critical flow area'][0]
            if (critical_flowarea > 0.0):
                vel_head = flow**2 / (critical_flowarea**2 * (2 * g))
            else:
                vel_head = 0.0

            fract = 1.0
            vhcoef = 1.0
            lmult = 1.0
            if shape not in ['circle', 'box', ]:
                q15 = self.flow_at_half_rise * 0.15
                # q10 = self.flow_at_half_rise * 0.1
                fract = (q15 - flow) / (q15)
                vhcoef = (1 - fract) / (1 + vel_head * fract)
            ih = critical_depth * lmult + (1 + self.kelow) * vel_head * vhcoef
        # if the flow is greater than half the rise and less than 3 times the rise, use the regression equations
        elif flow <= self.flow_at_triple_rise:
            ih = self.compute_hw_regression_equations_for_flow(flow)
        # if the flow is greater than 3 times the rise, use the orifice equation
        else:
            ih = flow**2 / self.cdahi**2 + 0.5 * self.input_dict['calc_data']['Rise']

        return ih

    def compute_flow_for_given_inlet_hw(self, hw):
        """Compute the flow required to reach a given HW.

        Args:
            hw (float): the given headwater depth
        Returns:
            float: flow for the given headwater
        """
        flow_list = list(self.flow_unk_sorted.values())
        hw_list = list(self.flow_unk_sorted.keys())
        if hw in hw_list:
            return self.flow_unk_sorted[hw]
        self.flow_interp.x = hw_list
        self.flow_interp.y = flow_list

        flow_guess, _ = self.flow_interp.interpolate_y(hw)

        _, hw_tol = self.get_data('HW error')
        _, max_loops = self.get_data('Max number of iterations')
        difference = 1.0
        count = 0

        while hw_tol < difference and count < max_loops:
            ih = self.compute_hw_regression_equations_for_flow(flow_guess)
            self.flow_unk_sorted[ih] = flow_guess

            # # Convert the dictionary to a list so we can interpolate from it
            flow_list = list(self.flow_unk_sorted.values())
            hw_list = list(self.flow_unk_sorted.keys())
            self.flow_interp.x = hw_list
            self.flow_interp.y = flow_list
            flow_guess, _ = self.flow_interp.interpolate_y(hw)

            difference = abs(hw - ih)
            count += 1

        return flow_guess

    def compute_hw_regression_equations_for_flow(self, flow):
        """Computes the inlet head from regression equation given only the flow.

        Requires the initialize_inlet_data_for_culvert to be run first

        Args:
            flow (float): Given flow to determine the inlet head

        Returns:
            float: inlet control headwater
        """
        rise = self.input_dict['calc_data']['Rise']
        span = self.input_dict['calc_data']['Span']
        fullflow_area = self.input_dict['calc_data']['Full flow area']
        slope = self.input_dict['calc_data']['Slope']

        return self.compute_hw_regression_equations_for_flow_area_span_rise_slope(flow, fullflow_area, span, rise,
                                                                                  slope)

    # flow_area == full flow area?  I think so...
    def compute_hw_regression_equations_for_flow_area_span_rise_slope(self, flow, flow_area, span, rise, slope):
        """Computes the data possible; stores results in self.

        Returns:
            float: inlet control headwater
        """
        hw = 0.0

        self.set_ke()

        if flow <= 0.0 or flow_area <= 0.0 or rise <= 0.0:
            return hw

        # TODO Find where KE is used in HY-8!
        sr = 0.0  # Slope correction coefficient

        if self.input_dict['calc_data']['inlet type'] == 'manual polynomial':
            x = flow / (span * rise**1.5)
            if self.use_qad_poly:
                # Compute Q/AD^0.5; some shapes like CONSPAN were developed with Q/AD^0.5
                x = flow / (flow_area * math.sqrt(rise))

            a = self.input_dict['calc_data']['A']
            b = self.input_dict['calc_data']['BS']
            c = self.input_dict['calc_data']['C']
            d = self.input_dict['calc_data']['DIP']
            e = self.input_dict['calc_data']['EE']
            f = self.input_dict['calc_data']['F']
            sr = self.input_dict['calc_data']['SR']

            # Polynomial Equation
            hw_d = a + b * x + c * x**2 + d * x**3 + e * x**4 + f * x**5 - sr * slope
            if hw_d < 0.0:
                hw_d = 0.0
        else:
            # Compute Q/AD^0.5
            q_ad = flow / (flow_area * math.sqrt(rise))
            x = self.user_x
            if self.input_dict['calc_data']['inlet type'] == 'interpolation: circular or elliptical':
                if self.input_dict['calc_data']['Inlet configuration'] == 'beveled edge':
                    y = self.user_arch_beveled_hwd
                    sr = 0.5
                elif self.input_dict['calc_data']['Inlet configuration'] == 'mitered to conform to slope':
                    y = self.user_arch_mitered_hwd
                    sr = -0.7
                elif self.input_dict['calc_data']['Inlet configuration'] == 'square edge with headwall':
                    y = self.user_arch_square_hwd
                    sr = 0.5
                elif self.input_dict['calc_data']['Inlet configuration'] == 'thin edge projecting':
                    y = self.user_arch_projecting_hwd
                    sr = 0.5
            elif self.input_dict['calc_data']['inlet type'] == 'interpolation: arch or embedded':
                if self.input_dict['calc_data']['Inlet configuration'] == 'beveled edge':
                    y = self.user_cir_beveled_hwd
                    sr = 0.5
                elif self.input_dict['calc_data']['Inlet configuration'] == 'mitered to conform to slope':
                    y = self.user_cir_mitered_hwd
                    sr = -0.7
                elif self.input_dict['calc_data']['Inlet configuration'] == 'square edge with headwall':
                    y = self.user_cir_square_hwd
                    sr = 0.5
                elif self.input_dict['calc_data']['Inlet configuration'] == 'thin edge projecting':
                    y = self.user_cir_projecting_hwd
                    sr = 0.5

            else:
                return
            _, null_data = self.get_data('Null data')
            _, zero_tol = self.get_data('Zero tolerance')
            interpolate = Interpolation(x, y, null_data=null_data, zero_tol=zero_tol)
            hw_d = interpolate.interpolate_y(q_ad, True)[0] - sr * slope
            if hw_d < 0.0:
                hw_d, _ = interpolate.interpolate_y(q_ad, True)
        hw = hw_d * rise
        return hw
