"""Provides a class that will handle the weir-flow computations across a roadway."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator import Calculator

# 4. Local modules


class RoadwayCalc(Calculator):
    """Provides a class that will handle the weir-flow computations across a roadway."""

    def _get_can_compute(self):
        """Determine whether we have enough data to compute.

        Returns:
            True, if we can compute; otherwise, False
        """
        roadway_type = self.input_dict['calc_data']['Roadway profile shape']

        # Check for flows or head
        if roadway_type == 'constant roadway elevation':
            if self.input_dict['calc_data']['Crest length'] <= 0.0:
                self.warnings['Crest length'] = 'Please enter a roadway weir crest length'
            if self.input_dict['calc_data']['Crest elevation'] <= 0.0:
                self.warnings['Crest elevation'] = 'Please enter a roadway weir crest elevation'
        elif roadway_type == 'irregular roadway':
            # Check for stations and elevations
            _, null_data = self.get_data('Null data')
            stations = self.input_dict['calc_data']['Irregular roadway geometry']['Roadway stations']
            # elevations = self.input_dict['calc_data']['Irregular roadway geometry']['Roadway elevations']
            filtered = [x for x in stations if x != null_data and x != 0 and stations.count(x) > 1]
            if len(filtered) == 0:
                self.warnings['irregular roadway'] = 'Please enter at least two unique roadway stations'
        if self.input_dict['calc_data']['Roadway top width'] <= 0.0:
            self.warnings['Roadway top width'] = 'Please enter a roadway top width'

        if len(self.warnings) > 0:
            return False
        return True

    def determine_overtopping_elevation(self):
        """Determines the overtopping elevation for the roadway."""
        if self.input_dict['calc_data']['Roadway profile shape'] == 'constant roadway elevation':
            return self.input_dict['calc_data']['Crest elevation']
        else:
            return min(self.input_dict['calc_data']['Irregular roadway geometry']['Roadway elevations'])

    def compute_data_for_wse(self, wse, tailwater=0.0):
        """Computes the data possible; stores results in self.

        Args:
            wse (float): the water surface elevation upstream
            tailwater (float): the tailwater surface elevation downstream
        Returns:
            tuple(bool, flow): (True if successful, resulting flow for given wse
        """
        roadway_type = self.input_dict['calc_data']['Roadway profile shape']
        if roadway_type == 'constant roadway elevation':
            crest_elevation = self.input_dict['calc_data']['Crest elevation']
        elif roadway_type == 'irregular roadway':
            elevations = self.input_dict['calc_data']['Irregular roadway geometry']['Roadway elevations']
            crest_elevation = min(elevations)
        if wse <= crest_elevation:
            # No overtopping
            self.results = {'Flows': 0.0}
            return True, [0.0]

        self.weir.input_dict['calc_data']['Calculate'] = 'Flows'
        self.weir.input_dict['calc_data']['Head'] = 'Elevation'
        self.weir.input_dict['calc_data']['WSE'] = [wse]
        self.weir.input_dict['calc_data']['Tailwater elevations'] = [tailwater]
        result = self._compute_data()
        flow = self.results['Flows']
        return result, flow

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        stations = []
        elevations = []

        self.weir.input_dict['calc_data']['Weir type'] = 'irregular'

        roadway_type = self.input_dict['calc_data']['Roadway profile shape']
        roadway_surface = self.input_dict['calc_data']['Roadway surface']
        self.weir.input_dict['calc_data']['Weir surface'] = roadway_surface
        self.weir.input_dict['calc_data']['Weir coefficient'] = self.input_dict['calc_data']['Weir coefficient']
        roadway_top_width = self.input_dict['calc_data']['Roadway top width']
        if roadway_type == 'constant roadway elevation':
            first_station = self.input_dict['calc_data']['First roadway station']
            crest_length = self.input_dict['calc_data']['Crest length']
            crest_elevation = self.input_dict['calc_data']['Crest elevation']
            stations = [first_station, first_station + crest_length]
            elevations = [crest_elevation, crest_elevation]
        elif roadway_type == 'irregular roadway':
            stations = self.input_dict['calc_data']['Irregular roadway geometry']['Roadway stations']
            elevations = self.input_dict['calc_data']['Irregular roadway geometry']['Roadway elevations']

        self.weir.input_dict['calc_data']['Weir width'] = roadway_top_width
        self.weir.input_dict['calc_data']['Tailwater'] = self.input_dict['calc_data']['Tailwater']

        self.weir.input_dict['calc_data']['Irregular weir crest geometry']['Crest stations'] = stations
        self.weir.input_dict['calc_data']['Irregular weir crest geometry']['Crest elevations'] = elevations

        # self.weir
        self.weir.input_dict['calc_data']['Flows'] = self.input_dict['calc_data']['Flows']

        self.weir.compute_data()

        self.results = self.weir.results

        return True
