"""Provides a class that will perform tailwater calculations and provide tailwater options."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
from uuid import uuid4

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.app_data.app_data import AppData
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.culvert.rating_curve import RatingCurveTable
from xms.HydraulicToolboxCalc.hydraulics.culvert.tailwater_calc import TailwaterCalc
from xms.HydraulicToolboxCalc.hydraulics.manning_n.manning_n_data import ManningNData


class TailwaterData(ManningNData):
    """Provides a class that will perform tailwater calculations and provide tailwater options."""

    def __init__(self, allowed_shapes: str = 'open channels', default_shape: str = 'trapezoidal',
                 hide_depth: bool = True, app_data: AppData = None, model_name: str = None,
                 project_uuid: uuid4 = None):  # , index_of_culvert=1):
        """Initializes the Tailwater Class.

        Args:
            allowed_shapes (string): sets which channel shapes the user can select: 'open', 'closed', 'both'
            default_shape (bool): which shape should be the default shape
            hide_depth (bool): Is this class being used for tailwater calculations?
            app_data (AppData): application data (settings, etc.).
            model_name (string): name of the model
            project_uuid (string): project uuid
        """
        super().__init__(standalone_calc=False, allowed_shapes=allowed_shapes,
                         hide_depth=hide_depth,
                         app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'Tailwater CalcData'
        self.type = 'TailwaterCalc'

        self.calculator = TailwaterCalc()

        # Input
        max_value = self.max_value
        self.tw_type_list = copy.copy(self.input['Geometry'].get_val().open_channel_shapes)
        self.tw_type_list.append('rating curve')
        self.tw_type_list.append('constant tailwater elevation')
        self.input['Tailwater type'] = Variable('Tailwater Channel type', "list", 0, self.tw_type_list)
        self.input['Tailwater type'].set_val(default_shape)

        self.input['Channel invert elevation'] = Variable(
            'Channel invert elevation', 'float', 0.0, [], limits=(-max_value, max_value), precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.input['Constant tailwater elevation'] = Variable(
            'Constant tailwater elevation', 'float', 0.0, [], limits=(-max_value, max_value), precision=2,
            unit_type=['length'], native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.flow_var = Variable(
            'Tailwater flows', 'float_list', 0.0, [], precision=2, unit_type=['flow'], native_unit='cfs',
            us_units=self.us_flow, si_units=self.si_flow, limits=(0.0, max_value))

        self.depths_var = Variable(
            'Tailwater depths', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.elevation_var = Variable(
            'Tailwater elevations', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))

        self.velocity_var = Variable(
            'Tailwater velocities', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft/s',
            us_units=[['mph', 'yd/s', 'ft/s', 'in/s']], si_units=[['km/hr', 'm/s', 'mm/s']])

        name = 'Rating curve'
        tw_input = {
            'Flow': self.flow_var,
            'Elevation': self.elevation_var,
            'Depth': self.depths_var,
            'Velocity': self.velocity_var
        }

        self.input['Rating curve'] = Variable(name, 'table', RatingCurveTable(
            self.theme, name=name, plot_names=[name], input=tw_input, min_items=2, app_data=app_data,
            model_name=model_name, project_uuid=project_uuid))

        # self.input['Rating flows'] = Variable(
        #     'Flow(s)', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
        #     us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(0.0, max_value))
        # self.input['Rating elevations'] = Variable(
        #     'Tailwater elevations', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
        #     us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        # self.input['Rating depths'] = Variable(
        #     'Tailwater depths', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
        #     us_units=self.us_mid_length, si_units=self.si_mid_length)
        # self.input['Rating velocities'] = Variable(
        #     'Tailwater velocities', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft/s',
        #     us_units=[['mph', 'yd/s', 'ft/s', 'in/s']], si_units=[['km/hr', 'm/s', 'mm/s']])

        self.warnings = []

        # Intermediate
        self.compute_prep_functions.extend([])
        # self.compute_finalize_functions.extend([])
        self.compute_finalize_functions = []
        self.intermediate_to_copy.extend([
            'tw_type_list'
        ])

        # self.results = {}  Use base class result dict

        self.results['Tailwater elevations'] = Variable(
            'Tailwater elevations', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length, limits=(-max_value, max_value))
        self.results['Tailwater depths'] = Variable(
            'Tailwater depths', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)
        self.results['Tailwater velocities'] = Variable(
            'Tailwater velocities', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft/s',
            us_units=[['mph', 'yd/s', 'ft/s', 'in/s']], si_units=[['km/hr', 'm/s', 'mm/s']])

        self.hw = []

    def get_input_group(self, unknown=None):
        """Get the input group for the input table.

        Args:
            unknown (string): unknown variable that we are computing
        """
        input_vars = {}
        input_vars['Tailwater type'] = self.input['Tailwater type']
        channel_type = self.input['Tailwater type'].get_val()

        if channel_type not in ['rating curve', 'constant tailwater elevation']:
            self.input['Geometry'].value.input['Shape'].set_val(channel_type)

        input_vars['Channel invert elevation'] = self.input[
            'Channel invert elevation']

        if channel_type == 'rating curve':
            input_vars['Rating curve'] = self.input['Rating curve']
            # if self.input['Head'].get_val() == 'Elevations':
            #     input_vars['Rating elevations'] = self.input[
            #         'Rating elevations']
            # else:
            #     input_vars['Rating depths'] = self.input['Rating depths']
            # input_vars['Rating velocities'] = self.input['Rating velocities']
        elif channel_type not in ['rating curve', 'constant tailwater elevation']:
            channel_input = super().get_input_group()
            for input in channel_input:
                if input not in ['Shape', 'Calculate', 'Head', 'Flows', 'Channel depth']:
                    input_vars[input] = channel_input[input]

        if channel_type == 'constant tailwater elevation':
            input_vars['Constant tailwater elevation'] = self.input[
                'Constant tailwater elevation']

        return input_vars
