"""CompositeManningN Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator import Calculator

# 4. Local modules


class CompositeManningNCalc(Calculator):
    """Provides a class that will define the site data of a culvert barrel."""

    def build_composite_n_list(self):
        """Build composite n list."""
        self.composite_n_method = []
        if self.open_shape:
            self.composite_n_method.append("Lotter Method")
        self.composite_n_method.append("Horton and Einstein Method")
        self.composite_n_method.append(
            "Pavlovskii, Mühlhofer, Einstein and Banks Method")

    def build_n_entry_list(self):
        """Build n entry list."""
        # build list for the ways to specify manning's n value
        self.n_entry_list = ['specify composite n value', 'specify n by geometry segment']
        if self.no_station_overlap:
            self.n_entry_list.append('specify n by station')
        if self.embedment_present:
            self.n_entry_list.append('specify channel and embedment n values')
        if self.gradation_present:
            self.n_entry_list.append('compute n values for gradation size')
        if not self.embedment_present:
            self.n_entry_list.append('specify material')  # HY-8 style
            found, complexity = self.get_data('Complexity', 0)
            if found and complexity >= 1:
                self.n_entry_list.append('specify material and condition by category')  # HY12 style

    def build_material_dict(self):
        """Build material dict."""
        self.material_dict = {}
        if self.shape == 'circle':
            self.material_dict['concrete (0.012)'] = 0.012
            self.material_dict['corrugated steel (0.024)'] = 0.024
            self.material_dict['corrugated aluminum (0.031)'] = 0.031
            self.material_dict['PVC (0.011)'] = 0.011
            self.material_dict['smooth HDPE (0.012)'] = 0.012
            self.material_dict['corrugated PE (0.024)'] = 0.024
        if self.shape == 'elliptical':
            self.material_dict['steel or aluminum (0.034)'] = 0.034
            self.material_dict['concrete (0.012)'] = 0.012
        if self.shape == 'pipe arch':
            self.material_dict['steel or aluminum (0.025)'] = 0.025
            self.material_dict['steel structural plate (0.035)'] = 0.035
            self.material_dict['aluminum structural plate (0.035)'] = 0.035
            self.material_dict['concrete (0.012)'] = 0.012
        if self.shape == 'box':
            self.material_dict['concrete (0.012)'] = 0.012
        else:
            self.material_dict['concrete (0.012)'] = 0.012
            self.material_dict['corrugated metal riveted or welded (0.035)'] = 0.035

    def build_material_categories(self):
        """Build material categories."""
        # self.material_category = ['channel lining', 'storm drain conduits', 'street and pavement gutters']
        # - Never changes
        self.material_name = []
        self.material_corrugation = []
        self.material_condition = {}
        if self.input_dict['calc_data']['Material category'] == 'channel lining':
            self.material_name = [
                'concrete', 'grouted riprap', 'stone masonry', 'soil element',
                'asphalt', 'bare soil', 'rock cut', 'open-weave textile',
                'erosion control blanket', 'turf reinforcement mat'
            ]
            if self.input_dict['calc_data']['Material name'] == 'concrete':
                self.material_condition['maximum (0.015)'] = 0.015
                self.material_condition['typical (0.013)'] = 0.013
                self.material_condition['minimum (0.011)'] = 0.011
            elif self.input_dict['calc_data']['Material name'] == 'grouted riprap':
                self.material_condition['maximum (0.04)'] = 0.04
                self.material_condition['typical (0.03)'] = 0.03
                self.material_condition['minimum (0.028)'] = 0.028
            elif self.input_dict['calc_data']['Material name'] == 'stone masonry':
                self.material_condition['maximum (0.042)'] = 0.042
                self.material_condition['typical (0.032)'] = 0.032
                self.material_condition['minimum (0.03)'] = 0.03
            elif self.input_dict['calc_data']['Material name'] == 'soil element':
                self.material_condition['maximum (0.025)'] = 0.025
                self.material_condition['typical (0.022)'] = 0.022
                self.material_condition['minimum (0.02)'] = 0.02
            elif self.input_dict['calc_data']['Material name'] == 'asphalt':
                self.material_condition['maximum (0.018)'] = 0.018
                self.material_condition['typical (0.016)'] = 0.016
                self.material_condition['minimum (0.016)'] = 0.016
            elif self.input_dict['calc_data']['Material name'] == 'bare soil':
                self.material_condition['maximum (0.025)'] = 0.025
                self.material_condition['typical (0.02)'] = 0.02
                self.material_condition['minimum (0.016)'] = 0.016
            elif self.input_dict['calc_data']['Material name'] == 'rock cut':
                self.material_condition['maximum (0.045)'] = 0.045
                self.material_condition['typical (0.035)'] = 0.035
                self.material_condition['minimum (0.016)'] = 0.016
            elif self.input_dict['calc_data']['Material name'] == 'open-weave textile':
                self.material_condition['maximum (0.028)'] = 0.028
                self.material_condition['typical (0.025)'] = 0.025
                self.material_condition['minimum (0.022)'] = 0.022
            elif self.input_dict['calc_data']['Material name'] == 'erosion control blanket':
                self.material_condition['maximum (0.045)'] = 0.045
                self.material_condition['typical (0.035)'] = 0.035
                self.material_condition['minimum (0.028)'] = 0.028
            elif self.input_dict['calc_data']['Material name'] == 'turf reinforcement mat':
                self.material_condition['maximum (0.036)'] = 0.036
                self.material_condition['typical (0.03)'] = 0.03
                self.material_condition['minimum (0.024)'] = 0.024
        elif self.input_dict['calc_data']['Material category'] == 'storm drain conduits':
            self.material_name = [
                'concrete pipe', 'concrete boxes', 'Spiral Rip Metal Pipe',
                'corrugated metal pipe', 'pipe-arch and box',
                'corrugated polyethylene', 'polyvinyl chloride (PVC)'
            ]
            if self.input_dict['calc_data']['Material name'] == 'concrete pipe':
                self.material_condition['good condition (0.01)'] = 0.01
                self.material_condition['poor condition (0.011)'] = 0.011
            elif self.input_dict['calc_data']['Material name'] == 'concrete boxes':
                self.material_condition['good condition (0.012)'] = 0.012
                self.material_condition['average condition (0.0135)'] = 0.0135
                self.material_condition['poor condition (0.015)'] = 0.015
            elif self.input_dict['calc_data']['Material name'] == 'Spiral Rip Metal Pipe':
                self.material_condition['good condition (0.012)'] = 0.012
                self.material_condition['poor condition (0.013)'] = 0.013
            elif self.input_dict['calc_data']['Material name'] == 'corrugated metal pipe, pipe-arch and box':
                self.material_corrugation = [
                    '2-2/3 X 1/2 annular', '2-2/3 X 1/2 helical',
                    '6 X 1 helical', '6 X 1 helical', '5 X 1 helical',
                    '6 X 2 structural plate', '9 X 2-1/2 structural plate'
                ]
                if self.input_dict['calc_data']['Material corrugation'] == '2-2/3 X 1/2 annular':
                    self.material_condition['good condition (0.022)'] = 0.022
                    self.material_condition['average condition (0.0245)'] = 0.0245
                    self.material_condition['poor condition (0.027)'] = 0.027
                elif self.input_dict['calc_data']['Material corrugation'] == '2-2/3 X 1/2 helical':
                    self.material_condition['good condition (0.011)'] = 0.011
                    self.material_condition['average condition (0.017)'] = 0.017
                    self.material_condition['poor condition (0.023)'] = 0.023
                elif self.input_dict['calc_data']['Material corrugation'] == '6 X 1 helical':
                    self.material_condition['good condition (0.022)'] = 0.022
                    self.material_condition['average condition (0.0235)'] = 0.0235
                    self.material_condition['poor condition (0.025)'] = 0.025
                elif self.input_dict['calc_data']['Material corrugation'] == '5 X 1 helical':
                    self.material_condition['good condition (0.025)'] = 0.025
                    self.material_condition['poor condition (0.026)'] = 0.026
                elif self.input_dict['calc_data']['Material corrugation'] == '6 X 2 structural plate':
                    self.material_condition['good condition (0.033)'] = 0.033
                    self.material_condition['average condition (0.034)'] = 0.034
                    self.material_condition['poor condition (0.035)'] = 0.035
                elif self.input_dict['calc_data']['Material corrugation'] == '9 X 2-1/2 structural plate':
                    self.material_condition['good condition (0.033)'] = 0.033
                    self.material_condition['average condition (0.035)'] = 0.035
                    self.material_condition['poor condition (0.037)'] = 0.037
            elif self.input_dict['calc_data']['Material name'] == 'corrugated polyethylene':
                self.material_corrugation = ['smooth', 'corrugated']
                if self.input_dict['calc_data']['Material corrugation'] == 'smooth':
                    self.material_condition['good condition (0.009)'] = 0.009
                    self.material_condition['average condition (0.012)'] = 0.012
                    self.material_condition['poor condition (0.015)'] = 0.015
                elif self.input_dict['calc_data']['Material corrugation'] == 'corrugated':
                    self.material_condition['good condition (0.018)'] = 0.018
                    self.material_condition['average condition (0.0215)'] = 0.0215
                    self.material_condition['poor condition (0.025)'] = 0.025
            elif self.input_dict['calc_data']['Material name'] == 'polyvinyl chloride (PVC) (smooth)':
                self.material_condition['good condition (0.009)'] = 0.009
                self.material_condition['average condition (0.01)'] = 0.01
                self.material_condition['poor condition (0.011)'] = 0.011
        elif self.input_dict['calc_data']['Material category'] == 'street and pavement gutters':
            self.material_name = [
                'concrete gutter (troweled finish)', 'asphalt pavement gutter',
                'concrete gutter-asphalt pavement', 'concrete pavement'
            ]
            if self.input_dict['calc_data']['Material name'] == 'concrete gutter (troweled finish)':
                self.material_condition['sloped (0.012)'] = 0.012
                self.material_condition['small slope causing sediment deposition (0.014)'] = 0.014
            elif self.input_dict['calc_data']['Material name'] == 'asphalt pavement gutter':
                self.material_condition['smooth texture, sloped (0.013)'] = 0.013
                self.material_condition['smooth texture, small slope causing sediment deposition (0.015)'] = 0.015
                self.material_condition['rough texture, sloped (0.016)'] = 0.016
                self.material_condition['rough texture, small slope causing sediment deposition (0.018)'] = 0.018
            elif self.input_dict['calc_data']['Material name'] == 'concrete gutter-asphalt pavement':
                self.material_condition['smooth texture, sloped (0.013)'] = 0.013
                self.material_condition['smooth texture, small slope causing sediment deposition (0.015)'] = 0.015
                self.material_condition['rough texture, sloped (0.015)'] = 0.015
                self.material_condition['rough texture, small slope causing sediment deposition (0.017)'] = 0.017
            elif self.input_dict['calc_data']['Material name'] == 'concrete pavement':
                self.material_condition['float finish, sloped (0.014)'] = 0.014
                self.material_condition['float finish, small slope causing sediment deposition (0.016)'] = 0.016
                self.material_condition['broom finish, sloped (0.013)'] = 0.013
                self.material_condition['broom finish, small slope causing sediment deposition (0.018)'] = 0.018

    def update_lists(self, no_station_overlap=None, embedment_present=None, gradation_present=None):
        """Update the lists and variables for current selections."""
        if no_station_overlap is not None:
            self.no_station_overlap = no_station_overlap
        if embedment_present is not None:
            self.embedment_present = embedment_present
        if gradation_present is not None:
            self.gradation_present = gradation_present
        self.build_n_entry_list()
        self.build_composite_n_list()
        self.build_material_categories()
        self.build_material_dict()

        # TODO: Add setting these lists in the data class.
        # self.input['n entry'].set_list(self.n_entry_list)

        # self.input['n method'].set_list(self.composite_n_method)

        # self.input['Material'].set_list(list(self.material_dict.keys()))

        # material_condition_list = list(self.material_condition.keys())
        # self.input['Material category'].set_list(self.material_category)
        # self.input['Material name'].set_list(self.material_name)
        # self.input['Material corrugation'].set_list(self.material_corrugation)
        # self.input['Material condition'].set_list(material_condition_list)

        # # Verify that the selections still work
        # size = len(self.material_name) - 1
        # if self.input['Material name'].value > size >= 0:
        #     self.input['Material name'].set_val(self.material_name[0])
        # elif size < 0 < self.input['Material name'].value:
        #     self.input['Material name'].value = 0
        # size = len(self.material_corrugation) - 1
        # if self.input['Material corrugation'].value > size >= 0:
        #     self.input['Material corrugation'].set_val(
        #         self.material_corrugation[0])
        # elif size < 0 < self.input['Material corrugation'].value:
        #     self.input['Material corrugation'].value = 0
        # size = len(material_condition_list) - 1
        # if self.input['Material condition'].value > size >= 0:
        #     self.input['Material condition'].set_val(
        #         material_condition_list[0])
        # elif size < 0 < self.input['Material condition'].value:
        #     self.input['Material condition'].value = 0

    def _get_can_compute(self, unknown=None):
        """Determine whether we have enough data to compute.

        Returns:
            True, if we can compute; otherwise, False
            message (list of str): warning message if we fail
        """
        if unknown is None:
            unknown = self.unknown
        else:
            self.unknown = unknown

        self.update_lists()

        if self.input_dict['calc_data']['n entry'] == 'specify composite n value' and unknown != 'Composite n':
            if self.input_dict['calc_data']['Composite n'] > 0.0:
                return True
            else:
                self.warnings['composite_n'] = "Please enter a composite Manning's n value."
                return False

        if self.input_dict['calc_data']['n entry'] in ['specify n by geometry segment']:
            found = False
            # station = self.input_dict['calc_data']["Manning's n by station"]['Station']
            # elevation = self.input_dict['calc_data']["Manning's n by station"]['Elevation']
            size = len(self.input_dict['calc_data']["Manning's n by station"]['calculator'].cs_stations)
            mannings_n_values = self.input_dict['calc_data']["Manning's n by station"]['calculator'].cs_mannings_n
            index = 0
            for n in mannings_n_values:
                if n <= 0.0 and index < size - 1:
                    found = True
                    break
                index += 1
            if found:
                self.warnings['mannings_n_by_segment'] = \
                    "Please enter all Manning's n values for the geometry segments."
                return False
        if self.input_dict['calc_data']['n entry'] in ['specify n by station']:
            found = False
            station = self.input_dict['calc_data']["Manning's n by station"]['Station']
            # elevation = self.input_dict['calc_data']["Manning's n by station"]['Elevation']
            # mannings_n_values = self.input_dict['calc_data']["Manning's n by station"]["Manning's n"]
            n_stations = self.input_dict['calc_data']["Manning's n by station"]["Manning's n stations"]
            mannings_n_values = self.input_dict['calc_data']["Manning's n by station"]["Manning's n coefficients"]
            size = len(self.input_dict['calc_data']["Manning's n by station"]['calculator'].cs_stations)
            index = 0
            for n in mannings_n_values:
                if n <= 0.0 and index < size - 1:
                    found = True
                    break
                index += 1
            if found:
                self.warnings['mannings_n_by_segment'] = \
                    "Please enter all Manning's n values for the Manning's n stations."
                return False
            if min(station) < min(n_stations) or max(station) > max(n_stations):
                self.warnings['mannings_n_by_segment'] = \
                    "Please ensure Manning's n stations cover the entire cross-section."
                return False

        n_entry = ''
        if self.input_dict['calc_data']['n entry'] in ['specify channel and embedment n values']:
            n_entry = self.input_dict['calc_data']['Channel n entry']
            if n_entry == 'specify n value':
                if self.input_dict['calc_data']['Channel n'] <= 0.0:
                    self.warnings['channel_n'] = "Please enter a manning's n value for the channel."
                    _, messages = self.check_embedment_input()
                    self.warnings.update(messages)
                    return False
                result, self.warnings = self.check_embedment_input()
                return result

        if self.input_dict['calc_data']['n entry'] in ['specify material'] or n_entry in ['specify material']:
            result, self.warnings = self.check_embedment_input()
            return result

        if self.input_dict['calc_data']['n entry'] in ['specify material and condition by category'] or \
                n_entry in ['specify material and condition by category']:
            result, self.warnings = self.check_embedment_input()
            return result

        return True

    def check_embedment_input(self):
        """Check the embedment input for errors.

        Returns:
            bool: True if we have embedment data; otherwise, False
        """
        _, zero_tol = self.get_data('Zero tolerance')
        if self.embedment_present:
            check_embedment_n = True
            if self.gradation_present:
                if self.input_dict['calc_data']['Embedment n entry'] == 'specify gradation equation':
                    check_embedment_n = False
                    if self.input_dict['calc_data']['Gradation method'] == 'Limerinos':
                        if self.d84 <= zero_tol:
                            return False, {'Embedment n': 'Please complete the gradation data.'}
                    if self.input_dict['calc_data']['Gradation method'] == 'Blodgett':
                        if self.d50 <= zero_tol:
                            return False, {'Embedment n': 'Please complete the gradation data.'}
                    if self.input_dict['calc_data']['Gradation method'] == 'Jarrett':
                        if self.d84 <= zero_tol:
                            return False, {'Embedment n': 'Please complete the gradation data.'}
                    if self.input_dict['calc_data']['Gradation method'] == 'Mussetter':
                        if self.d84 <= zero_tol or self.d50 <= zero_tol:
                            return False, {'Embedment n': 'Please complete the gradation data.'}
            if check_embedment_n:
                if self.input_dict['calc_data']['Embedment n'] <= 0.0:
                    return False, {'Embedment n': "Please enter a Manning's n value for the embedment."}
        return True, {}

    # def get_val(self):
    #     """Computes and returns the results.

    #     Returns:
    #         self.results['result'] (variable): result of the computations.
    #     """
    #     self.compute_data()
    #     return self.results['result']

    # def add_embedment_input(self, input_vars):
    #     """Add the embedment input to the input_vars dictionary.

    #     Args:
    #         input_vars (dict): dictionary of input variables
    #     """
    #     if self.embedment_present:
    #         add_embedment_n = True
    #         if self.gradation_present:
    #             input_vars['Embedment n entry'] = self.input_dict['calc_data']['Embedment n entry']
    #             if self.input_dict['calc_data']['Embedment n entry'] == 'specify gradation equation':
    #                 input_vars['Gradation method'] = self.input_dict['calc_data']['Gradation method']
    #                 add_embedment_n = False
    #         if add_embedment_n:
    #             input_vars['Embedment n'] = self.input_dict['calc_data']['Embedment n']

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        computed_n = self.get_channel_n()
        if 'Composite n value' not in self.results:
            self.results['Composite n value'] = []
        self.results['Channel n'] = computed_n
        self.results['Composite n value'].append(computed_n)
        if computed_n == self.null_value:
            return False
        return True

    def compute_gradation_n(self):
        """Computes the Manning's n value using the gradation method."""
        if self.input_dict['calc_data']['Gradation method'] == 'Limerinos':
            return self.compute_limerinos_n()
        elif self.input_dict['calc_data']['Gradation method'] == 'Blodgett':
            return self.compute_blodgett_n()
        elif self.input_dict['calc_data']['Gradation method'] == 'Jarrett':
            return self.compute_jarrett_n()
        elif self.input_dict['calc_data']['Gradation method'] == 'Mussetter':
            return self.compute_mussetter_n()

    def compute_limerinos_n(self):
        """Computes the Manning's n value using the Limerinos equation."""
        # HEC 26
        # Page C-1
        _, null_data = self.get_data('Null data')
        self.limerinos_n_applicable = False
        self.n_applicable = False
        self.limerinos_n = null_data
        if self.d84 <= 0.0:
            self.warnings['d84'] = 'Please enter the D84 gradation to complete the Limerinos equation.'
            return self.seed_value, False
        if self.hyd_radius <= 0.0:
            return self.seed_value, False
        rh_over_d84 = self.hyd_radius / self.d84
        alpha = 0.0926
        one_sixth = 1.0 / 6.0

        if 0.9 <= rh_over_d84 <= 68.5:
            self.limerinos_n_applicable = True
            self.n_applicable = True
        else:
            self.limerinos_n_applicable = False
            # self.n_applicable = True
            self.warnings['limerinos'] = 'This channel does match the applicable range for the Limerinos equation.'
            self.warnings['limerinos_range'] = f'The hydraulic radius over D84 is {rh_over_d84} and needs to be '
            self.warnings['limerinos_range'] += 'between 0.9 and 68.5'

        self.limerinos_n = (alpha * self.hyd_radius ** one_sixth) / (1.16 + 2.0 * math.log10(rh_over_d84))

        return self.limerinos_n, True

    def compute_blodgett_n(self):
        """Computes the Manning's n value using the Blodgett equation."""
        # HEC 26
        # Page C-2
        _, null_data = self.get_data('Null data')
        self.blodgett_n_applicable = False
        self.n_applicable = False
        self.blodgett_n = null_data
        if self.d50 <= 0.0:
            self.warnings['d50'] = 'Please enter the D50 gradation to complete the Blodgett equation.'
            return self.seed_value, False
        if self.average_flow_depth <= 0.0:
            return self.seed_value, False
        ya_over_d50 = self.average_flow_depth / self.d50
        alpha = 0.262
        one_sixth = 1.0 / 6.0

        if 1.5 <= ya_over_d50 <= 185:
            self.blodgett_n_applicable = True
            self.n_applicable = True
        else:
            self.warnings['blodgett'] = 'This channel does match the applicable range for the Blodgett equation.'
            self.warnings['blodgett_range'] = f'The depth over Dd50 is {ya_over_d50} and needs to be between 1.5 '
            self.warnings['blodgett_range'] += 'and 185'

        self.blodgett_n = (alpha * self.average_flow_depth ** one_sixth) / (2.25 + 5.23 * math.log10(ya_over_d50))

        return self.blodgett_n, True

    def compute_jarrett_n(self):
        """Computes the Manning's n value using the Jarrett equation."""
        # HEC 26
        # Page C-1
        _, null_data = self.get_data('Null data')
        self.jarrett_n_applicable = False
        self.n_applicable = False
        self.jarrett_n = null_data
        if self.hyd_radius <= 0.0:
            return self.seed_value, False
        alpha = 0.39
        if self.d84 > 0.0:
            rh_over_d84 = self.hyd_radius / self.d84
        else:
            self.warnings['d84'] = 'Please enter the D84 gradation to complete the Jarrett equation.'
            return self.seed_value, False

        first_test = False
        second_test = False
        third_test = False
        if 0.002 < self.friction_slope < 0.04:
            first_test = True
        else:
            self.warnings['friction'] = 'This channel does match the applicable range for the Jarrett equation.'
            self.warnings['friction_range'] = f'The friction slope is {self.friction_slope} and needs to be between '
            self.warnings['friction_range'] += '0.002 and 0.04'

        if 0.5 < self.hyd_radius < 7.0:
            second_test = True
        else:
            self.warnings['hyd_radius'] = 'This channel does match the applicable range for the Jarrett equation.'
            self.warnings['hyd_radius_range'] = f'The hydraulic radius is {self.hyd_radius} and needs to be between '
            self.warnings['hyd_radius_range'] += '0.5 and 7.0'

        if 0.4 < rh_over_d84 < 11:
            third_test = True
        else:
            self.warnings['rh_over_d84'] = 'This channel does match the applicable range for the Jarrett equation.'
            self.warnings['rh_over_d84_range'] = f'The hydraulic radius over D84 is {rh_over_d84} and needs to be '
            self.warnings['rh_over_d84_range'] += 'between 0.4 and 11.0'

        if first_test and second_test and third_test:
            self.jarrett_n_applicable = True
            self.n_applicable = True

        self.jarrett_n = alpha * self.friction_slope**0.38 * self.hyd_radius**-0.16

        return self.jarrett_n, True

    def compute_mussetter_n(self):
        """Computes the Manning's n value using the Mussetter equation."""
        # HEC 26
        # Page C-2
        _, null_data = self.get_data('Null data')
        self.mussetter_n_applicable = False
        self.n_applicable = False
        self.mussetter_n = null_data
        if self.d50 <= 0.0:
            self.warnings['d50'] = 'Please enter the D50 gradation to complete the Mussetter equation.'
            return self.seed_value, False
        if self.d84 <= 0.0:
            self.warnings['d84'] = 'Please enter the D84 gradation to complete the Mussetter equation.'
            return self.seed_value, False
        if self.average_flow_depth <= 0.0:
            return self.seed_value, False
        rh_over_d84 = self.hyd_radius / self.d84
        ya_over_d84 = self.average_flow_depth / self.d84
        d84_over_d50 = self.d84 / self.d50
        alpha = 0.24

        if 0.25 < rh_over_d84 < 3.72:
            self.mussetter_n_applicable = True
            self.n_applicable = True
        else:
            self.warnings['rh_over_d84'] = 'This channel does match the applicable range for the Mussetter equation.'
            self.warnings['rh_over_d84_range'] = f'The hydraulic radius over D84 is {rh_over_d84} '
            self.warnings['rh_over_d84_range'] += 'and needs to be between 0.25 and 3.72'

        self.mussetter_n = alpha * ya_over_d84**-0.46 * d84_over_d50**0.85 * self.friction_slope**0.39

        return self.mussetter_n, True

    def get_channel_n(self):
        """Returns the channel n value based on the selections."""
        self.null_value = self.get_data('Null data')[1]
        specify_n = False
        specify_material = False
        specify_mat_and_category = False
        if self.input_dict['calc_data']['n entry'] == 'specify composite n value':
            return self.input_dict['calc_data']['Composite n']
        if self.input_dict['calc_data']['n entry'] in ['specify channel and embedment n values']:
            if self.input_dict['calc_data']['Channel n entry'] == 'specify n value':
                specify_n = True
            elif self.input_dict['calc_data']['Channel n entry'] == 'specify material':
                specify_material = True
            elif self.input_dict['calc_data']['Channel n entry'] == 'specify material and condition by category':
                specify_mat_and_category = True
        if self.input_dict['calc_data']['n entry'] == 'specify n value' or specify_n:
            return self.input_dict['calc_data']['Channel n']
        elif self.input_dict['calc_data']['n entry'] == 'specify material' or specify_material:
            return self.material_dict[self.input_dict['calc_data']['Material']]
        elif self.input_dict['calc_data']['n entry'] == 'specify material and condition by category' or \
                specify_mat_and_category:
            return self.material_condition[self.input_dict['calc_data']['Material condition']]
        elif self.input_dict['calc_data']['n entry'] == 'compute n values for gradation size':
            return self.compute_gradation_n()
        elif self.input_dict['calc_data']['n entry'] in ['specify n by geometry segment']:
            self.mannings_n_long = self.input_dict['calc_data']["Manning's n by station"][
                'calculator'].cs_mannings_n
        elif self.input_dict['calc_data']['n entry'] in ['specify n by station']:
            self.mannings_n_long = self.input_dict['calc_data']["Manning's n by station"]['calculator'].mannings_n_long
        return self.null_value

    def get_embedment_n(self):
        """Returns the embedment n value based on the selections."""
        if self.input_dict['calc_data']['Embedment n entry'] == 'specify n value':
            return self.input_dict['calc_data']['Embedment n']
        else:  # 'specify gradation equation'
            # When result == False, the n is a 0.03 seed value that we use to start the calculation with
            n, result = self.compute_gradation_n()
            # if not self.n_applicable and n < 0.0:
            if n < 0.0:
                n = abs(n)  # Try it as a new seed
            return n

    def get_intermediate_composite_n(self):
        """Returns the composite n value based on the selections.

        Returns:
            n (float): Manning's n value
        """
        if self.input_dict['calc_data']['n entry'] == 'specify composite n value':
            # TODO: Need to say can't compute if specified n = 0.0; perhaps log an error here
            return self.input_dict['calc_data']['Composite n']
        elif self.input_dict['calc_data']['n entry'] in ['specify n by geometry segment', 'specify n by station',
                                                         'specify channel and embedment n values']:
            if self.input_dict['calc_data']['n method'] == "Lotter Method":
                return self.lotter_manning_value
            if self.input_dict['calc_data']['n method'] == "Horton and Einstein Method":
                return self.horton_manning_value
            if self.input_dict['calc_data']['n method'] == "Pavlovskii, Mühlhofer, Einstein and Banks Method":
                return self.pavlovskii_manning_value
        elif self.embedment_present and self.input_dict['calc_data']['n entry'] == \
                'compute n values for gradation size':
            n, result = self.compute_gradation_n()
            _, zero_tol = self.get_data('Zero tolerance')
            if n <= zero_tol:
                n = self.seed_value
            return n
        # HY-8 style, HY12 style
        elif self.input_dict['calc_data']['n entry'] in ['specify material',
                                                         'specify material and condition by category']:
            return self.get_channel_n()

        _, null_data = self.get_data('Null data')
        return null_data

    def get_intermediate_n_and_applicable(self):
        """Returns the composite n value based on the selections.

        Returns:
            n (float): Manning's n value
        """
        self.gradation_calculation = False
        self.warnings = {}
        self.seed_value = self.input_dict['calc_data']['Invalid n value']
        n = self.get_intermediate_composite_n()
        if self.embedment_present and self.input_dict['calc_data']['Embedment n entry'] == 'specify gradation equation':
            self.gradation_calculation = True
        return n, True

    def get_composite_n(self):
        """Returns the composite n value based on the selections.

        Returns:
            n (float): Manning's n value
        """
        if 'Composite n value' not in self.results:
            self.results['Composite n value'] = []
        return self.results['Composite n value']

    def finalize_last_result(self):
        """Adds the last computed values to the results."""
        self.final_horton_manning_value = self.horton_manning_value
        self.final_pavlovskii_manning_value = self.pavlovskii_manning_value
        self.final_lotter_manning_value = self.lotter_manning_value

        if 'Composite n computation method' not in self.results:
            self.results['Composite n computation method'] = []
        if 'Composite n value' not in self.results:
            self.results['Composite n value'] = []

        if self.input_dict['calc_data']['n entry'] == 'specify composite n value':
            self.results['Composite n computation method'].append('specified composite n value')
            self.results['Composite n value'].append(self.input_dict['calc_data']['Composite n'])
        elif self.input_dict['calc_data']['n entry'] in ['specify n by geometry segment', 'specify n by station',
                                                         'specify channel and embedment n values']:
            if self.input_dict['calc_data']['n method'] == "Horton and Einstein Method":
                self.results['Composite n computation method'].append('Horton and Einstein Method')
                self.results['Composite n value'].append(self.horton_manning_value)
            elif self.input_dict['calc_data']['n method'] == "Pavlovskii, Mühlhofer, Einstein and Banks Method":
                self.results['Composite n computation method'].append(
                    'Pavlovskii, Mühlhofer, Einstein and Banks Method')
                self.results['Composite n value'].append(self.pavlovskii_manning_value)
            elif self.input_dict['calc_data']['n method'] == "Lotter Method":
                self.results['Composite n computation method'].append('Lotter Method')
                self.results['Composite n value'].append(self.lotter_manning_value)
        elif self.input_dict['calc_data']['n entry'] == 'compute n values for gradation size':
            self.results['Composite n computation method'].append(
                f'{self.input_dict["calc_data"]["Gradation method"]} for gradation size')
            self.results['Composite n value'].append(self.compute_gradation_n()[0])

        # HY-8 style, HY12 style
        elif self.input_dict['calc_data']['n entry'] in ['specify material',
                                                         'specify material and condition by category']:
            self.results['Composite n computation method'].append('specified by material')
            self.results['Composite n value'].append(self.get_channel_n())
