"""CompositeManningN Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calcdata import VariableGroup
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.manning_n.composite_manning_n_calc import CompositeManningNCalc


class CompositeManningNData(VariableGroup):
    """Provides a class that will define the site data of a culvert barrel."""

    def __init__(self, shape, open_shape, embedment_present, gradation_present,
                 total_embedment, stations, elevations, app_data=None, model_name=None,
                 project_uuid=None):
        """Initializes the Composite n Class.

        Args:
            shape (str): The shape of the channel.
            open_shape (bool): True if the culvert is open.
            embedment_present (bool): True if embedment is present.
            gradation_present (bool): True if gradation is present.
            total_embedment (float): The total embedment.
            stations (list): The stations.
            elevations (list): The elevations.
            app_data (AppData): The application data (settings).
            model_name (str): The name of the model.
            project_uuid (str): The project uuid.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = 'Composite Manning n CalcData'
        self.type = 'CompositeManningN'

        self.calculator = CompositeManningNCalc()

        # Input
        self.shape = shape
        self.open_shape = open_shape
        self.embedment_present = embedment_present
        self.gradation_present = gradation_present
        self.total_embedment = total_embedment
        self.stations = stations
        self.elevations = elevations
        self.no_station_overlap = False
        stations_never_decreases = True
        stations_never_increases = True
        for index in range(1, len(stations)):
            if stations[index - 1] < stations[index]:
                stations_never_decreases = False
            if stations[index - 1] > stations[index]:
                stations_never_increases = False
        if stations_never_decreases or stations_never_increases:
            self.no_station_overlap = True

        self.n_entry_list = []
        self.build_n_entry_list()

        self.n_channel_entry_list = [
            'specify n value', 'specify material',
            'specify material and condition by category'
        ]
        self.n_embedment_entry_list = [
            'specify n value', 'specify gradation equation'
        ]

        # Composite Manning's n method
        self.composite_n_method = []
        self.build_composite_n_list()

        gradation_equations_list = [
            'Limerinos', 'Blodgett', 'Jarrett', 'Mussetter'
        ]

        self.material_dict = {}
        self.build_material_dict()

        self.material_category = [
            'channel lining', 'storm drain conduits',
            'street and pavement gutters'
        ]
        self.material_name = ['concrete']
        self.material_corrugation = ['2-2/3 X 1/2 annular']
        self.material_condition = {'typical (0.013)': 0.013}
        self.material_condition_list = []

        self.input['n entry'] = Variable('n value entry', "list", 0,
                                         self.n_entry_list)

        self.input['n method'] = Variable('composite n method', "list", 0,
                                          self.composite_n_method)

        self.input['Channel n entry'] = Variable('Channel n entry', "list", 0,
                                                 self.n_channel_entry_list)
        self.input['Embedment n entry'] = Variable('Embedment n entry', "list",
                                                   0,
                                                   self.n_embedment_entry_list)

        self.input['Composite n'] = Variable('Composite n',
                                             'float',
                                             0.0, [],
                                             limits=[0.0, 1.0],
                                             precision=4,
                                             unit_type=['coefficient'],
                                             native_unit='coefficient',
                                             us_units=[['coefficient']],
                                             si_units=[['coefficient']])

        self.input['Channel n'] = Variable('Channel n',
                                           'float',
                                           0.0, [],
                                           limits=[0.0, 1.0],
                                           precision=4,
                                           unit_type=['coefficient'],
                                           native_unit='coefficient',
                                           us_units=[['coefficient']],
                                           si_units=[['coefficient']])

        self.input['Embedment n'] = Variable('Embedment n',
                                             'float',
                                             0.0, [],
                                             limits=[0.0, 1.0],
                                             precision=4,
                                             unit_type=['coefficient'],
                                             native_unit='coefficient',
                                             us_units=[['coefficient']],
                                             si_units=[['coefficient']])

        self.input['Gradation method'] = Variable('Gradation method', "list",
                                                  0, gradation_equations_list)

        self.input['Material'] = Variable('Select material', "list", 0,
                                          list(self.material_dict.keys()))

        self.input['Material category'] = Variable('Material category', "list",
                                                   0, self.material_category)
        self.input['Material name'] = Variable('Material', "list", 0,
                                               self.material_name)
        self.input['Material corrugation'] = Variable(
            'Material corrugation', "list", 0, self.material_corrugation)
        self.input['Material condition'] = Variable(
            'Material condition', "list", 0,
            list(self.material_condition.keys()))

        self.input['Invalid n value'] = Variable(
            'Invalid n value', 'float', 0.04, [], limits=[0.0, 1.0], precision=4, complexity=1,
            unit_type=['coefficient'], native_unit='coefficient', us_units=[['coefficient']],
            si_units=[['coefficient']], note='This value will be used when the method determines an invalid n value.')

        # Now build and update lists for our options
        self.update_lists()

        # self.input['n per geometry segment'] = Variable(
        #     'n per geometry segment', 'float_list', 0, [0.0, 0.0, 0.0], precision=4, unit_type=['coefficient'],
        #     native_unit='coefficient', us_units=[['coefficient']], si_units=[['coefficient']])
        self.input["Manning's n by station"] = None

        # Intermediate
        self.compute_prep_functions.extend([])
        self.intermediate_to_copy.extend(
            ['horton_manning_value', 'pavlovskii_manning_value', 'lotter_manning_value',
             'final_horton_manning_value', 'final_pavlovskii_manning_value', 'final_lotter_manning_value',
             'd50', 'd84', 'hyd_radius', 'average_flow_depth', 'friction_slope',
             'limerinos_n_applicable', 'limerinos_n', 'blodgett_n_applicable', 'blodgett_n', 'jarrett_n_applicable',
             'jarrett_n', 'mussetter_n_applicable', 'mussetter_n', 'gradation_calculation', 'n_applicable',
             'seed_value', 'unknown',
             'shape', 'open_shape', 'embedment_present', 'gradation_present', 'total_embedment', 'stations',
             'elevations', 'no_station_overlap', 'n_entry_list', 'n_channel_entry_list', 'n_embedment_entry_list',
             'composite_n_method', 'material_dict', 'material_category', 'material_name', 'material_corrugation',
             'material_condition', 'material_condition_list', 'mannings_n_long'])

        self.unknown = None
        self.horton_manning_value = 0.0
        self.pavlovskii_manning_value = 0.0
        self.lotter_manning_value = 0.0
        self.final_horton_manning_value = self.horton_manning_value
        self.final_pavlovskii_manning_value = self.pavlovskii_manning_value
        self.final_lotter_manning_value = self.lotter_manning_value

        # Intermediate Gradation input
        self.d50 = 0.0
        self.d84 = 0.0
        self.hyd_radius = 0.0
        self.average_flow_depth = 0.0
        self.friction_slope = 0.0
        self.limerinos_n_applicable = False
        self.limerinos_n = 0.0
        self.blodgett_n_applicable = False
        self.blodgett_n = 0.0
        self.jarrett_n_applicable = False
        self.jarrett_n = 0.0
        self.mussetter_n_applicable = False
        self.mussetter_n = 0.0
        self.gradation_calculation = False
        self.n_applicable = False

        self.seed_value = 0.03

        self.mannings_n_long = []

        self.warnings = []

        self.results = {}

        self.results['Composite n computation method'] = Variable(
            'Composite n computation method', 'string_list', '')

        self.results['Composite n value'] = Variable('Composite n value', 'float_list', 0.0, [], precision=4,
                                                     unit_type='coefficient', native_unit='coefficient',
                                                     us_units='coefficient', si_units='coefficient')

    def build_composite_n_list(self):
        """Build composite n list."""
        self.calculator.shape = self.shape
        self.calculator.open_shape = self.open_shape

        self.calculator.composite_n_method = self.composite_n_method
        self.calculator.build_composite_n_list()
        self.composite_n_method = self.calculator.composite_n_method

    def build_n_entry_list(self):
        """Build n entry list."""
        # build list for the ways to specify manning's n value
        self.calculator.no_station_overlap = self.no_station_overlap
        self.calculator.embedment_present = self.embedment_present
        self.calculator.gradation_present = self.gradation_present
        results, complex_var, _ = self.get_setting_var('Complexity', 0)
        if results:
            if 'calc_data' not in self.calculator.input_dict:
                self.calculator.input_dict['calc_data'] = {}
            self.calculator.input_dict['calc_data']['Complexity'] = complex_var.value

        self.calculator.n_entry_list = self.n_entry_list
        self.calculator.build_n_entry_list()
        self.n_entry_list = self.calculator.n_entry_list

    def build_material_dict(self):
        """Build material dict."""
        self.calculator.shape = self.shape

        self.calculator.material_dict = self.material_dict
        self.calculator.build_material_dict()
        self.material_dict = self.calculator.material_dict

    def build_material_categories(self):
        """Build material categories."""
        # self.material_category = ['channel lining', 'storm drain conduits', 'street and pavement gutters']
        # - Never changes
        self.calculator.material_name = self.material_name
        self.calculator.material_corrugation = self.material_corrugation
        self.calculator.material_condition = self.material_condition
        self.calculator.build_material_categories()
        self.material_name = self.calculator.material_name
        self.material_corrugation = self.calculator.material_corrugation
        self.material_condition = self.calculator.material_condition

    def update_lists(self):
        """Update the lists and variables for current selections."""
        # Transfer necessary data to the calculator
        self.calculator.shape = self.shape
        self.calculator.open_shape = self.open_shape
        self.calculator.no_station_overlap = self.no_station_overlap
        self.calculator.embedment_present = self.embedment_present
        self.calculator.gradation_present = self.gradation_present

        self.calculator.n_entry_list = self.n_entry_list
        self.calculator.composite_n_method = self.composite_n_method
        self.calculator.material_dict = self.material_dict

        self.calculator.material_category = self.material_category
        self.calculator.material_name = self.material_name
        self.calculator.material_corrugation = self.material_corrugation
        self.calculator.material_condition = self.material_condition

        if 'calc_data' not in self.calculator.input_dict:
            self.calculator.input_dict['calc_data'] = {}
        self.calculator.input_dict['calc_data']['Material category'] = self.input['Material category'].get_val()
        self.calculator.input_dict['calc_data']['Material name'] = self.input['Material name'].get_val()
        self.calculator.input_dict['calc_data']['Material corrugation'] = self.input['Material corrugation'].get_val()
        self.calculator.input_dict['calc_data']['Material condition'] = self.input['Material condition'].get_val()

        self.calculator.update_lists()

        # Transfer necessary data from the calculator
        self.n_entry_list = self.calculator.n_entry_list
        self.composite_n_method = self.calculator.composite_n_method
        self.material_dict = self.calculator.material_dict

        self.material_category = self.calculator.material_category
        self.material_name = self.calculator.material_name
        self.material_corrugation = self.calculator.material_corrugation
        self.material_condition = self.calculator.material_condition
        self.material_condition_list = list(self.material_condition.keys())

        # Update the selection lists
        self.input['n entry'].value_options = self.n_entry_list
        self.input['n method'].value_options = self.composite_n_method
        self.input['Material name'].value_options = self.material_name
        self.input['Material corrugation'].value_options = self.material_corrugation
        self.input['Material condition'].value_options = self.material_condition_list

        # Verify that the selections still work
        size = len(self.material_name) - 1
        if self.input['Material name'].value > size >= 0:
            self.input['Material name'].set_val(self.material_name[0])
        elif size < 0 < self.input['Material name'].value:
            self.input['Material name'].value = 0
        size = len(self.material_corrugation) - 1
        if self.input['Material corrugation'].value > size >= 0:
            self.input['Material corrugation'].set_val(
                self.material_corrugation[0])
        elif size < 0 < self.input['Material corrugation'].value:
            self.input['Material corrugation'].value = 0
        size = len(self.material_condition_list) - 1
        if self.input['Material condition'].value > size >= 0:
            self.input['Material condition'].set_val(
                self.material_condition_list[0])
        elif size < 0 < self.input['Material condition'].value:
            self.input['Material condition'].value = 0

    # def _get_can_compute(self):
    #     """Determine whether we have enough data to compute.

    #     Returns:
    #         True, if we can compute; otherwise, False
    #         message (list of str): warning message if we fail
    #     """
    #     self.update_lists()

    #     self.seed_value = self.input['Invalid n value'].get_val()

    #     if self.input['n entry'].get_val() == 'specify composite n value':
    #         if self.input['Composite n'].get_val() > 0.0:
    #             return True
    #         else:
    #             self.warnings.append("Please enter a composite Manning's n value.")
    #             return False

    #     if self.input['n entry'].get_val() in ['specify n by geometry segment', 'specify n by station']:
    #         # TODO: We need a define button that may load 'read-only' station and elevation with a editable n value
    #         #  column or a station and n value columns.
    #         self.warnings.append('Feature not yet implemented.')
    #         return False

    #     n_entry = ''
    #     if self.input['n entry'].get_val() in ['specify channel and embedment n values']:
    #         n_entry = self.input['Channel n entry'].get_val()
    #         if n_entry == 'specify n value':
    #             if self.input['Channel n'].get_val() <= 0.0:
    #                 all_messages = ["Please enter a manning's n value for the channel.",]
    #                 _, messages = self.check_embedment_input()
    #                 all_messages.extend(messages)
    #                 self.warnings.extend(all_messages)
    #                 return False
    #             return self.check_embedment_input()

    #     if self.input['n entry'].get_val() in ['specify material'] or n_entry in ['specify material']:
    #         return self.check_embedment_input()

    #     if self.input['n entry'].get_val() in ['specify material and condition by category'] or \
    #             n_entry in ['specify material and condition by category']:
    #         return self.check_embedment_input()

    #     return True

    def check_embedment_input(self):
        """Check the embedment input for errors.

        Returns:
            bool: True if we have embedment data; otherwise, False
        """
        _, zero_tol = self.get_setting('Zero tolerance')
        if self.embedment_present:
            check_embedment_n = True
            if self.gradation_present:
                if self.input['Embedment n entry'].get_val() == 'specify gradation equation':
                    check_embedment_n = False
                    if self.input['Gradation method'].get_val() == 'Limerinos':
                        if self.d84 <= zero_tol:
                            return False, ['Please complete the gradation data.']
                    if self.input['Gradation method'].get_val() == 'Blodgett':
                        if self.d50 <= zero_tol:
                            return False, ['Please complete the gradation data.']
                    if self.input['Gradation method'].get_val() == 'Jarrett':
                        if self.d84 <= zero_tol:
                            return False, ['Please complete the gradation data.']
                    if self.input['Gradation method'].get_val() == 'Mussetter':
                        if self.d84 <= zero_tol or self.d50 <= zero_tol:
                            return False, ['Please complete the gradation data.']
            if check_embedment_n:
                if self.input['Embedment n'].get_val() <= 0.0:
                    return False, ["Please enter a Manning's n value for the embedment."]
        return True, ['']

    def get_val(self):
        """Computes and returns the results.

        Returns:
            self.results['result'] (variable): result of the computations.
        """
        self.compute_data()
        return self.results['result']

    def get_input_group(self, unknown=None):
        """Get the input group (for user-input).

        Args:
            unknown (str): unknown variable that we want to compute

        Returns:
            dict: dictionary of input variables
        """
        input_vars = {}

        input_vars['n entry'] = self.input['n entry']

        # Gather the input options
        if self.input['n entry'].get_val() == 'specify composite n value':
            input_vars['Composite n'] = self.input['Composite n']
        elif self.input['n entry'].get_val() in ['specify material', 'specify material and condition by category'] and \
                self.embedment_present:
            input_vars['n method'] = self.input['n method']
        elif self.input['n entry'].get_val() not in [
                'specify composite n value', 'specify material',
                'specify material and condition by category',
                'compute n values for gradation size']:
            input_vars['n method'] = self.input['n method']

        if self.input['n entry'].get_val() in ['specify n by geometry segment']:
            pass
        elif self.input['n entry'].get_val() in ['specify n by station']:
            input_vars["Manning's n by station"] = self.input["Manning's n by station"]

        elif self.input['n entry'].get_val() in ['specify channel and embedment n values']:
            input_vars['Channel n entry'] = self.input['Channel n entry']
            if self.input['Channel n entry'].get_val() == 'specify n value':
                input_vars['Channel n'] = self.input['Channel n']
            elif self.input['Channel n entry'].get_val() == 'specify material':
                input_vars['Material'] = self.input['Material']
            elif self.input['Channel n entry'].get_val() == 'specify material and condition by category':
                input_vars['Material category'] = self.input['Material category']
                input_vars['Material name'] = self.input['Material name']
                if len(self.material_corrugation):
                    input_vars['Material corrugation'] = self.input['Material corrugation']
                input_vars['Material condition'] = self.input['Material condition']
            self.add_embedment_input(input_vars)

        elif self.input['n entry'].get_val() in ['specify material']:
            input_vars['Material'] = self.input['Material']
            self.add_embedment_input(input_vars)

        elif self.input['n entry'].get_val() in ['specify material and condition by category']:
            input_vars['Material category'] = self.input['Material category']
            input_vars['Material name'] = self.input['Material name']
            if len(self.material_corrugation):
                input_vars['Material corrugation'] = self.input['Material corrugation']
            input_vars['Material condition'] = self.input['Material condition']
            self.add_embedment_input(input_vars)

        elif self.input['n entry'].get_val() in ['compute n values for gradation size']:
            input_vars['Gradation method'] = self.input['Gradation method']
            input_vars['Invalid n value'] = self.input['Invalid n value']

        return input_vars

    def add_embedment_input(self, input_vars):
        """Add the embedment input to the input_vars dictionary.

        Args:
            input_vars (dict): dictionary of input variables
        """
        if self.embedment_present:
            add_embedment_n = True
            if self.gradation_present:
                input_vars['Embedment n entry'] = self.input['Embedment n entry']
                if self.input['Embedment n entry'].get_val() == 'specify gradation equation':
                    input_vars['Gradation method'] = self.input['Gradation method']
                    add_embedment_n = False
            if add_embedment_n:
                input_vars['Embedment n'] = self.input['Embedment n']

    def clear_results(self):
        """Clears the results and those of subclasses to prepare for computation.

        """
        self.results['Composite n computation method'].set_val([])
        self.results['Composite n value'].set_val([])

    def _compute_data(self):
        """Computes the data possible; stores results in self.

        Returns:
            bool: True if successful
        """
        pass

    def compute_gradation_n(self):
        """Computes the Manning's n value using the gradation method."""
        if self.input['Gradation method'].get_val() == 'Limerinos':
            return self.compute_limerinos_n()
        elif self.input['Gradation method'].get_val() == 'Blodgett':
            return self.compute_blodgett_n()
        elif self.input['Gradation method'].get_val() == 'Jarrett':
            return self.compute_jarrett_n()
        elif self.input['Gradation method'].get_val() == 'Mussetter':
            return self.compute_mussetter_n()

    def compute_limerinos_n(self):
        """Computes the Manning's n value using the Limerinos equation."""
        # HEC 26
        # Page C-1
        _, null_data = self.get_setting('Null data')
        self.limerinos_n_applicable = False
        self.n_applicable = False
        self.limerinos_n = null_data
        if self.d84 <= 0.0:
            self.warnings.append(
                'Please enter the D84 gradation to complete the Limerinos equation.'
            )
            return self.seed_value, False
        if self.hyd_radius <= 0.0:
            return self.seed_value, False
        rh_over_d84 = self.hyd_radius / self.d84
        alpha = 0.0926
        one_sixth = 1.0 / 6.0

        if 0.9 <= rh_over_d84 <= 68.5:
            self.limerinos_n_applicable = True
            self.n_applicable = True
        else:
            self.warnings.append('This channel does match the applicable range for the Limerinos equation.')
            self.warnings.append(f'The hydraulic radius over D84 is {rh_over_d84} and needs to be between 0.9 and 68.5')

        self.limerinos_n = (alpha * self.hyd_radius ** one_sixth) / 1.16 + 2.0 * math.log10(rh_over_d84)

        return self.limerinos_n, True

    def compute_blodgett_n(self):
        """Computes the Manning's n value using the Blodgett equation."""
        # HEC 26
        # Page C-2
        _, null_data = self.get_setting('Null data')
        self.blodgett_n_applicable = False
        self.n_applicable = False
        self.blodgett_n = null_data
        if self.d50 <= 0.0:
            self.warnings.append('Please enter the D50 gradation to complete the Blodgett equation.')
            return self.seed_value, False
        if self.average_flow_depth <= 0.0:
            return self.seed_value, False
        ya_over_d50 = self.average_flow_depth / self.d50
        alpha = 0.262
        one_sixth = 1.0 / 6.0

        if 1.5 <= ya_over_d50 <= 185:
            self.blodgett_n_applicable = True
            self.n_applicable = True
        else:
            self.warnings.append('This channel does match the applicable range for the Blodgett equation.')
            self.warnings.append(f'The depth over Dd50 is {ya_over_d50} and needs to be between 1.5 and 185')

        self.blodgett_n = (alpha * self.average_flow_depth ** one_sixth) / 2.25 + 5.23 * math.log10(ya_over_d50)

        return self.blodgett_n, True

    def compute_jarrett_n(self):
        """Computes the Manning's n value using the Jarrett equation."""
        # HEC 26
        # Page C-1
        _, null_data = self.get_setting('Null data')
        self.jarrett_n_applicable = False
        self.n_applicable = False
        self.jarrett_n = null_data
        if self.hyd_radius <= 0.0:
            return self.seed_value, False
        alpha = 0.39
        if self.d84 > 0.0:
            rh_over_d84 = self.hyd_radius / self.d84
        else:
            self.warnings.append('Please enter the D84 gradation to complete the Jarrett equation.')
            return self.seed_value, False

        first_test = False
        second_test = False
        third_test = False
        if 0.002 < self.friction_slope < 0.04:
            first_test = True
        else:
            self.warnings.append('This channel does match the applicable range for the Jarrett equation.')
            self.warnings.append(f'The friction slope is {self.friction_slope} and needs to be between 0.002 and 0.04')

        if 0.5 < self.hyd_radius < 7.0:
            second_test = True
        else:
            self.warnings.append('This channel does match the applicable range for the Jarrett equation.')
            self.warnings.append(f'The hydraulic radius is {self.hyd_radius} and needs to be between 0.5 and 7.0')

        if 0.4 < rh_over_d84 < 11:
            third_test = True
        else:
            self.warnings.append('This channel does match the applicable range for the Jarrett equation.')
            self.warnings.append(
                f'The hydraulic radius over D84 is {rh_over_d84} and needs to be between 0.4 and 11.0'
            )

        if first_test and second_test and third_test:
            self.jarrett_n_applicable = True
            self.n_applicable = True

        self.jarrett_n = alpha * self.friction_slope**0.38 * self.hyd_radius**-0.16

        return self.jarrett_n, True

    def compute_mussetter_n(self):
        """Computes the Manning's n value using the Mussetter equation."""
        # HEC 26
        # Page C-2
        _, null_data = self.get_setting('Null data')
        self.mussetter_n_applicable = False
        self.n_applicable = False
        self.mussetter_n = null_data
        if self.d50 <= 0.0:
            self.warnings.append('Please enter the D50 gradation to complete the Mussetter equation.')
            return self.seed_value, False
        if self.d84 <= 0.0:
            self.warnings.append('Please enter the D84 gradation to complete the Mussetter equation.')
            return self.seed_value, False
        if self.average_flow_depth <= 0.0:
            return self.seed_value, False
        rh_over_d84 = self.hyd_radius / self.d84
        ya_over_d84 = self.average_flow_depth / self.d84
        d84_over_d50 = self.d84 / self.d50
        alpha = 0.24

        if 0.25 < rh_over_d84 < 3.72:
            self.mussetter_n_applicable = True
            self.n_applicable = True
        else:
            self.warnings.append('This channel does match the applicable range for the Mussetter equation.')
            self.warnings.append(f'The hydraulic radius over D84 is {rh_over_d84} '
                                 f'and needs to be between 0.25 and 3.72')

        self.mussetter_n = alpha * ya_over_d84**-0.46 * d84_over_d50**0.85 * self.friction_slope**0.39

        return self.mussetter_n, True

    def get_channel_n(self):
        """Returns the channel n value based on the selections."""
        specify_n = False
        specify_material = False
        specify_mat_and_category = False
        if self.input['n entry'].get_val() in ['specify channel and embedment n values']:
            if self.input['Channel n entry'].get_val() == 'specify n value':
                specify_n = True
            elif self.input['Channel n entry'].get_val() == 'specify material':
                specify_material = True
            elif self.input['Channel n entry'].get_val() == 'specify material and condition by category':
                specify_mat_and_category = True
        if self.input['n entry'].get_val() == 'specify n value' or specify_n:
            return self.input['Channel n'].get_val()
        elif self.input['n entry'].get_val() == 'specify material' or specify_material:
            return self.material_dict[self.input['Material'].get_val()]
        elif self.input['n entry'].get_val() == 'specify material and condition by category' or \
                specify_mat_and_category:
            return self.material_condition[self.input['Material condition'].get_val()]
        elif self.input['n entry'].get_val() == 'compute n values for gradation size':
            return self.compute_gradation_n()

    def get_embedment_n(self):
        """Returns the embedment n value based on the selections."""
        if self.input['Embedment n entry'].get_val() == 'specify n value':
            return self.input['Embedment n'].get_val()
        else:  # 'specify gradation equation'
            # When result == False, the n is a 0.03 seed value that we use to start the calculation with
            n, result = self.compute_gradation_n()
            # if not self.n_applicable and n < 0.0:
            if n < 0.0:
                n = abs(n)  # Try it as a new seed
            return n

    def get_intermediate_composite_n(self):
        """Returns the composite n value based on the selections.

        Returns:
            n (float): Manning's n value
        """
        if self.input['n entry'].get_val() == 'specify composite n value':
            # TODO: Need to say can't compute if specified n = 0.0; perhaps log an error here
            return self.input['Composite n'].get_val()
        elif self.input['n entry'].get_val() in ['specify n by geometry segment', 'specify n by station',
                                                 'specify channel and embedment n values']:
            if self.input['n method'].get_val() == "Lotter Method":
                return self.lotter_manning_value
            if self.input['n method'].get_val() == "Horton and Einstein Method":
                return self.horton_manning_value
            if self.input['n method'].get_val() == "Pavlovskii, Mühlhofer, Einstein and Banks Method":
                return self.pavlovskii_manning_value
        elif self.embedment_present and self.input['n entry'].get_val() == 'compute n values for gradation size':
            return self.compute_gradation_n()
        # HY-8 style, HY12 style
        elif self.input['n entry'].get_val() in ['specify material',
                                                 'specify material and condition by category']:
            return self.get_channel_n()

        _, null_data = self.get_setting('Null data')
        return null_data

    def get_intermediate_n_and_applicable(self):
        """Returns the composite n value based on the selections.

        Returns:
            n (float): Manning's n value
        """
        self.gradation_calculation = False
        self.warnings = []
        n = self.get_intermediate_composite_n()
        if self.embedment_present and self.input['Embedment n entry'].get_val() == 'specify gradation equation':
            self.gradation_calculation = True
        return n, True

    def get_composite_n(self):
        """Returns the composite n value based on the selections.

        Returns:
            n (float): Manning's n value
        """
        return self.results['Composite n value']

    def finalize_last_result(self):
        """Adds the last computed values to the results."""
        self.final_horton_manning_value = self.horton_manning_value
        self.final_pavlovskii_manning_value = self.pavlovskii_manning_value
        self.final_lotter_manning_value = self.lotter_manning_value
        if self.input['n entry'].get_val() == 'specify composite n value':
            self.results['Composite n computation method'].add_result('specified composite n value')
            self.results['Composite n value'].add_result(self.input['Composite n'].get_val())
        elif self.input['n entry'].get_val() in ['specify n by geometry segment', 'specify n by station',
                                                 'specify channel and embedment n values']:
            if self.input['n method'].get_val() == "Horton and Einstein Method":
                self.results['Composite n computation method'].add_result('Horton and Einstein Method')
                self.results['Composite n value'].add_result(self.horton_manning_value)
            elif self.input['n method'].get_val() == "Pavlovskii, Mühlhofer, Einstein and Banks Method":
                self.results['Composite n computation method'].add_result(
                    'Pavlovskii, Mühlhofer, Einstein and Banks Method')
                self.results['Composite n value'].add_result(self.pavlovskii_manning_value)
            elif self.input['n method'].get_val() == "Lotter Method":
                self.results['Composite n computation method'].add_result('Lotter Method')
                self.results['Composite n value'].add_result(self.lotter_manning_value)
        elif self.input['n entry'].get_val() == 'compute n values for gradation size':
            self.results['Composite n computation method'].add_result(
                f'{self.input["gradation method"].get_val()} for gradation size')
            self.results['Composite n value'].add_result(self.compute_gradation_n())

        # HY-8 style, HY12 style
        elif self.input['n entry'].get_val() in ['specify material',
                                                 'specify material and condition by category']:
            self.results['Composite n computation method'].add_result('specified by material')
            self.results['Composite n value'].add_result(self.get_channel_n())
