"""Worker thread for the Boundary Conditions coverage mapping operation."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackThread

# 4. Local modules
from xms.adcirc.data.sim_data import SimData
from xms.adcirc.dmi.sim_component_queries import SimComponentQueries
from xms.adcirc.feedback.xmlog import XmLog


class BcMappingWorkerThread(ProcessFeedbackThread):
    """Worker thread class to do computations in another thread."""
    def __init__(self, query, sim_comp, parent):
        """Constuct the worker.

        Args:
            query (:obj:`Query`): XMS interprocess communication object
            sim_comp (:obj:`SimComponent`): The simulation data
            parent (:obj:`QWidget`): The Qt parent
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self.query = query
        self.sim_comp = sim_comp

    def _do_work(self):
        """Worker code to execute in a separate thread."""
        XmLog().instance.info('Retrieving source Boundary Conditions coverage from SMS.')
        query_helper = SimComponentQueries(self.sim_comp, self.query)
        bc_main_file = query_helper.get_bc_main_file()
        if not bc_main_file:
            query_helper.unlink_item('TI_COVER_PTR', coverage_type='Boundary Conditions')
            return

        # Perform the mapping
        try:
            from xms.adcirc.components.bc_component import BcComponent
            bc_comp = BcComponent(bc_main_file)
            # Request and load the component coverage id map
            XmLog().instance.info('Retrieving boundary condition arc ids and assignments.')
            self.query.load_component_ids(bc_comp, points=True, arcs=True)
            bc_comp.clean_unused_comp_ids()
            data = SimData(self.sim_comp.main_file)
            bc_comp.apply_to_sim(self.query, data.timing.attrs['RUNDAY'])
            if not LogEchoQSignalStream.logged_error:  # All done, no errors
                XmLog().instance.info('Boundary Condition coverage successfully applied to ADCIRC simulation.')
        except Exception:
            query_helper.unlink_item('TI_COVER_PTR')
            XmLog().instance.exception('Unable to apply boundary condition attributes.')
