"""Module that manages logging for xmsadcirc."""

# 1. Standard Python modules
import logging
import os
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv

# 4. Local modules


def report_error(msg, sim_export=True):
    """Log an error message to the logger and the XMS stderr file.

    Used by export scripts because they do not have a feedback dialog.

    Args:
        msg (:obj:`str`): The error message to log
        sim_export (:obj:`bool`): If True, will write to the XMS stderr file. If False, only logs the error.
            Set to False when exporting from the partial export dialog.
    """
    if sim_export:
        XmEnv.report_error(msg)
    XmLog().instance.error(msg)


class XmLog:
    """Class for logging messages in XMS Python scripts."""
    top_module = 'xms.adcirc'
    echo_output = False

    class __XmLog:
        MAX_LOG_SIZE = 1e+6  # In bytes (1 MB), log files larger than this will be wiped.

        def __init__(self):
            """Construct the singleton."""
            self._logger = logging.getLogger(XmLog.top_module)
            self._logger.setLevel(logging.DEBUG)
            log_file = self._setup_log_file()
            if log_file:
                fh = logging.FileHandler(log_file)
                fh.setLevel(logging.DEBUG)
                formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
                fh.setFormatter(formatter)
                self._logger.addHandler(fh)

        def _setup_log_file(self):
            """Make sure the log file directory exists and any existing log file is reasonable in size."""
            try:
                log_file = self._get_log_filename()
                if os.path.isfile(log_file) and os.path.getsize(log_file) > self.MAX_LOG_SIZE:
                    os.remove(log_file)
            except Exception:
                return None
            return log_file

        def _get_log_filename(self):
            """Get the log filename."""
            appdata = os.path.join(os.environ.get('APPDATA', ''), 'xms_python_logs')
            os.makedirs(appdata, exist_ok=True)
            return os.path.join(appdata, f'{XmLog.top_module}.log')

        def debug(self, msg, logger_name=None):
            """Log a debug level message."""
            logger = self._get_logger(logger_name)
            logger.debug(msg)
            if XmLog.echo_output:
                sys.stdout.write(f'{msg}\n')

        def info(self, msg, logger_name=None):
            """Log an info level message."""
            logger = self._get_logger(logger_name)
            logger.info(msg)
            if XmLog.echo_output:
                sys.stdout.write(f'{msg}\n')

        def warning(self, msg, logger_name=None):
            """Log a warning level message."""
            logger = self._get_logger(logger_name)
            logger.warning(msg)
            if XmLog.echo_output:
                sys.stdout.write(f'{msg}\n')

        def error(self, msg, logger_name=None):
            """Log an error level message."""
            logger = self._get_logger(logger_name)
            logger.error(msg)
            if XmLog.echo_output:
                sys.stderr.write(f'{msg}\n')

        def critical(self, msg, logger_name=None):
            """Log a critical level message."""
            logger = self._get_logger(logger_name)
            logger.error(msg)
            if XmLog.echo_output:
                sys.stderr.write(f'{msg}\n')

        def exception(self, msg, logger_name=None):
            """Log a critical level message."""
            logger = self._get_logger(logger_name)
            logger.exception(msg)
            if XmLog.echo_output:
                sys.stderr.write(f'{msg}\n')

        def _get_logger(self, logger_name):
            """Get the appropriate logger to use with a log message.

            Args:
                logger_name (:obj:`str`): Name of the logger to use (using logging standard library conventions).
                    If None, returns the top-level module logger.

            Returns:
                See description
            """
            if logger_name is not None:
                return logging.getLogger(logger_name)
            else:
                return self._logger

        def set_logger(self, logger):
            """Sets the logger.

            Args:
                logger (:obj:`logging.Logger`): the logger
            """
            self._logger = logger

    instance = None

    def __init__(self):
        """Construct the logger."""
        if not XmLog.instance:
            XmLog.instance = XmLog.__XmLog()

    def __getattr__(self, name):
        """Call through to singleton instance for attributes."""
        return getattr(self.instance, name)
